<?php
/**
 * Handles the Institute Settings administrative page and related logic.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Tohidul_Certificate_Verification_System_Settings {
    
    private $database;
    
    public function __construct( Tohidul_Certificate_Verification_System_Database $database ) {
        $this->database = $database;
        add_action( 'admin_post_tcvs_save_institute_settings', array( $this, 'save_settings' ) );
    }
    
    public function settings_page(): void {
        $settings = $this->database->get_institute_settings();
        
        if ( ! $settings ) {
            $settings = (object) array(
                'institute_name'    => '',
                'institute_logo'    => '',
                'institute_address' => '',
                'institute_phone'   => '',
                'institute_email'   => '',
                'signature_image'   => '',
                'verified_by_image' => '',
            );
        }
        
        // Pass settings with the correct prefix
        $tcvs_settings = $settings;
        
        include TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_PATH . 'templates/settings-page.php';
    }
    
    public function save_settings(): void {
        if ( ! isset( $_SERVER['REQUEST_METHOD'] ) || 'POST' !== $_SERVER['REQUEST_METHOD'] ) {
            wp_die( esc_html__( 'Invalid request method.', 'tohidul-certificate-verification-system' ) );
        }

        if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'tcvs_save_institute_settings' ) ) {
            wp_die( esc_html__( 'Security check failed. Please try again.', 'tohidul-certificate-verification-system' ) );
        }
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'Unauthorized access.', 'tohidul-certificate-verification-system' ) );
        }
        
        $data = array(
            'institute_name'    => sanitize_text_field( wp_unslash( $_POST['institute_name'] ?? '' ) ),
            'institute_logo'    => esc_url_raw( wp_unslash( $_POST['institute_logo'] ?? '' ) ),
            'institute_address' => sanitize_textarea_field( wp_unslash( $_POST['institute_address'] ?? '' ) ),
            'institute_phone'   => sanitize_text_field( wp_unslash( $_POST['institute_phone'] ?? '' ) ),
            'institute_email'   => sanitize_email( wp_unslash( $_POST['institute_email'] ?? '' ) ),
            'signature_image'   => esc_url_raw( wp_unslash( $_POST['signature_image'] ?? '' ) ),
            'verified_by_image' => esc_url_raw( wp_unslash( $_POST['verified_by_image'] ?? '' ) ), 
        );
        
        if ( empty( $data['institute_name'] ) ) {
            wp_safe_redirect( admin_url( 'admin.php?page=certificate-settings&message=error' ) );
            exit;
        }
        
        $this->database->update_institute_settings( $data );
        wp_safe_redirect( admin_url( 'admin.php?page=certificate-settings&message=saved' ) );
        exit;
    }
}