<?php
/**
 * Handles various security measures for the Certificate Verification System.
 * 
 * @package Tohidul_Certificate_Verification_System
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Tohidul_Certificate_Verification_System_Security {
    
    public function __construct() {
        add_filter( 'wp_kses_allowed_html', array( $this, 'custom_kses_allowed_html' ), 10, 2 );
        add_action( 'init', array( $this, 'security_headers' ) );
        add_filter( 'upload_mimes', array( $this, 'restrict_upload_mimes' ) );
    }
    
    /**
     * Extends allowed HTML tags for a custom context.
     */
    public function custom_kses_allowed_html( array $allowed_tags, string $context ): array {
        if ( 'tohidul_certificate_verification_system' === $context ) {
            $allowed_tags = array(
                'input'    => array( 
                    'type' => true, 'name' => true, 'value' => true, 'class' => true, 
                    'id' => true, 'placeholder' => true, 'required' => true, 'checked' => true,
                    'maxlength' => true, 'size' => true
                ),
                'select'   => array( 'name' => true, 'class' => true, 'id' => true ),
                'option'   => array( 'value' => true, 'selected' => true ),
                'textarea' => array( 'name' => true, 'class' => true, 'id' => true, 'placeholder' => true, 'rows' => true, 'cols' => true ),
                'form'     => array( 'method' => true, 'action' => true, 'class' => true, 'id' => true, 'enctype' => true ),
                'img'      => array( 'src' => true, 'alt' => true, 'class' => true, 'width' => true, 'height' => true, 'style' => true ),
                'table'    => array( 'class' => true, 'id' => true ),
                'thead'    => array( 'class' => true ),
                'tbody'    => array( 'class' => true ),
                'tr'       => array( 'class' => true ),
                'th'       => array( 'scope' => true, 'class' => true ),
                'td'       => array( 'class' => true, 'colspan' => true, 'rowspan' => true ),
                'a'        => array( 'href' => true, 'class' => true, 'target' => true, 'rel' => true ),
                'div'      => array( 'class' => true, 'id' => true, 'style' => true ),
                'span'     => array( 'class' => true, 'style' => true ),
                'p'        => array( 'class' => true, 'style' => true ),
                'br'       => array(),
                'strong'   => array(),
                'em'       => array(),
                'ul'       => array( 'class' => true ),
                'ol'       => array( 'class' => true ),
                'li'       => array( 'class' => true ),
                'h1'       => array( 'class' => true ),
                'h2'       => array( 'class' => true ),
                'h3'       => array( 'class' => true ),
                'h4'       => array( 'class' => true ),
                'h5'       => array( 'class' => true ),
                'h6'       => array( 'class' => true ),
                'button'   => array( 'type' => true, 'class' => true, 'id' => true ),
            );
            return $allowed_tags;
        }
        
        return $allowed_tags;
    }
    
    /**
     * Adds recommended security headers.
     */
    public function security_headers(): void {
        if ( ! headers_sent() ) {
            header( 'X-Content-Type-Options: nosniff' );
            header( 'X-Frame-Options: SAMEORIGIN' );
            header( 'X-XSS-Protection: 1; mode=block' );
        }
    }
    
    /**
     * Restricts file types that can be uploaded.
     */
    public function restrict_upload_mimes( array $mimes ): array {
        // Get the page parameter safely
        $page = $this->get_safe_page_parameter();
        
        // Only modify mimes for our plugin's uploads
        if ( $page && strpos( $page, 'certificate-verification' ) === 0 ) {
            return array(
                'csv'        => 'text/csv',
                'json'       => 'application/json',
                'jpg|jpeg|jpe' => 'image/jpeg',
                'png'        => 'image/png',
                'gif'        => 'image/gif',
                'pdf'        => 'application/pdf',
            );
        }
        
        return $mimes;
    }
    
    /**
     * Safely gets and validates the 'page' parameter from $_GET
     */
    private function get_safe_page_parameter(): string {
        // Check if page parameter exists
        if ( ! isset( $_GET['page'] ) ) {
            return '';
        }
        
        // Verify nonce if this is a form submission
        // Note: For GET requests that don't modify data, nonce verification might not be strictly necessary
        // but we'll include it if a nonce was provided
        if ( isset( $_GET['_wpnonce'] ) ) {
            $nonce = sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) );
            if ( ! wp_verify_nonce( $nonce, 'certificate_verification_action' ) ) {
                return '';
            }
        }
        
        // Unslash and sanitize the page parameter
        $page = sanitize_text_field( wp_unslash( $_GET['page'] ) );
        
        return $page;
    }
    
    /**
     * Performs basic validation on certificate data.
     */
    public function validate_certificate_data( array $data ): array {
        $errors = array();
        
        // Sanitize input data first
        $data = $this->sanitize_array( $data );
        
        if ( empty( $data['student_name'] ) || strlen( $data['student_name'] ) > 255 ) {
            $errors[] = esc_html__( 'Student name is required and must be less than 255 characters.', 'tohidul-certificate-verification-system' );
        }
        
        if ( empty( $data['certificate_number'] ) || strlen( $data['certificate_number'] ) > 100 ) {
            $errors[] = esc_html__( 'Certificate number is required and must be less than 100 characters.', 'tohidul-certificate-verification-system' );
        }
        
        if ( ! empty( $data['student_image'] ) && ! filter_var( $data['student_image'], FILTER_VALIDATE_URL ) ) {
            $errors[] = esc_html__( 'Student image must be a valid URL.', 'tohidul-certificate-verification-system' );
        }
        
        if ( empty( $data['course_name'] ) || strlen( $data['course_name'] ) > 255 ) {
            $errors[] = esc_html__( 'Course name is required and must be less than 255 characters.', 'tohidul-certificate-verification-system' );
        }
        
        return $errors;
    }
    
    /**
     * Generates a secure nonce.
     */
    public function generate_secure_nonce( string $action ): string {
        return wp_create_nonce( $action );
    }
    
    /**
     * Verifies a nonce.
     */
    public function verify_nonce( string $nonce, string $action ): bool|int {
        return wp_verify_nonce( $nonce, $action );
    }
    
    /**
     * Recursively sanitizes an array.
     */
    public function sanitize_array( $data ) {
        if ( is_array( $data ) ) {
            foreach ( $data as $key => $value ) {
                $data[ $key ] = $this->sanitize_array( $value );
            }
            return $data;
        } else {
            return sanitize_text_field( wp_unslash( $data ) );
        }
    }
    
    /**
     * Sanitizes and validates file uploads
     */
    public function sanitize_file_upload( array $file ): array {
        $sanitized_file = array();
        
        foreach ( $file as $key => $value ) {
            if ( is_array( $value ) ) {
                $sanitized_file[ $key ] = $this->sanitize_array( $value );
            } else {
                $sanitized_file[ $key ] = sanitize_text_field( wp_unslash( $value ) );
            }
        }
        
        return $sanitized_file;
    }
    
    /**
     * Validates user capabilities for specific actions
     */
    public function check_user_capability( string $capability = 'manage_options' ): bool {
        return current_user_can( $capability );
    }
    
    /**
     * Escapes output for different contexts
     */
    public function escape_output( string $output, string $context = 'html' ): string {
        switch ( $context ) {
            case 'html':
                return esc_html( $output );
            case 'attr':
                return esc_attr( $output );
            case 'url':
                return esc_url( $output );
            case 'js':
                return esc_js( $output );
            case 'textarea':
                return esc_textarea( $output );
            default:
                return esc_html( $output );
        }
    }
}