<?php
/**
 * Handles export/import logic.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Tohidul_Certificate_Verification_System_Export_Import {

    private $database;

    public function __construct( Tohidul_Certificate_Verification_System_Database $database ) {
        $this->database = $database;
        add_action( 'admin_post_tcvs_export_certificates', array( $this, 'export_certificates' ) );
        add_action( 'admin_post_tcvs_import_certificates', array( $this, 'import_certificates' ) );
    }

    public function render_page(): void {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Import/Export Certificates', 'tohidul-certificate-verification-system' ); ?></h1>
            <div class="tcvs-form-grid" style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-top: 20px;">
                <div class="tcvs-content-card" style="background:#fff; padding:20px; border:1px solid #ccc;">
                    <h2>Export Data</h2>
                    <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                        <input type="hidden" name="action" value="tcvs_export_certificates">
                        <?php wp_nonce_field( 'tcvs_export_nonce', 'export_nonce' ); ?>
                        <button type="submit" class="button button-primary">Export to CSV</button>
                    </form>
                </div>

                <div class="tcvs-content-card" style="background:#fff; padding:20px; border:1px solid #ccc;">
                    <h2>Import Data</h2>
                    <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" enctype="multipart/form-data">
                        <input type="hidden" name="action" value="tcvs_import_certificates">
                        <?php wp_nonce_field( 'tcvs_import_nonce', 'import_nonce' ); ?>
                        <input type="file" name="import_file" accept=".csv" required style="margin-bottom: 10px;">
                        <button type="submit" class="button button-primary">Import CSV</button>
                    </form>
                </div>
            </div>
        </div>
        <?php
    }

    public function export_certificates(): void {
        if ( ! isset( $_POST['export_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['export_nonce'] ) ), 'tcvs_export_nonce' ) ) { wp_die( 'Security check failed.' ); }
        if ( ! current_user_can( 'manage_options' ) ) { wp_die( 'Permission denied.' ); }

        $filename = 'tcvs_certificates_' . gmdate( 'Y-m-d' ) . '.csv';
        header( 'Content-Type: text/csv; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename=' . $filename );
        
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen
        $output = fopen( 'php://output', 'w' );
        
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fwrite
        fprintf( $output, chr(0xEF).chr(0xBB).chr(0xBF) );
        fputcsv( $output, array( 'ID', 'Student Name', 'Father Name', 'Course Name', 'Reg No', 'Batch No', 'Issue Date', 'Image URL' ));

        $certificates = $this->database->get_all_certificates_for_export();
        foreach ( $certificates as $row ) {
            fputcsv( $output, array($row->id, $row->student_name, $row->father_name, $row->course_name, $row->certificate_number, $row->batch_number, $row->issue_date, $row->student_image));
        }
        
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
        fclose( $output ); 
        exit;
    }

    public function import_certificates(): void {
        if ( ! isset( $_POST['import_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['import_nonce'] ) ), 'tcvs_import_nonce' ) ) { wp_die( 'Security check failed.' ); }
        
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        if ( empty( $_FILES['import_file']['tmp_name'] ) ) { wp_die( 'No file uploaded.' ); }

        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        $file = fopen( $_FILES['import_file']['tmp_name'], 'r' );
        
        fgetcsv( $file ); // Skip header
        
        while ( ( $row = fgetcsv( $file ) ) !== false ) {
            if ( count($row) < 5 ) continue;
            // Simplified mapping for brevity based on export format
            $data = array(
                'student_name' => $row[1] ?? '',
                'father_name' => $row[2] ?? '',
                'course_name' => $row[3] ?? '',
                'certificate_number' => $row[4] ?? '',
                'batch_number' => $row[5] ?? '',
                'issue_date' => $row[6] ?? '',
                'student_image' => $row[7] ?? ''
            );
            if( ! $this->database->get_certificate_by_number($data['certificate_number']) ) {
                $this->database->insert_certificate($data);
            }
        }
        
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
        fclose($file);
        
        wp_safe_redirect( admin_url( 'admin.php?page=certificate-import-export&message=import_success' ) ); exit;
    }
}