<?php
/**
 * Handles all database interactions.
 */
class Tohidul_Certificate_Verification_System_Database {
    
    private string $table_name;
    private string $settings_table;
    private string $cache_group = 'tohidul_certificate_verification_system';
    
    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_TABLE_NAME;
        $this->settings_table = $wpdb->prefix . 'tohidul_certificate_verification_system_institute_settings';
    }
    
    public function create_tables(): void {
        global $wpdb;
        require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
        $charset_collate = $wpdb->get_charset_collate();
        
        // 1. Certificates Table
        $sql = "CREATE TABLE `{$this->table_name}` (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            student_name varchar(255) NOT NULL,
            father_name varchar(255) NOT NULL,
            student_image varchar(500) NOT NULL,
            certificate_number varchar(100) NOT NULL,
            batch_number varchar(100) NOT NULL,
            course_name varchar(255) NOT NULL,
            issue_date date DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY certificate_number (certificate_number),
            KEY student_name (student_name)
        ) $charset_collate;";
        dbDelta( $sql );
        
        // 2. Settings Table
        $settings_sql = "CREATE TABLE `{$this->settings_table}` (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            institute_name varchar(255) DEFAULT '',
            institute_logo varchar(500) DEFAULT '',
            institute_address text,
            institute_phone varchar(50) DEFAULT '',
            institute_email varchar(100) DEFAULT '',
            signature_image varchar(500) DEFAULT '',
            verified_by_image varchar(500) DEFAULT '', 
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) $charset_collate;";
        dbDelta( $settings_sql );
        
        $this->insert_default_settings();
    }

    private function insert_default_settings(): void {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        $existing = $wpdb->get_var( "SELECT COUNT(*) FROM `" . esc_sql($this->settings_table) . "`" );
        if ( (int) $existing === 0 ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            $wpdb->insert( $this->settings_table, array('institute_name' => 'Your Institute Name'));
        }
    }
    
    // --- Settings Methods ---
    public function update_institute_settings( array $data ): bool {
        global $wpdb;
        $sanitized = array(
            'institute_name'    => sanitize_text_field( $data['institute_name'] ?? '' ),
            'institute_logo'    => esc_url_raw( $data['institute_logo'] ?? '' ),
            'institute_address' => sanitize_textarea_field( $data['institute_address'] ?? '' ),
            'institute_phone'   => sanitize_text_field( $data['institute_phone'] ?? '' ),
            'institute_email'   => sanitize_email( $data['institute_email'] ?? '' ),
            'signature_image'   => esc_url_raw( $data['signature_image'] ?? '' ),
            'verified_by_image' => esc_url_raw( $data['verified_by_image'] ?? '' ),
        );
        $existing = $this->get_institute_settings();
        
        $result = false;
        if ( $existing ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            $result = $wpdb->update( $this->settings_table, $sanitized, array( 'id' => $existing->id ) );
        } else {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            $result = $wpdb->insert( $this->settings_table, $sanitized );
        }
        
        if ( false !== $result ) {
            wp_cache_delete( 'institute_settings', $this->cache_group );
            return true;
        }
        return false;
    }

    public function get_institute_settings(): ?object {
        global $wpdb;
        $settings = wp_cache_get( 'institute_settings', $this->cache_group );
        
        if ( false === $settings ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            $settings = $wpdb->get_row( "SELECT * FROM `" . esc_sql($this->settings_table) . "` ORDER BY id DESC LIMIT 1" );
            wp_cache_set( 'institute_settings', $settings, $this->cache_group );
        }
        
        return $settings;
    }

    // --- Certificate Methods ---
    private function sanitize_certificate_data( array $data ): array {
        return array(
            'student_name'       => sanitize_text_field( $data['student_name'] ?? '' ),
            'father_name'        => sanitize_text_field( $data['father_name'] ?? '' ),
            'student_image'      => esc_url_raw( $data['student_image'] ?? '' ),
            'certificate_number' => sanitize_text_field( $data['certificate_number'] ?? '' ),
            'batch_number'       => sanitize_text_field( $data['batch_number'] ?? '' ),
            'course_name'        => sanitize_text_field( $data['course_name'] ?? '' ),
            'issue_date'         => sanitize_text_field( $data['issue_date'] ?? current_time('Y-m-d') ),
        );
    }
    
    public function insert_certificate( array $data ): int|false {
        global $wpdb;
        $sanitized_data = $this->sanitize_certificate_data( $data );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        $result = $wpdb->insert( $this->table_name, $sanitized_data );
        
        if ( $result ) {
            wp_cache_flush_group( $this->cache_group );
            return $wpdb->insert_id;
        }
        return false;
    }
    
    public function update_certificate( int $id, array $data ): bool {
        global $wpdb;
        $sanitized_data = $this->sanitize_certificate_data( $data );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        $result = $wpdb->update( $this->table_name, $sanitized_data, array( 'id' => $id ) );
        
        if ( false !== $result ) {
            wp_cache_flush_group( $this->cache_group );
            return true;
        }
        return false;
    }

    public function get_certificate( int $id ): ?object {
        global $wpdb;
        $cache_key = 'certificate_' . $id;
        $certificate = wp_cache_get( $cache_key, $this->cache_group );
        
        if ( false === $certificate ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            $certificate = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM `" . esc_sql($this->table_name) . "` WHERE id = %d", $id ) );
            wp_cache_set( $cache_key, $certificate, $this->cache_group );
        }
        return $certificate;
    }

    public function get_certificate_by_number( string $number ): ?object {
        global $wpdb;
        $cache_key = 'certificate_num_' . md5( $number );
        $certificate = wp_cache_get( $cache_key, $this->cache_group );
        
        if ( false === $certificate ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            $certificate = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM `" . esc_sql($this->table_name) . "` WHERE certificate_number = %s", $number ) );
            wp_cache_set( $cache_key, $certificate, $this->cache_group );
        }
        return $certificate;
    }
    
    public function get_all_certificates( int $limit = 20, int $offset = 0 ): array {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        return $wpdb->get_results( $wpdb->prepare( "SELECT * FROM `" . esc_sql($this->table_name) . "` ORDER BY id DESC LIMIT %d OFFSET %d", $limit, $offset ) );
    }
    
    public function get_total_certificates(): int {
        global $wpdb;
        $count = wp_cache_get( 'total_certificates', $this->cache_group );
        if ( false === $count ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            $count = (int) $wpdb->get_var( "SELECT COUNT(*) FROM `" . esc_sql($this->table_name) . "`" );
            wp_cache_set( 'total_certificates', $count, $this->cache_group );
        }
        return $count;
    }
    
    public function get_all_certificates_for_export(): array {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        return $wpdb->get_results( "SELECT * FROM `" . esc_sql($this->table_name) . "` ORDER BY id DESC" );
    }

    public function search_certificates( string $term, string $type = 'all' ): array {
        global $wpdb;
        $like = '%' . $wpdb->esc_like( $term ) . '%';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        return $wpdb->get_results( $wpdb->prepare( "SELECT * FROM `" . esc_sql($this->table_name) . "` WHERE certificate_number LIKE %s OR student_name LIKE %s LIMIT 50", $like, $like ) );
    }
    
    public function delete_certificate( int $id ): bool {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        $result = $wpdb->delete( $this->table_name, array( 'id' => $id ), array( '%d' ) );
        if ( $result ) {
            wp_cache_flush_group( $this->cache_group );
            return true;
        }
        return false;
    }
    
    // --- BULK DELETE METHOD ---
    public function bulk_delete_certificates( array $ids ): bool {
        global $wpdb;
        if ( empty( $ids ) ) return false;
        
        // Sanitize IDs
        $ids = array_map( 'intval', $ids );
        
        // Prepare placeholders for IN clause to avoid SQL injection warnings
        $placeholders = implode( ',', array_fill( 0, count( $ids ), '%d' ) );
        
        $sql = "DELETE FROM `{$this->table_name}` WHERE id IN ($placeholders)";
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery, WordPress.DB.PreparedSQL.NotPrepared
        $result = $wpdb->query( $wpdb->prepare( $sql, $ids ) );
        
        if ( $result ) {
            wp_cache_flush_group( $this->cache_group );
            return true;
        }
        return false;
    }
}