<?php
/**
 * Handles all backup logic.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Tohidul_Certificate_Verification_System_Backup {
    
    private $database;
    private const MAX_FILE_SIZE = 5242880;

    public function __construct( Tohidul_Certificate_Verification_System_Database $database ) {
        $this->database = $database;
        add_action( 'admin_post_tcvs_create_backup', array( $this, 'create_backup' ) );
        add_action( 'admin_post_tcvs_restore_backup', array( $this, 'restore_backup' ) );
    }

    public function render_page(): void {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Backup & Restore', 'tohidul-certificate-verification-system' ); ?></h1>
            <?php $this->display_backup_notices(); ?>

            <div class="card" style="max-width: 800px; margin-top: 20px; padding: 20px;">
                <h2><?php esc_html_e( 'Create Backup', 'tohidul-certificate-verification-system' ); ?></h2>
                <p><?php esc_html_e( 'Create a backup of all certificates in JSON format.', 'tohidul-certificate-verification-system' ); ?></p>
                <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                    <?php wp_nonce_field( 'tcvs_create_backup', '_wpnonce' ); ?>
                    <input type="hidden" name="action" value="tcvs_create_backup">
                    <button type="submit" class="button button-primary"><?php esc_html_e( 'Create Backup', 'tohidul-certificate-verification-system' ); ?></button>
                </form>
            </div>

            <div class="card" style="max-width: 800px; margin-top: 20px; padding: 20px;">
                <h2><?php esc_html_e( 'Restore Backup', 'tohidul-certificate-verification-system' ); ?></h2>
                <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" enctype="multipart/form-data">
                    <?php wp_nonce_field( 'tcvs_restore_backup', '_wpnonce' ); ?>
                    <input type="hidden" name="action" value="tcvs_restore_backup">
                    <input type="file" name="backup_file" accept=".json" required>
                    <button type="submit" class="button button-primary"><?php esc_html_e( 'Restore Backup', 'tohidul-certificate-verification-system' ); ?></button>
                </form>
            </div>
        </div>
        <?php
    }

    public function display_backup_notices(): void {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $page = isset( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : '';
        if ( 'certificate-backup' !== $page ) { return; }
        
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $message_key = isset( $_GET['message'] ) ? sanitize_text_field( wp_unslash( $_GET['message'] ) ) : '';
        if ( empty( $message_key ) ) { return; }

        $text = ''; $type = 'success';
        switch ( $message_key ) {
            case 'upload_error': $text = 'Error uploading file.'; $type = 'error'; break;
            case 'invalid_json': $text = 'Invalid JSON file.'; $type = 'error'; break;
            case 'restore_report': $text = 'Restore Complete!'; break;
        }
        if($text) printf('<div class="notice notice-%s is-dismissible"><p>%s</p></div>', esc_attr($type), esc_html($text));
    }

    public function create_backup(): void {
        if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'tcvs_create_backup' ) ) { wp_die( 'Security check failed.' ); }
        if ( ! current_user_can( 'manage_options' ) ) { wp_die( 'Unauthorized.' ); }

        $certificates = $this->database->get_all_certificates_for_export();
        $backup_data = array(
            'created_at' => gmdate( 'Y-m-d H:i:s' ),
            'certificates' => $certificates,
        );

        $json_data = wp_json_encode( $backup_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE );
        $filename = 'tcvs_backup_' . gmdate( 'Y-m-d_H-i-s' ) . '.json';

        header( 'Content-Type: application/json; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename="' . $filename . '"' );
        
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
        echo $json_data;
        exit;
    }

    public function restore_backup(): void {
        if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'tcvs_restore_backup' ) ) { wp_die( 'Security check failed.' ); }
        if ( ! current_user_can( 'manage_options' ) ) { wp_die( 'Unauthorized.' ); }

        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        $file = $_FILES['backup_file'] ?? [];
        
        if ( empty($file) || $file['error'] !== UPLOAD_ERR_OK ) {
            wp_safe_redirect( admin_url( 'admin.php?page=certificate-backup&message=upload_error' ) ); exit;
        }

        $content = file_get_contents( $file['tmp_name'] );
        $data = json_decode( $content, true );

        if ( ! $data || ! isset( $data['certificates'] ) ) {
            wp_safe_redirect( admin_url( 'admin.php?page=certificate-backup&message=invalid_json' ) ); exit;
        }

        foreach ( $data['certificates'] as $cert ) {
            if ( isset( $cert['certificate_number'] ) && ! $this->database->get_certificate_by_number( $cert['certificate_number'] ) ) {
                $this->database->insert_certificate( $cert );
            }
        }
        
        // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink
        @unlink( $file['tmp_name'] );
        
        wp_safe_redirect( admin_url( 'admin.php?page=certificate-backup&message=restore_report' ) );
        exit;
    }
}