<?php
/**
 * Handles all administrative UI and form submissions.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Tohidul_Certificate_Verification_System_Admin {
    
    private $database;
    
    public function __construct( Tohidul_Certificate_Verification_System_Database $database ) {
        $this->database = $database;
        
        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
        add_action( 'admin_init', array( $this, 'handle_bulk_actions' ) );
        add_action( 'admin_post_tcvs_add_certificate', array( $this, 'handle_add_certificate' ) );
        add_action( 'admin_post_tcvs_edit_certificate', array( $this, 'handle_edit_certificate' ) );
        add_action( 'admin_post_tcvs_delete_certificate', array( $this, 'handle_delete_certificate' ) );
        add_action( 'admin_post_tcvs_save_institute_settings', array( $this, 'handle_save_institute_settings' ) );
        add_action( 'admin_notices', array( $this, 'show_admin_notices' ) );
    }
    
    public function enqueue_admin_assets( $hook ) {
        if ( strpos( $hook, 'certificate-' ) !== false ) {
            wp_enqueue_media();
            wp_enqueue_style( 'tcvs-admin-style', TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_URL . 'assets/css/admin-style.css', array(), TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_VERSION );
            wp_enqueue_script( 'tcvs-admin-script', TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_URL . 'assets/js/admin-script.js', array( 'jquery' ), TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_VERSION, true );
            wp_localize_script( 'tcvs-admin-script', 'tcvs_admin_obj', array(
                'confirm_delete' => esc_html__( 'Are you sure you want to delete this certificate? This action cannot be undone.', 'tohidul-certificate-verification-system' ),
            ));
        }
    }

    public function add_admin_menu(): void {
        add_menu_page('Certificate Verification', 'Certificates', 'manage_options', 'certificate-verification', array($this, 'admin_dashboard'), 'dashicons-awards', 30);
        add_submenu_page('certificate-verification', 'Add New', 'Add New', 'manage_options', 'certificate-add', array($this, 'add_certificate_page'));
        add_submenu_page('certificate-verification', 'Import/Export', 'Import/Export', 'manage_options', 'certificate-import-export', array($this, 'import_export_page'));
        add_submenu_page('certificate-verification', 'Institute Settings', 'Institute Settings', 'manage_options', 'certificate-settings', array($this, 'settings_page'));
        add_submenu_page('certificate-verification', 'Backup/Restore', 'Backup/Restore', 'manage_options', 'certificate-backup', array($this, 'backup_page'));
    }
    
    public function admin_dashboard(): void {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
        $limit = 20; 
        $offset = ($page - 1) * $limit;
        
        $total = $this->database->get_total_certificates();
        $total_pages = ceil($total / $limit);
        $certificates = $this->database->get_all_certificates($limit, $offset);
        
        include TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_PATH . 'templates/admin-dashboard.php';
    }
    
    public function add_certificate_page(): void {
        $certificate = null; $is_edit = false;
        $form_action = 'tcvs_add_certificate';
        $button_text = 'Add Certificate'; $form_title = 'Add New Certificate';
        
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( isset($_GET['action'], $_GET['id']) && 'edit' === $_GET['action'] ) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $certificate_id = intval( $_GET['id'] );
            
            // Verify Nonce for Edit Action
            check_admin_referer( 'tcvs_edit_cert_' . $certificate_id );

            $certificate = $this->database->get_certificate( $certificate_id );
            if($certificate) {
                $is_edit = true; 
                $form_action = 'tcvs_edit_certificate';
                $button_text = 'Update Certificate'; 
                $form_title = 'Edit Certificate';
            }
        }
        include TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_PATH . 'templates/certificate-form.php';
    }

    public function import_export_page(): void { 
        $ei = new Tohidul_Certificate_Verification_System_Export_Import($this->database); 
        $ei->render_page(); 
    }
    
    public function settings_page(): void { 
        $s = new Tohidul_Certificate_Verification_System_Settings($this->database); 
        $s->settings_page(); 
    }
    
    public function backup_page(): void { 
        $b = new Tohidul_Certificate_Verification_System_Backup($this->database); 
        $b->render_page(); 
    }

    /**
     * Helper function to get POST data.
     * Nonce verification is handled by the caller functions (handle_add_certificate, handle_edit_certificate).
     */
    private function get_post_data() {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $issue_date = ! empty( $_POST['issue_date'] ) ? sanitize_text_field( wp_unslash( $_POST['issue_date'] ) ) : current_time( 'Y-m-d' );
        
        return array(
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'student_name'       => sanitize_text_field( wp_unslash( $_POST['student_name'] ?? '' ) ),
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'father_name'        => sanitize_text_field( wp_unslash( $_POST['father_name'] ?? '' ) ),
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'student_image'      => esc_url_raw( wp_unslash( $_POST['student_image'] ?? '' ) ),
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'certificate_number' => sanitize_text_field( wp_unslash( $_POST['certificate_number'] ?? '' ) ),
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'batch_number'       => sanitize_text_field( wp_unslash( $_POST['batch_number'] ?? '' ) ),
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'course_name'        => sanitize_text_field( wp_unslash( $_POST['course_name'] ?? '' ) ),
            'issue_date'         => $issue_date,
        );
    }

    public function handle_add_certificate(): void {
        if (!isset($_POST['tcvs_certificate_nonce']) || !wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['tcvs_certificate_nonce'] ) ), 'tcvs_add_certificate')) { wp_die('Security check failed.'); }
        if (!current_user_can('manage_options')) { wp_die('Unauthorized'); }

        $data = $this->get_post_data();
        if ($this->database->get_certificate_by_number($data['certificate_number'])) { wp_die('Error: Certificate number already exists.'); }
        
        if ($this->database->insert_certificate($data)) { 
            wp_safe_redirect(admin_url('admin.php?page=certificate-verification&message=added')); 
        } else { 
            global $wpdb; 
            wp_die('Database Error: ' . esc_html($wpdb->last_error)); 
        } 
        exit;
    }
    
    public function handle_edit_certificate(): void {
        if (!isset($_POST['tcvs_certificate_nonce']) || !wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['tcvs_certificate_nonce'] ) ), 'tcvs_edit_certificate')) { wp_die('Security check failed.'); }
        if (!current_user_can('manage_options')) { wp_die('Unauthorized'); }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $id = isset( $_POST['certificate_id'] ) ? intval( wp_unslash( $_POST['certificate_id'] ) ) : 0;
        $data = $this->get_post_data();
        
        if ($this->database->update_certificate($id, $data)) { 
            wp_safe_redirect(admin_url('admin.php?page=certificate-verification&message=updated')); 
        } else { 
            wp_safe_redirect(admin_url('admin.php?page=certificate-verification&message=error')); 
        } 
        exit;
    }

    public function handle_delete_certificate(): void {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $nonce = isset($_POST['tcvs_delete_nonce']) ? sanitize_text_field( wp_unslash( $_POST['tcvs_delete_nonce'] ) ) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $cert_id = isset($_POST['certificate_id']) ? intval( wp_unslash( $_POST['certificate_id'] ) ) : 0;
        
        if( !$cert_id ) wp_die('Invalid ID');
        if (!wp_verify_nonce($nonce, 'tcvs_delete_certificate_' . $cert_id)) { wp_die('Security check failed.'); }
        if (!current_user_can('manage_options')) { wp_die('Unauthorized'); }

        if ($this->database->delete_certificate($cert_id)) { 
            wp_safe_redirect(admin_url('admin.php?page=certificate-verification&message=deleted')); 
        } else { 
            wp_safe_redirect(admin_url('admin.php?page=certificate-verification&message=error')); 
        } 
        exit;
    }

    public function handle_bulk_actions() {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( ! isset( $_GET['page'] ) || $_GET['page'] !== 'certificate-verification' ) { return; }
        
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated
        if ( ! isset( $_SERVER['REQUEST_METHOD'] ) || 'POST' !== $_SERVER['REQUEST_METHOD'] ) {
            return;
        }

        $action = '';
        
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        if ( isset( $_POST['action'] ) && $_POST['action'] !== '-1' ) { 
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            $action = sanitize_text_field( wp_unslash( $_POST['action'] ) ); 
        
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        } elseif ( isset( $_POST['action2'] ) && $_POST['action2'] !== '-1' ) { 
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            $action = sanitize_text_field( wp_unslash( $_POST['action2'] ) ); 
        }

        if ( 'tcvs_bulk_delete' === $action ) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            if ( ! isset( $_POST['tcvs_admin_nonce_field'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['tcvs_admin_nonce_field'] ) ), 'tcvs_admin_nonce' ) ) { return; }
            
            if ( ! current_user_can('manage_options') ) { return; }

            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            $ids = isset( $_POST['certificate_ids'] ) ? array_map( 'intval', wp_unslash( $_POST['certificate_ids'] ) ) : array();
            
            if ( ! empty( $ids ) ) {
                $this->database->bulk_delete_certificates( $ids );
                wp_safe_redirect( add_query_arg( array( 'page' => 'certificate-verification', 'message' => 'deleted' ), admin_url( 'admin.php' ) ) );
                exit;
            } else {
                wp_safe_redirect( add_query_arg( array( 'page' => 'certificate-verification' ), admin_url( 'admin.php' ) ) );
                exit;
            }
        }
    }

    public function handle_save_institute_settings(): void {
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated
        if ( ! isset( $_SERVER['REQUEST_METHOD'] ) || 'POST' !== $_SERVER['REQUEST_METHOD'] ) {
            return;
        }

        if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'tcvs_save_institute_settings' ) ) { wp_die( 'Security check failed.' ); }
        if ( ! current_user_can('manage_options') ) { wp_die( 'Unauthorized.' ); }

        $data = array(
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'institute_name'    => sanitize_text_field( wp_unslash( $_POST['institute_name'] ?? '' ) ),
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'institute_logo'    => esc_url_raw( wp_unslash( $_POST['institute_logo'] ?? '' ) ),
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'institute_address' => sanitize_textarea_field( wp_unslash( $_POST['institute_address'] ?? '' ) ),
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'institute_phone'   => sanitize_text_field( wp_unslash( $_POST['institute_phone'] ?? '' ) ),
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'institute_email'   => sanitize_email( wp_unslash( $_POST['institute_email'] ?? '' ) ),
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'signature_image'   => esc_url_raw( wp_unslash( $_POST['signature_image'] ?? '' ) ),
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            'verified_by_image' => esc_url_raw( wp_unslash( $_POST['verified_by_image'] ?? '' ) ),
        );
        
        $this->database->update_institute_settings( $data );
        wp_safe_redirect( admin_url( 'admin.php?page=certificate-settings&message=saved' ) ); 
        exit;
    }
    
    public function show_admin_notices(): void {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if (isset($_GET['message'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $msg = sanitize_text_field( wp_unslash( $_GET['message'] ) );
            $class = ($msg === 'error') ? 'notice-error' : 'notice-success';
            $text = 'Action completed.';
            
            if ($msg === 'deleted') $text = 'Item(s) deleted successfully.';
            elseif ($msg === 'added') $text = 'Added successfully.';
            elseif ($msg === 'updated') $text = 'Updated successfully.';
            elseif ($msg === 'saved') $text = 'Settings saved successfully.';
            elseif ($msg === 'error') $text = 'An error occurred. Please try again.';
            
            printf('<div class="notice %1$s is-dismissible"><p>%2$s</p></div>', esc_attr( $class ), esc_html( $text ));
        }
    }
}