(function($) {
    'use strict';

    const TCVS = {
        config: {
            ajaxUrl: (typeof tcvs_ajax !== 'undefined') ? tcvs_ajax.ajax_url : '',
            nonce: (typeof tcvs_ajax !== 'undefined') ? tcvs_ajax.nonce : ''
        }
    };

    $(document).ready(function() {
        if (!TCVS.config.ajaxUrl) return;
        setupEventListeners();
    });

    function setupEventListeners() {
        $(document).on('submit', '#tcvs-verification-form', handleVerificationSubmit);
        
        $(document).on('click', '.print-button, .tcvs-btn-print', function(e) {
            e.preventDefault();
            printCertificateArea();
        });

        $(document).on('click', '.back-to-search, .tcvs-btn-back', function(e) {
            e.preventDefault();
            $('#tcvs-result').slideUp();
            $('.tcvs-messages').empty();
            $('#tcvs_certificate_number').val('');
        });
        
        $(document).on('input', '#tcvs_certificate_number', function() {
            $(this).val($(this).val().toUpperCase().replace(/\s+/g, ''));
        });
    }

    function handleVerificationSubmit(e) {
        e.preventDefault();
        const btn = $(this).find('.tcvs-submit-button');
        const certNo = $('#tcvs_certificate_number').val().trim();
        const resultBox = $('#tcvs-result');
        const msgBox = $('.tcvs-messages').first();

        if (!certNo) {
            showMessage(msgBox, 'error', 'Please enter a certificate number.');
            return;
        }

        btn.prop('disabled', true).text('Verifying...');
        resultBox.hide();
        msgBox.empty();

        $.ajax({
            url: TCVS.config.ajaxUrl,
            type: 'POST',
            data: {
                action: 'tcvs_verify_certificate',
                certificate_number: certNo,
                nonce: TCVS.config.nonce
            },
            success: function(response) {
                if (response.success) {
                    resultBox.html(response.data.html).slideDown();
                    showMessage(msgBox, 'success', 'Certificate verified successfully!');
                    $('html, body').animate({ scrollTop: resultBox.offset().top - 50 }, 500);
                } else {
                    showMessage(msgBox, 'error', response.data || 'Certificate not found.');
                }
            },
            error: function() {
                showMessage(msgBox, 'error', 'Server error. Please try again.');
            },
            complete: function() {
                btn.prop('disabled', false).text('Verify Now');
            }
        });
    }

    function printCertificateArea() {
        const $certContent = $('#tcvs-result').clone();
        $certContent.find('.tcvs-action-buttons, .no-print').remove();
        
        const printContent = $certContent.html();
        if (!printContent) { return; }

        $('#tcvs-print-frame').remove();
        const $iframe = $('<iframe id="tcvs-print-frame" name="tcvs-print-frame"></iframe>');
        $iframe.css({ position: 'fixed', top: '-10000px', left: '-10000px' });
        $('body').append($iframe);

        let cssUrl = '';
        $('link[rel="stylesheet"]').each(function() {
            const href = $(this).attr('href');
            if (href && (href.indexOf('assets/css/style.css') !== -1 || href.indexOf('certificate-verification') !== -1)) {
                cssUrl = href;
            }
        });

        const frameDoc = $iframe[0].contentWindow.document;
        frameDoc.open();
        frameDoc.write(`
            <!DOCTYPE html>
            <html>
            <head>
                <title>Print Certificate</title>
                <link href="https://fonts.googleapis.com/css2?family=Cinzel:wght@700&family=Great+Vibes&family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
                ${cssUrl ? `<link rel="stylesheet" href="${cssUrl}" type="text/css" media="all">` : ''}
                <style>
                    body { margin: 0; padding: 0; background: #fff !important; }
                    @page { size: A4 landscape; margin: 0; }
                    .tcvs-print-view { display: block !important; width: 100%; height: 100%; }
                    .tcvs-screen-view { display: none !important; }
                    * { -webkit-print-color-adjust: exact !important; print-color-adjust: exact !important; }
                    .cert-page { width: 100%; max-width: 297mm; margin: 0 auto; page-break-after: always; }
                </style>
            </head>
            <body>
                <div class="tcvs-print-view">${printContent}</div>
            </body>
            </html>
        `);
        frameDoc.close();

        $iframe.on('load', function() {
            setTimeout(function() {
                const frameWindow = $iframe[0].contentWindow;
                frameWindow.focus();
                try { frameWindow.print(); } catch (e) { console.error("Print failed:", e); }
            }, 800);
        });
    }

    function showMessage(container, type, message) {
        container.html(`<div class="tcvs-message ${type}" style="padding:10px; margin-bottom:10px; border-radius:4px; border:1px solid ${type === 'error' ? '#f5c6cb' : '#c3e6cb'}; background:${type === 'error' ? '#f8d7da' : '#d4edda'}; color:${type === 'error' ? '#721c24' : '#155724'};">${message}</div>`).show();
    }
})(jQuery);