<?php
/**
 * Plugin Name: To8gal Fullslider
 * Description: Fullscreen continuous horizontal slider with overlay and a CTA button.
 * Version: 2.2.3
 * Author: to8gal
 * Text Domain: to8gal-fullslider
 * License: GPLv2 or later
 */

if ( ! defined('ABSPATH') ) exit;

require_once plugin_dir_path(__FILE__) . 'includes/class-fullslider.php';


class To8gal_Fullscreen_Slider {

    private $option_name = 'to8gal_fullslider_options';

    public function __construct() {
        add_action( 'init', [ $this, 'register_post_type' ] );
        add_action( 'admin_menu', [ $this, 'add_settings_page' ] );
        add_action( 'admin_init', [ $this, 'register_settings' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'admin_assets' ] );
        add_action( 'wp_enqueue_scripts', [ $this, 'frontend_assets' ] );
        add_shortcode( 'to8gal_fullslider', [ $this, 'render_shortcode' ] );
        add_action( 'plugins_loaded', [ $this, 'load_textdomain' ] );
        // Add Pro link under plugin name
        add_filter('plugin_row_meta', 'to8gal_fullslider_pro_link', 10, 2);
        function to8gal_fullslider_pro_link($links, $file) {
            if ($file === plugin_basename(__FILE__)) {
                $pro_link = '<a href="https://to8gal.gumroad.com/l/fullslider" target="_blank" class="to8gal-pro-link">🌟 Upgrade to Pro 🌟</a>';
                $links[] = $pro_link;
            }
            return $links;
        }

        // Add Settings and Pro links beside plugin name
            add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'to8gal_fullslider_action_links');
            function to8gal_fullslider_action_links($links) {

                // Settings link
                $settings_link = '<a href="options-general.php?page=to8gal-fullslider">Settings</a>';

                // Pro link (catchy, styled)
                if (!defined('TO8GAL_FULLSLIDER_PRO')) { // only show in free version
                    $pro_link = '<a href="https://to8gal.gumroad.com/l/fullslider" target="_blank" class="to8gal-pro-action">🌟 Upgrade to Pro 🌟</a>';
                    array_unshift($links, $settings_link, $pro_link); // Add to left of Activate
                } else {
                    array_unshift($links, $settings_link);
                }

                return $links;
            }




    }

    public function load_textdomain() {
        
    }

    public function register_post_type() {
    $labels = [
        'name'                  => __( 'Fullscreen Slides', 'to8gal-fullslider' ),
        'singular_name'         => __( 'Fullscreen Slide', 'to8gal-fullslider' ),
        'menu_name'             => __( 'Fullscreen Slides', 'to8gal-fullslider' ),
        'name_admin_bar'        => __( 'Fullscreen Slide', 'to8gal-fullslider' ),
        'add_new'               => __( 'Add New', 'to8gal-fullslider' ),
        'add_new_item'          => __( 'Add New Slide', 'to8gal-fullslider' ),
        'edit_item'             => __( 'Edit Slide', 'to8gal-fullslider' ),
        'new_item'              => __( 'New Slide', 'to8gal-fullslider' ),
        'view_item'             => __( 'View Slide', 'to8gal-fullslider' ),
        'search_items'          => __( 'Search Slides', 'to8gal-fullslider' ),
        'not_found'             => __( 'No slides found', 'to8gal-fullslider' ),
        'not_found_in_trash'    => __( 'No slides found in Trash', 'to8gal-fullslider' ),
        'all_items'             => __( 'All Slides', 'to8gal-fullslider' ),
    ];

    $args = [
        'labels'                => $labels,
        'public'                => false,       // not publicly queryable
        'show_ui'               => true,        // show in admin
        'show_in_menu'          => true,
        'supports'              => [ 'title', 'thumbnail', 'page-attributes' ],
        'menu_icon'             => 'dashicons-images-alt2',
        'capability_type'       => 'post',
        'hierarchical'          => false,
        'has_archive'           => false,
        'rewrite'               => false,
        'show_in_rest'          => false,
    ];

    register_post_type( 'to8gal_slide', $args );
	}


    public function add_settings_page() {
        add_options_page(
            __( 'Fullscreen Slider', 'to8gal-fullslider' ),
            __( 'Fullscreen Slider', 'to8gal-fullslider' ),
            'manage_options',
            'to8gal-fullslider',
            [ $this, 'settings_page_html' ]
        );
    }

    public function register_settings() {
        register_setting( $this->option_name, $this->option_name, [ $this, 'validate_options' ] );
        add_settings_section( 'to8gal_main', __( 'Overlay & CTA settings', 'to8gal-fullslider' ), '__return_false', 'to8gal-fullslider' );

        add_settings_field( 'overlay_color', __( 'Overlay color (hex)', 'to8gal-fullslider' ), [ $this, 'field_color' ], 'to8gal-fullslider', 'to8gal_main' );
        add_settings_field( 'overlay_opacity', __( 'Overlay opacity (0-1)', 'to8gal-fullslider' ), [ $this, 'field_opacity' ], 'to8gal-fullslider', 'to8gal_main' );
        add_settings_field( 'cta_text', __( 'CTA Button Text', 'to8gal-fullslider' ), [ $this, 'field_cta_text' ], 'to8gal-fullslider', 'to8gal_main' );
        add_settings_field( 'cta_url', __( 'CTA URL', 'to8gal-fullslider' ), [ $this, 'field_cta_url' ], 'to8gal-fullslider', 'to8gal_main' );
        add_settings_field( 'cta_position', __( 'CTA Position', 'to8gal-fullslider' ), [ $this, 'field_cta_position' ], 'to8gal-fullslider', 'to8gal_main' );
        // Instructions / general description field
        // add_settings_field( 'general_description', __( 'Instructions', 'to8gal-fullslider' ), [ $this, 'field_general_description' ], 'to8gal-fullslider', 'to8gal_main');
        
    }

    // public function field_general_description() {
    //$opts = get_option( $this->option_name, [] );
    //$val = $opts['general_description'] ?? '';
    //echo '<p class="description">' . esc_html__('Use this space to show instructions for the slider.', //'to8gal-fullslider') . '</p>';
    //}



    public function validate_options( $input ) {
        $out = [];
        $out['overlay_color'] = preg_match( '/^#?[0-9a-fA-F]{3,6}$/', $input['overlay_color'] ?? '' ) ? ltrim( $input['overlay_color'], '#' ) : '000000';
        $opacity = floatval( $input['overlay_opacity'] ?? 0.5 );
        $out['overlay_opacity'] = min(1, max(0, $opacity));
        $out['cta_text'] = sanitize_text_field( $input['cta_text'] ?? '' );
        $out['cta_url'] = esc_url_raw( $input['cta_url'] ?? '' );
        $allowed_positions = [ 'top-left','top-center','top-right','center-left','center','center-right','bottom-left','bottom-center','bottom-right' ];
        $out['cta_position'] = in_array( $input['cta_position'] ?? 'top-right', $allowed_positions, true ) ? $input['cta_position'] : 'top-right';

        // Preserve **all extra fields**, including Pro fields
        foreach($input as $k=>$v){
            if(!isset($out[$k])) {
                $out[$k] = $v;
            }
        }
        return $out;
    }

    public function field_color() {
        $opts = get_option( $this->option_name, [] );
        $val = isset( $opts['overlay_color'] ) ? '#'.esc_attr( $opts['overlay_color'] ) : '#000000';
        echo '<input type="text" name="'.esc_attr($this->option_name).'[overlay_color]" value="'.esc_attr($val).'" class="regular-text" />';
    }

    public function field_opacity() {
        $opts = get_option( $this->option_name, [] );
        $val = isset( $opts['overlay_opacity'] ) ? esc_attr( $opts['overlay_opacity'] ) : '0.5';
        echo '<input type="number" step="0.05" min="0" max="1" name="'.esc_attr($this->option_name).'[overlay_opacity]" value="'.esc_attr($val).'" />';
    }

    public function field_cta_text() {
        $opts = get_option( $this->option_name, [] );
        $val = $opts['cta_text'] ?? '';
        echo '<input type="text" name="'.esc_attr($this->option_name).'[cta_text]" value="'.esc_attr($val).'" class="regular-text" />';
    }

    public function field_cta_url() {
        $opts = get_option( $this->option_name, [] );
        $val = $opts['cta_url'] ?? '';
        echo '<input type="url" name="'.esc_attr($this->option_name).'[cta_url]" value="'.esc_attr($val).'" class="regular-text" />';
    }

    public function field_cta_position() {
        $opts = get_option( $this->option_name, [] );
        $val = $opts['cta_position'] ?? 'top-right';
        $positions = [
            'top-left' => 'Top Left','top-center' => 'Top Center','top-right' => 'Top Right',
            'center-left' => 'Center Left','center' => 'Center','center-right' => 'Center Right',
            'bottom-left' => 'Bottom Left','bottom-center' => 'Bottom Center','bottom-right' => 'Bottom Right',
        ];
        echo '<select name="'.esc_attr($this->option_name).'[cta_position]">';
        foreach ( $positions as $k => $label ) {
            printf('<option value="%s"%s>%s</option>', esc_attr($k), selected( $val, $k, false ), esc_html( $label ) );
        }
        echo '</select>';
    }

    public function settings_page_html() {
        if ( ! current_user_can( 'manage_options' ) ) return;

        $opts = get_option( $this->option_name, [] );
        $instructions = $opts['general_description'] ?? 'To create your slide go to: Sidebar --> Fullscren Slides';
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'FullSlider', 'to8gal-fullslider' ); ?></h1>
             <?php if (!defined('TO8GAL_FULLSLIDER_PRO')): ?>
                <a href="https://to8gal.gumroad.com/l/fullslider" target="_blank" class="to8gal-pro-admin-link">🌟 Upgrade to Pro 🌟</a>
            <?php endif; ?>

            <?php if ( $instructions ) : ?>
            <p style="font-size:16px; padding:8px 8px; background:#f9f9f9; border-left:4px solid #0073aa;">
                <?php echo esc_html( $instructions ); ?>
            </p>
        <?php endif; ?>
            <form method="post" action="options.php">
                <?php
                settings_fields( $this->option_name );
                do_settings_sections( 'to8gal-fullslider' );
                submit_button();
                ?>
            </form>
            <p><?php esc_html_e( 'Use shortcode [to8gal_fullslider] in a page to display the slider.', 'to8gal-fullslider' ); ?></p>
        </div>
        <?php
    }

    public function admin_assets( $hook ) {
        if ( strpos( $hook, 'settings_page_to8gal-fullslider' ) === false ) return;
        wp_enqueue_media();
        wp_enqueue_script( 'to8gal-admin', plugins_url( 'admin.js', __FILE__ ), [ 'jquery' ], '0.2', true );
        wp_enqueue_style( 'to8gal-admin-style', plugins_url( 'admin.css', __FILE__ ) );
    }

    public function frontend_assets() {
        wp_enqueue_style( 'to8gal-frontend', plugins_url( 'frontend.css', __FILE__ ), [], '0.2' );
        wp_enqueue_script( 'to8gal-frontend', plugins_url( 'frontend.js', __FILE__ ), [ 'jquery' ], '0.2', true );
        $opts = get_option( $this->option_name, [] );
        wp_localize_script( 'to8gal-frontend', 'to8gal_front', [
        'overlay_color'  => $opts['overlay_color'] ?? '000000',
        'overlay_opacity'=> (float) ($opts['overlay_opacity'] ?? 0.5),
        'cta_text'       => $opts['cta_text'] ?? '',
        'cta_url'        => $opts['cta_url'] ?? '',
        'cta_position'   => $opts['cta_position'] ?? 'top-right',
        // Pro options
        'slide_height'   => $opts['slide_height'] ?? '100vh',
        'pause_on_hover' => !empty($opts['pause_on_hover']) ? true : false,
        'shuffle_slides' => !empty($opts['shuffle_slides']) ? true : false,
        'speed'          => absint($opts['speed'] ?? 30000),
        'general_description' => $opts['general_description'] ?? '',
        ] );
    }

    private function hex_to_rgba( $hex, $opacity = 0.5 ) {
        $hex = ltrim( $hex, '#' );
        if ( strlen( $hex ) === 3 ) {
            $r = hexdec(str_repeat($hex[0],2));
            $g = hexdec(str_repeat($hex[1],2));
            $b = hexdec(str_repeat($hex[2],2));
        } else {
            $r = hexdec(substr($hex,0,2));
            $g = hexdec(substr($hex,2,2));
            $b = hexdec(substr($hex,4,2));
        }
        return "rgba($r,$g,$b,{$opacity})";
    }

    public function render_shortcode( $atts = [] ) {
    $query = new WP_Query([
        'post_type' => 'to8gal_slide',
        'posts_per_page' => -1,
        'orderby' => 'menu_order',
        'order' => 'ASC',
    ]);
    if ( ! $query->have_posts() ) return '<!-- no slides -->';

    $opts = get_option( $this->option_name, [] );

    ob_start(); ?>
    <div class="to8gal-fullslider" data-overlay="<?php echo esc_attr( $this->hex_to_rgba( $opts['overlay_color'] ?? '000000', $opts['overlay_opacity'] ?? 0.5 ) ); ?>" style="position:relative;">
        <div class="to8gal-fullslider-track">
            <?php 
            $slides = [];
            while($query->have_posts()): $query->the_post();
                $thumb_url = get_the_post_thumbnail_url(get_the_ID(),'full');
                if($thumb_url) {
                    $slides[] = '<div class="slide" style="background-image:url(' . esc_url($thumb_url) . ');"></div>';
                }
            endwhile; 
            wp_reset_postdata();

            // Output slides twice for seamless loop, safely
            echo wp_kses_post( implode('', $slides) . implode('', $slides) ); 
            ?>
        </div>

        <div class="to8gal-overlay" style="background:<?php echo esc_attr( $this->hex_to_rgba( $opts['overlay_color'] ?? '000000', $opts['overlay_opacity'] ?? 0.5 ) ); ?>;position:absolute;top:0;left:0;width:100%;height:100%;"></div>

        <?php if ( ! empty( $opts['cta_text'] ) && ! empty( $opts['cta_url'] ) ): ?>
            <a href="<?php echo esc_url( $opts['cta_url'] ); ?>" class="to8gal-cta <?php echo esc_attr( $opts['cta_position'] ?? 'top-right' ); ?>">
                <?php echo esc_html( $opts['cta_text'] ); ?>
            </a>
        <?php endif; ?>
    </div>
    <?php

    return ob_get_clean();
}


}

new To8gal_Fullscreen_Slider();

