<?php
/**
 * Plugin Name: Tiny Talk
 * Description: Add Tiny Talk AI agents to your WordPress site. Supports multiple agents with rule-based assignment.
 * Version: 1.0.2
 * Author: Tiny Talk
 * Author URI: https://tinytalk.ai
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: tiny-talk
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.4
 *
 * @package TinyTalk
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

define( 'TINY_TALK_VERSION', '1.0.2' );
define( 'TINY_TALK_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'TINY_TALK_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'TINY_TALK_SDK_URL', 'https://cdn.tinytalk.ai/@tinytalk/browser/0.0.16/tiny-talk-sdk.min.umd.js' );
define( 'TINY_TALK_DASHBOARD_URL', 'https://dashboard.tinytalk.ai' );
define( 'TINY_TALK_WEBSITE_URL', 'https://tinytalk.ai' );

/**
 * Check if a string is a valid UUID v4 format.
 *
 * @param string $value The value to check.
 * @return bool
 */
function tiny_talk_is_valid_uuid( $value ) {
	return (bool) preg_match( '/^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i', $value );
}

// Include required files.
require_once TINY_TALK_PLUGIN_DIR . 'includes/class-tiny-talk-admin.php';
require_once TINY_TALK_PLUGIN_DIR . 'includes/class-tiny-talk-rules.php';
require_once TINY_TALK_PLUGIN_DIR . 'includes/class-tiny-talk-meta-box.php';
require_once TINY_TALK_PLUGIN_DIR . 'includes/class-tiny-talk-shortcode.php';

/**
 * Main Tiny Talk Plugin Class
 */
class TinyTalk {

	/**
	 * Single instance of the class.
	 *
	 * @var TinyTalk|null
	 */
	private static $instance = null;

	/**
	 * Admin instance.
	 *
	 * @var TinyTalk_Admin
	 */
	public $admin;

	/**
	 * Rules engine instance.
	 *
	 * @var TinyTalk_Rules
	 */
	public $rules;

	/**
	 * Meta box instance.
	 *
	 * @var TinyTalk_Meta_Box
	 */
	public $meta_box;

	/**
	 * Shortcode instance.
	 *
	 * @var TinyTalk_Shortcode
	 */
	public $shortcode;

	/**
	 * Agent ID for the widget script tag.
	 *
	 * @var string|null
	 */
	private $widget_agent_id = null;

	/**
	 * Whether the script_loader_tag filter has been registered.
	 *
	 * @var bool
	 */
	private $widget_filter_added = false;

	/**
	 * Get single instance of the class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		$this->init_components();
		$this->init_hooks();
	}

	/**
	 * Initialize components.
	 */
	private function init_components() {
		$this->admin     = new TinyTalk_Admin();
		$this->rules     = new TinyTalk_Rules();
		$this->meta_box  = new TinyTalk_Meta_Box();
		$this->shortcode = new TinyTalk_Shortcode();
	}

	/**
	 * Initialize hooks.
	 */
	private function init_hooks() {
		add_action( 'wp_footer', array( $this, 'render_widget' ) );
		// Activation and deactivation hooks.
		register_activation_hook( __FILE__, array( $this, 'activate' ) );
		register_deactivation_hook( __FILE__, array( $this, 'deactivate' ) );
	}

	/**
	 * Plugin activation.
	 */
	public function activate() {
		// Set default options.
		if ( false === get_option( 'tiny_talk_agents' ) ) {
			update_option( 'tiny_talk_agents', array() );
		}
		if ( false === get_option( 'tiny_talk_rules' ) ) {
			update_option( 'tiny_talk_rules', array() );
		}
		if ( false === get_option( 'tiny_talk_default_agent' ) ) {
			update_option( 'tiny_talk_default_agent', '' );
		}
		if ( false === get_option( 'tiny_talk_enabled' ) ) {
			update_option( 'tiny_talk_enabled', true );
		}

		// Flush rewrite rules.
		flush_rewrite_rules();
	}

	/**
	 * Plugin deactivation.
	 */
	public function deactivate() {
		flush_rewrite_rules();
	}

	/**
	 * Set the agent ID for the widget and enqueue the SDK script.
	 * Only registers the script_loader_tag filter once, no matter how many callers invoke this.
	 *
	 * @param string $agent_id Agent UUID.
	 */
	public function set_widget_agent_id( $agent_id ) {
		$this->widget_agent_id = $agent_id;

		if ( ! $this->widget_filter_added ) {
			wp_enqueue_script(
				'tiny-talk-sdk',
				TINY_TALK_SDK_URL,
				array(),
				TINY_TALK_VERSION,
				array(
					'in_footer' => true,
					'strategy'  => 'defer',
				)
			);

			add_filter( 'script_loader_tag', array( $this, 'filter_sdk_script_tag' ), 10, 2 );
			$this->widget_filter_added = true;
		}
	}

	/**
	 * Add the data-tiny-bot-id attribute to the SDK script tag.
	 *
	 * @param string $tag    Script HTML tag.
	 * @param string $handle Script handle.
	 * @return string Modified tag.
	 */
	public function filter_sdk_script_tag( $tag, $handle ) {
		if ( 'tiny-talk-sdk' === $handle && $this->widget_agent_id ) {
			$tag = str_replace( ' src=', ' data-tiny-bot-id="' . esc_attr( $this->widget_agent_id ) . '" src=', $tag );
		}
		return $tag;
	}

	/**
	 * Render the widget script in footer.
	 */
	public function render_widget() {
		// Check if plugin is enabled.
		if ( ! get_option( 'tiny_talk_enabled', true ) ) {
			return;
		}

		// Don't render in admin.
		if ( is_admin() ) {
			return;
		}

		// Don't override if a shortcode widget already set the agent.
		if ( null !== $this->widget_agent_id ) {
			return;
		}

		$agent_id = $this->get_current_agent_id();

		if ( empty( $agent_id ) ) {
			return;
		}

		$this->set_widget_agent_id( $agent_id );
	}

	/**
	 * Get the agent ID for the current page.
	 * Priority: Page override > Rules > Default.
	 */
	public function get_current_agent_id() {
		// Check for page-level override.
		$post_id = get_the_ID();
		if ( $post_id ) {
			$page_agent = get_post_meta( $post_id, '_tiny_talk_agent_id', true );

			// Explicitly disabled.
			if ( 'none' === $page_agent ) {
				return null;
			}

			// Specific agent selected.
			if ( ! empty( $page_agent ) && 'default' !== $page_agent ) {
				return $page_agent;
			}
		}

		// Check rules.
		$rule_agent = $this->rules->get_matching_agent();
		if ( ! empty( $rule_agent ) ) {
			return $rule_agent;
		}

		// Fall back to default.
		return get_option( 'tiny_talk_default_agent', '' );
	}

	/**
	 * Get all registered agents.
	 */
	public function get_agents() {
		return get_option( 'tiny_talk_agents', array() );
	}

	/**
	 * Get an agent by ID.
	 *
	 * @param string $agent_id Agent UUID.
	 */
	public function get_agent( $agent_id ) {
		$agents = $this->get_agents();
		foreach ( $agents as $agent ) {
			if ( $agent['id'] === $agent_id ) {
				return $agent;
			}
		}
		return null;
	}

	/**
	 * Get agent name by ID.
	 *
	 * @param string $agent_id Agent UUID.
	 */
	public function get_agent_name( $agent_id ) {
		$agent = $this->get_agent( $agent_id );
		return $agent ? $agent['name'] : $agent_id;
	}
}

/**
 * Get the main plugin instance.
 */
function tiny_talk() {
	return TinyTalk::get_instance();
}

// Initialize the plugin.
tiny_talk();
