<?php
/**
 * Tiny Talk Shortcode Handler.
 *
 * Provides [tinytalk] shortcode for embedding the agent.
 *
 * @package TinyTalk
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Registers and renders the [tinytalk] shortcode.
 */
class TinyTalk_Shortcode {

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_shortcode( 'tinytalk', array( $this, 'render_shortcode' ) );
	}

	/**
	 * Render the shortcode.
	 *
	 * @param array  $atts    Shortcode attributes.
	 * @param string $content Shortcode content.
	 * @return string HTML output.
	 */
	public function render_shortcode( $atts, $content = null ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.FoundAfterLastUsed
		$atts = shortcode_atts(
			array(
				'agent'    => '',       // Agent name.
				'agent_id' => '',       // Agent ID (takes precedence).
				'type'     => 'iframe', // Display type: 'iframe' or 'widget'.
				'width'    => '100%',
				'height'   => '660',
				'styled'   => 'yes',    // Styled iframe: 'yes' or 'no'.
			),
			$atts,
			'tinytalk'
		);

		// Determine which agent ID to use.
		$agent_id = $this->resolve_agent_id( $atts );

		if ( empty( $agent_id ) ) {
			if ( current_user_can( 'manage_options' ) ) {
				return '<p style="color: #d63638; padding: 10px; background: #fff; border: 1px solid #d63638;">'
					. esc_html__( 'Tiny Talk: No agent configured. Please add an agent in Settings → Tiny Talk.', 'tiny-talk' )
					. '</p>';
			}
			return '<!-- Tiny Talk: No agent configured -->';
		}

		// Render based on type.
		if ( 'widget' === $atts['type'] ) {
			return $this->render_widget( $agent_id );
		}

		return $this->render_iframe( $agent_id, $atts );
	}

	/**
	 * Resolve the agent ID from shortcode attributes.
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string|null Agent ID.
	 */
	private function resolve_agent_id( $atts ) {
		// Direct agent_id takes precedence.
		if ( ! empty( $atts['agent_id'] ) ) {
			return tiny_talk_is_valid_uuid( $atts['agent_id'] ) ? $atts['agent_id'] : null;
		}

		// Look up by agent name.
		if ( ! empty( $atts['agent'] ) ) {
			$agents = get_option( 'tiny_talk_agents', array() );
			foreach ( $agents as $agent ) {
				if ( strcasecmp( $agent['name'], $atts['agent'] ) === 0 ) {
					return $agent['id'];
				}
			}
			// Agent name not found, try as UUID.
			return tiny_talk_is_valid_uuid( $atts['agent'] ) ? $atts['agent'] : null;
		}

		// Use the current page's resolved agent.
		return tiny_talk()->get_current_agent_id();
	}

	/**
	 * Render widget script.
	 *
	 * @param string $agent_id Agent ID.
	 * @return string HTML.
	 */
	private function render_widget( $agent_id ) {
		tiny_talk()->set_widget_agent_id( $agent_id );

		return '<!-- Tiny Talk widget loaded via enqueue -->';
	}

	/**
	 * Render iframe embed.
	 *
	 * @param string $agent_id Agent ID.
	 * @param array  $atts     Shortcode attributes.
	 * @return string HTML.
	 */
	private function render_iframe( $agent_id, $atts ) {
		$src = sprintf(
			'%s/bots/%s/chat',
			TINY_TALK_DASHBOARD_URL,
			rawurlencode( $agent_id )
		);

		$style = 'max-width: 100%;';
		if ( 'no' !== $atts['styled'] ) {
			$style .= ' border: 1px solid #f4f4f4; border-radius: 12px; box-shadow: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);';
		} else {
			$style .= ' border: none;';
		}

		return sprintf(
			'<iframe src="%s" width="%s" height="%s" frameborder="0" style="%s"></iframe>',
			esc_url( $src ),
			esc_attr( $atts['width'] ),
			esc_attr( $atts['height'] ),
			esc_attr( $style )
		);
	}
}
