<?php
/**
 * Tiny Talk Rules Engine.
 *
 * Evaluates rules to determine which agent to show on the current page.
 *
 * @package TinyTalk
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Evaluates assignment rules against the current page context.
 */
class TinyTalk_Rules {

	/**
	 * Available condition types.
	 */
	const CONDITION_POST_TYPE     = 'post_type';
	const CONDITION_TAXONOMY      = 'taxonomy';
	const CONDITION_URL_PATTERN   = 'url_pattern';
	const CONDITION_PAGE_TEMPLATE = 'page_template';
	const CONDITION_USER_ROLE     = 'user_role';
	const CONDITION_PAGE_ID       = 'page_id';

	/**
	 * Get the agent ID for the current page based on rules.
	 *
	 * @return string|null Agent ID or null if no rule matches.
	 */
	public function get_matching_agent() {
		$rules = get_option( 'tiny_talk_rules', array() );

		if ( empty( $rules ) ) {
			return null;
		}

		foreach ( $rules as $rule ) {
			if ( $this->evaluate_rule( $rule ) ) {
				return $rule['agent_id'];
			}
		}

		return null;
	}

	/**
	 * Evaluate a single rule against the current page.
	 *
	 * @param array $rule The rule to evaluate.
	 * @return bool True if rule matches.
	 */
	private function evaluate_rule( $rule ) {
		$condition_type  = $rule['condition_type'] ?? '';
		$condition_value = $rule['condition_value'] ?? '';

		switch ( $condition_type ) {
			case self::CONDITION_POST_TYPE:
				return $this->evaluate_post_type( $condition_value );

			case self::CONDITION_TAXONOMY:
				return $this->evaluate_taxonomy( $condition_value );

			case self::CONDITION_URL_PATTERN:
				return $this->evaluate_url_pattern( $condition_value );

			case self::CONDITION_PAGE_TEMPLATE:
				return $this->evaluate_page_template( $condition_value );

			case self::CONDITION_USER_ROLE:
				return $this->evaluate_user_role( $condition_value );

			case self::CONDITION_PAGE_ID:
				return $this->evaluate_page_id( $condition_value );

			default:
				return false;
		}
	}

	/**
	 * Check if current post matches a post type.
	 *
	 * @param string $post_type Post type to check.
	 * @return bool
	 */
	private function evaluate_post_type( $post_type ) {
		if ( empty( $post_type ) ) {
			return false;
		}

		// Handle WooCommerce product.
		if ( 'product' === $post_type && function_exists( 'is_product' ) ) {
			return is_product() || is_product_category() || is_product_tag();
		}

		return get_post_type() === $post_type;
	}

	/**
	 * Check if current post has a specific taxonomy term.
	 *
	 * @param string $taxonomy_term Format: "taxonomy:term_slug".
	 * @return bool
	 */
	private function evaluate_taxonomy( $taxonomy_term ) {
		if ( empty( $taxonomy_term ) || false === strpos( $taxonomy_term, ':' ) ) {
			return false;
		}

		list($taxonomy, $term_slug) = explode( ':', $taxonomy_term, 2 );

		$post_id = get_the_ID();
		if ( ! $post_id ) {
			return false;
		}

		return has_term( $term_slug, $taxonomy, $post_id );
	}

	/**
	 * Check if current URL matches a pattern.
	 *
	 * @param string $pattern URL pattern with wildcards.
	 * @return bool
	 */
	private function evaluate_url_pattern( $pattern ) {
		if ( empty( $pattern ) ) {
			return false;
		}

		$current_path = isset( $_SERVER['REQUEST_URI'] ) ? esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';

		// Remove query string.
		$current_path = strtok( $current_path, '?' );

		// Normalize trailing slashes so /sample-page matches /sample-page/.
		$current_path = rtrim( $current_path, '/' ) . '/';
		$pattern      = rtrim( $pattern, '/' ) . '/';

		// Convert pattern to regex.
		// * matches anything except /.
		// ** matches anything including /.
		$regex = $this->pattern_to_regex( $pattern );

		return (bool) preg_match( $regex, $current_path );
	}

	/**
	 * Convert URL pattern to regex.
	 *
	 * @param string $pattern URL pattern with wildcards.
	 * @return string Regex pattern.
	 */
	private function pattern_to_regex( $pattern ) {
		// Escape regex special characters except * and /.
		$pattern = preg_quote( $pattern, '/' );

		// Convert ** to match anything.
		$pattern = str_replace( '\\*\\*', '.*', $pattern );

		// Convert * to match anything except /.
		$pattern = str_replace( '\\*', '[^/]*', $pattern );

		return '/^' . $pattern . '$/i';
	}

	/**
	 * Check if current page uses a specific template.
	 *
	 * @param string $template Template file name.
	 * @return bool
	 */
	private function evaluate_page_template( $template ) {
		if ( empty( $template ) ) {
			return false;
		}

		return is_page_template( $template );
	}

	/**
	 * Check if current user has a specific role.
	 *
	 * @param string $role User role.
	 * @return bool
	 */
	private function evaluate_user_role( $role ) {
		if ( empty( $role ) ) {
			return false;
		}

		if ( ! is_user_logged_in() ) {
			return 'guest' === $role;
		}

		$user = wp_get_current_user();
		return in_array( $role, (array) $user->roles, true );
	}

	/**
	 * Check if current page matches a specific page ID.
	 *
	 * @param string $page_ids Comma-separated page IDs.
	 * @return bool
	 */
	private function evaluate_page_id( $page_ids ) {
		if ( empty( $page_ids ) ) {
			return false;
		}

		$ids = array_map( 'trim', explode( ',', $page_ids ) );
		$ids = array_map( 'absint', $ids );

		$current_id = get_the_ID();

		return in_array( $current_id, $ids, true );
	}

	/**
	 * Get available condition types for UI.
	 *
	 * @return array Condition type definitions.
	 */
	public static function get_condition_types() {
		return array(
			array(
				'value'     => self::CONDITION_POST_TYPE,
				'label'     => __( 'Post Type is', 'tiny-talk' ),
				'has_value' => true,
			),
			array(
				'value'     => self::CONDITION_TAXONOMY,
				'label'     => __( 'Category/Term is', 'tiny-talk' ),
				'has_value' => true,
			),
			array(
				'value'       => self::CONDITION_URL_PATTERN,
				'label'       => __( 'URL matches', 'tiny-talk' ),
				'has_value'   => true,
				'placeholder' => '/shop/*',
			),
			array(
				'value'     => self::CONDITION_PAGE_TEMPLATE,
				'label'     => __( 'Page Template is', 'tiny-talk' ),
				'has_value' => true,
			),
			array(
				'value'     => self::CONDITION_USER_ROLE,
				'label'     => __( 'User Role is', 'tiny-talk' ),
				'has_value' => true,
			),
			array(
				'value'       => self::CONDITION_PAGE_ID,
				'label'       => __( 'Page ID is', 'tiny-talk' ),
				'has_value'   => true,
				'placeholder' => '123, 456',
			),
		);
	}
}
