<?php
/**
 * Tiny Talk Meta Box.
 *
 * Adds a meta box to pages/posts for per-page agent override.
 *
 * @package TinyTalk
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Provides a per-post meta box for overriding the active agent.
 */
class TinyTalk_Meta_Box {

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'add_meta_boxes', array( $this, 'add_meta_box' ) );
		add_action( 'save_post', array( $this, 'save_meta_box' ) );
	}

	/**
	 * Add meta box to post types.
	 */
	public function add_meta_box() {
		$post_types = get_post_types( array( 'public' => true ), 'names' );

		foreach ( $post_types as $post_type ) {
			add_meta_box(
				'tiny_talk_meta_box',
				__( 'Tiny Talk Agent', 'tiny-talk' ),
				array( $this, 'render_meta_box' ),
				$post_type,
				'side',
				'default'
			);
		}
	}

	/**
	 * Render meta box content.
	 *
	 * @param WP_Post $post Current post object.
	 */
	public function render_meta_box( $post ) {
		// Add nonce for security.
		wp_nonce_field( 'tiny_talk_meta_box', 'tiny_talk_meta_box_nonce' );

		// Get current value.
		$current_agent = get_post_meta( $post->ID, '_tiny_talk_agent_id', true );

		// Get available agents.
		$agents        = get_option( 'tiny_talk_agents', array() );
		$default_agent = get_option( 'tiny_talk_default_agent', '' );

		?>
		<p>
			<label for="tiny_talk_agent_id">
				<?php esc_html_e( 'Select Agent:', 'tiny-talk' ); ?>
			</label>
		</p>

		<select name="tiny_talk_agent_id" id="tiny_talk_agent_id" class="widefat">
			<option value="default"
			<?php
			selected( $current_agent, '' );
			selected( $current_agent, 'default' );
			?>
			>
				<?php
				if ( $default_agent ) {
					$default_name = '';
					foreach ( $agents as $agent ) {
						if ( $agent['id'] === $default_agent ) {
							$default_name = $agent['name'];
							break;
						}
					}
					/* translators: %s: name of the default agent */
					printf( esc_html__( 'Use Default (%s)', 'tiny-talk' ), esc_html( $default_name ? $default_name : $default_agent ) );
				} else {
					esc_html_e( 'Use Default / Rules', 'tiny-talk' );
				}
				?>
			</option>

			<option value="none" <?php selected( $current_agent, 'none' ); ?>>
				<?php esc_html_e( 'Disable on this page', 'tiny-talk' ); ?>
			</option>

			<?php if ( ! empty( $agents ) ) : ?>
				<optgroup label="<?php esc_attr_e( 'Your Agents', 'tiny-talk' ); ?>">
					<?php foreach ( $agents as $agent ) : ?>
						<option value="<?php echo esc_attr( $agent['id'] ); ?>" <?php selected( $current_agent, $agent['id'] ); ?>>
							<?php echo esc_html( $agent['name'] ); ?>
						</option>
					<?php endforeach; ?>
				</optgroup>
			<?php endif; ?>
		</select>

		<p class="description" style="margin-top: 10px;">
			<?php esc_html_e( 'Override the agent for this specific page. Select "Use Default / Rules" to let the assignment rules decide.', 'tiny-talk' ); ?>
		</p>

		<?php if ( empty( $agents ) ) : ?>
			<p class="description" style="color: #d63638;">
				<?php
				echo wp_kses(
					sprintf(
						/* translators: %s: URL to the Tiny Talk settings page */
						__( 'No agents configured. <a href="%s">Add agents in settings</a>.', 'tiny-talk' ),
						esc_url( admin_url( 'admin.php?page=tiny-talk' ) )
					),
					array( 'a' => array( 'href' => array() ) )
				);
				?>
			</p>
		<?php endif; ?>
		<?php
	}

	/**
	 * Save meta box data.
	 *
	 * @param int $post_id Post ID.
	 */
	public function save_meta_box( $post_id ) {
		// Check nonce.
		if ( ! isset( $_POST['tiny_talk_meta_box_nonce'] ) ) {
			return;
		}

		if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['tiny_talk_meta_box_nonce'] ) ), 'tiny_talk_meta_box' ) ) {
			return;
		}

		// Check autosave.
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}

		// Check permissions.
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		// Save or delete the meta.
		if ( isset( $_POST['tiny_talk_agent_id'] ) ) {
			$agent_id = sanitize_text_field( wp_unslash( $_POST['tiny_talk_agent_id'] ) );

			if ( 'default' === $agent_id || empty( $agent_id ) ) {
				delete_post_meta( $post_id, '_tiny_talk_agent_id' );
			} elseif ( 'none' === $agent_id || tiny_talk_is_valid_uuid( $agent_id ) ) {
				update_post_meta( $post_id, '_tiny_talk_agent_id', $agent_id );
			}
		}
	}
}
