<?php
/**
 * Tiny Talk Admin Settings.
 *
 * @package TinyTalk
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles admin settings page, AJAX saves, and script enqueuing.
 */
class TinyTalk_Admin {

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_action( 'wp_ajax_tiny_talk_save_agents', array( $this, 'ajax_save_agents' ) );
		add_action( 'wp_ajax_tiny_talk_save_rules', array( $this, 'ajax_save_rules' ) );
	}

	/**
	 * Add admin menu.
	 */
	public function add_admin_menu() {
		add_menu_page(
			__( 'Tiny Talk Settings', 'tiny-talk' ),
			__( 'Tiny Talk', 'tiny-talk' ),
			'manage_options',
			'tiny-talk',
			array( $this, 'render_settings_page' ),
			'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIGZpbGw9Im5vbmUiIHZpZXdCb3g9IjAgMCAyMCAyMCI+PHBhdGggZmlsbD0iY3VycmVudENvbG9yIiBmaWxsLXJ1bGU9ImV2ZW5vZGQiIGQ9Ik0yLjQ1OCAzLjMxNEMxLjUgNC41NjMgMS41IDYuMzc1IDEuNSAxMHMwIDUuNDM3Ljk1OCA2LjY4NnEuMzcyLjQ4NS44NTYuODU2YzEuMjQ5Ljk1OCAzLjA2MS45NTggNi42ODYuOTU4czUuNDM3IDAgNi42ODYtLjk1OHEuNDg1LS4zNzIuODU2LS44NTZjLjk1OC0xLjI0OC45NTgtMy4wNjEuOTU4LTYuNjg2czAtNS40MzctLjk1OC02LjY4NmE0LjYgNC42IDAgMCAwLS44NTYtLjg1NkMxNS40MzcgMS41IDEzLjYyNSAxLjUgMTAgMS41cy01LjQzNyAwLTYuNjg2Ljk1OHEtLjQ4NS4zNzItLjg1Ni44NTZtMTAuNjM4IDcuODY3YS42Ni42NiAwIDAgMSAuODc2LS4zMDRjLjMyMy4xNjEuNDY2LjU1NS4yODcuODU5YTQuNzQgNC43NCAwIDAgMS00LjI3NyAyLjY2NiA0LjcgNC43IDAgMCAxLTEuNTc0LS4yNjkuNjU3LjY1NyAwIDAgMSAuNDQ3LTEuMjM0Yy4zNzUuMTI2Ljc1MS4xOTcgMS4xNDUuMTk3YTMuNDQgMy40NCAwIDAgMCAzLjA5Ni0xLjkxNU04LjM1NCA3LjUxM2ExLjI4OCAxLjI4OCAwIDEgMS0yLjU3NiAwIDEuMjg4IDEuMjg4IDAgMCAxIDIuNTc2IDBtNC41OCAxLjI4OGExLjI4OSAxLjI4OSAwIDEgMCAwLTIuNTc3IDEuMjg5IDEuMjg5IDAgMCAwIDAgMi41NzciIGNsaXAtcnVsZT0iZXZlbm9kZCIvPjwvc3ZnPg==',
			80
		);
	}

	/**
	 * Register settings.
	 */
	public function register_settings() {
		register_setting(
			'tiny_talk_options',
			'tiny_talk_agents',
			array(
				'type'              => 'array',
				'sanitize_callback' => array( $this, 'sanitize_agents' ),
				'default'           => array(),
			)
		);

		register_setting(
			'tiny_talk_options',
			'tiny_talk_rules',
			array(
				'type'              => 'array',
				'sanitize_callback' => array( $this, 'sanitize_rules' ),
				'default'           => array(),
			)
		);

		register_setting(
			'tiny_talk_options',
			'tiny_talk_default_agent',
			array(
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
				'default'           => '',
			)
		);

		register_setting(
			'tiny_talk_options',
			'tiny_talk_enabled',
			array(
				'type'              => 'boolean',
				'sanitize_callback' => 'wp_validate_boolean',
				'default'           => true,
			)
		);
	}

	/**
	 * Sanitize agents array.
	 *
	 * @param mixed $input Raw input to sanitize.
	 * @return array Sanitized agents.
	 */
	public function sanitize_agents( $input ) {
		if ( ! is_array( $input ) ) {
			return array();
		}

		$sanitized = array();
		foreach ( $input as $agent ) {
			if ( empty( $agent['id'] ) || empty( $agent['name'] ) ) {
				continue;
			}

			$id   = sanitize_text_field( $agent['id'] );
			$name = sanitize_text_field( $agent['name'] );

			if ( ! empty( $name ) && tiny_talk_is_valid_uuid( $id ) ) {
				$sanitized[] = array(
					'id'   => $id,
					'name' => $name,
				);
			}
		}
		return $sanitized;
	}

	/**
	 * Sanitize rules array.
	 *
	 * @param mixed $input Raw input to sanitize.
	 * @return array Sanitized rules.
	 */
	public function sanitize_rules( $input ) {
		if ( ! is_array( $input ) ) {
			return array();
		}

		$valid_condition_types = array(
			'post_type',
			'taxonomy',
			'url_pattern',
			'page_template',
			'user_role',
			'page_id',
		);

		$sanitized = array();
		foreach ( $input as $rule ) {
			if ( ! empty( $rule['condition_type'] ) && ! empty( $rule['agent_id'] ) ) {
				$condition_type = sanitize_text_field( $rule['condition_type'] );
				if ( ! in_array( $condition_type, $valid_condition_types, true ) ) {
					continue;
				}
				$sanitized[] = array(
					'condition_type'  => $condition_type,
					'condition_value' => sanitize_text_field( $rule['condition_value'] ?? '' ),
					'agent_id'        => sanitize_text_field( $rule['agent_id'] ),
				);
			}
		}
		return $sanitized;
	}

	/**
	 * Enqueue admin scripts and styles.
	 *
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_scripts( $hook ) {
		if ( 'toplevel_page_tiny-talk' !== $hook ) {
			return;
		}

		wp_enqueue_style(
			'tiny-talk-admin',
			TINY_TALK_PLUGIN_URL . 'admin/css/admin-style.css',
			array(),
			TINY_TALK_VERSION
		);

		wp_enqueue_script(
			'tiny-talk-admin',
			TINY_TALK_PLUGIN_URL . 'admin/js/admin-script.js',
			array( 'jquery', 'jquery-ui-sortable' ),
			TINY_TALK_VERSION,
			true
		);

		// Pass data to JS.
		wp_localize_script(
			'tiny-talk-admin',
			'tinyTalkAdmin',
			array(
				'ajaxUrl'      => admin_url( 'admin-ajax.php' ),
				'nonce'        => wp_create_nonce( 'tiny_talk_admin' ),
				'agents'       => get_option( 'tiny_talk_agents', array() ),
				'rules'        => get_option( 'tiny_talk_rules', array() ),
				'defaultAgent' => get_option( 'tiny_talk_default_agent', '' ),
				'postTypes'    => $this->get_post_types(),
				'taxonomies'   => $this->get_taxonomies(),
				'i18n'         => array(
					'confirmDelete'         => __( 'Are you sure you want to delete this?', 'tiny-talk' ),
					'saved'                 => __( 'Settings saved successfully.', 'tiny-talk' ),
					'error'                 => __( 'An error occurred. Please try again.', 'tiny-talk' ),
					'agentNamePlaceholder'  => __( 'e.g., Sales Agent', 'tiny-talk' ),
					'agentIdPlaceholder'    => __( 'e.g., 550e8400-e29b-41d4-a716-446655440000', 'tiny-talk' ),
					'validationAgentFields' => __( 'Please fill in both Agent Name and Agent ID for all rows.', 'tiny-talk' ),
					'validationAgentId'     => __( 'Agent ID must be a valid UUID (e.g., 550e8400-e29b-41d4-a716-446655440000).', 'tiny-talk' ),
					'addAgentFirst'         => __( 'Please add at least one agent before creating rules.', 'tiny-talk' ),
					'agentUsedInRules'      => __( 'This agent is used in assignment rules. Please remove or update those rules first.', 'tiny-talk' ),
					'emptyAgents'           => __( 'Click "Add Agent" to get started.', 'tiny-talk' ),
					'emptyRules'            => __( 'No rules defined. The default agent will be used on all pages.', 'tiny-talk' ),
					'saving'                => __( 'Saving...', 'tiny-talk' ),
					'saveAgents'            => __( 'Save Agents', 'tiny-talk' ),
					'saveRules'             => __( 'Save Rules', 'tiny-talk' ),
					'select'                => __( '-- Select --', 'tiny-talk' ),
					'selectAgent'           => __( '-- Select Agent --', 'tiny-talk' ),
					'dragToReorder'         => __( 'Drag to reorder', 'tiny-talk' ),
					'conditionPostType'     => __( 'Post Type is', 'tiny-talk' ),
					'conditionTaxonomy'     => __( 'Category/Term is', 'tiny-talk' ),
					'conditionUrlPattern'   => __( 'URL matches', 'tiny-talk' ),
					'conditionPageTemplate' => __( 'Page Template is', 'tiny-talk' ),
					'conditionUserRole'     => __( 'User Role is', 'tiny-talk' ),
					'conditionPageId'       => __( 'Page ID is', 'tiny-talk' ),
					'roleGuest'             => __( 'Guest (Not logged in)', 'tiny-talk' ),
					'roleSubscriber'        => __( 'Subscriber', 'tiny-talk' ),
					'roleCustomer'          => __( 'Customer', 'tiny-talk' ),
					'roleContributor'       => __( 'Contributor', 'tiny-talk' ),
					'roleAuthor'            => __( 'Author', 'tiny-talk' ),
					'roleEditor'            => __( 'Editor', 'tiny-talk' ),
					'roleAdministrator'     => __( 'Administrator', 'tiny-talk' ),
					'placeholderUrl'        => '/shop/*',
					'placeholderTemplate'   => 'template-landing.php',
					'placeholderPageId'     => '123, 456',
				),
			)
		);
	}

	/**
	 * Get available post types.
	 */
	private function get_post_types() {
		$post_types = get_post_types( array( 'public' => true ), 'objects' );
		$result     = array();

		foreach ( $post_types as $post_type ) {
			$result[] = array(
				'value' => $post_type->name,
				'label' => $post_type->labels->singular_name,
			);
		}

		return $result;
	}

	/**
	 * Get available taxonomies with terms.
	 */
	private function get_taxonomies() {
		$taxonomies = get_taxonomies( array( 'public' => true ), 'objects' );
		$result     = array();

		foreach ( $taxonomies as $taxonomy ) {
			$terms = get_terms(
				array(
					'taxonomy'   => $taxonomy->name,
					'hide_empty' => false,
					'number'     => 100,
				)
			);

			if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
				$term_options = array();
				foreach ( $terms as $term ) {
					$term_options[] = array(
						'value' => $taxonomy->name . ':' . $term->slug,
						'label' => $term->name,
					);
				}

				$result[] = array(
					'name'  => $taxonomy->name,
					'label' => $taxonomy->labels->singular_name,
					'terms' => $term_options,
				);
			}
		}

		return $result;
	}

	/**
	 * AJAX handler for saving agents.
	 */
	public function ajax_save_agents() {
		check_ajax_referer( 'tiny_talk_admin', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'tiny-talk' ) ) );
		}

		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- JSON string decoded then sanitized via sanitize_agents().
		$agents        = isset( $_POST['agents'] ) ? json_decode( wp_unslash( $_POST['agents'] ), true ) : array();
		$default_agent = isset( $_POST['default_agent'] ) ? sanitize_text_field( wp_unslash( $_POST['default_agent'] ) ) : '';
		if ( '' !== $default_agent && ! tiny_talk_is_valid_uuid( $default_agent ) ) {
			$default_agent = '';
		}

		// Reject if any agent name contains HTML tags.
		if ( is_array( $agents ) ) {
			foreach ( $agents as $agent ) {
				$name = $agent['name'] ?? '';
				if ( wp_strip_all_tags( $name ) !== $name ) {
					wp_send_json_error( array( 'message' => __( 'Agent names cannot contain HTML.', 'tiny-talk' ) ) );
				}
			}
		}

		$sanitized_agents = $this->sanitize_agents( $agents );
		update_option( 'tiny_talk_agents', $sanitized_agents );
		update_option( 'tiny_talk_default_agent', $default_agent );

		wp_send_json_success(
			array(
				'message'      => __( 'Agents saved successfully.', 'tiny-talk' ),
				'agents'       => $sanitized_agents,
				'defaultAgent' => $default_agent,
			)
		);
	}

	/**
	 * AJAX handler for saving rules.
	 */
	public function ajax_save_rules() {
		check_ajax_referer( 'tiny_talk_admin', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'tiny-talk' ) ) );
		}

		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- JSON string decoded then sanitized via sanitize_rules().
		$rules = isset( $_POST['rules'] ) ? json_decode( wp_unslash( $_POST['rules'] ), true ) : array();

		$sanitized_rules = $this->sanitize_rules( $rules );
		update_option( 'tiny_talk_rules', $sanitized_rules );

		wp_send_json_success(
			array(
				'message' => __( 'Rules saved successfully.', 'tiny-talk' ),
				'rules'   => $sanitized_rules,
			)
		);
	}

	/**
	 * Render settings page.
	 */
	public function render_settings_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Handle form submission for enable/disable.
		if ( isset( $_POST['tiny_talk_save_general'] ) && check_admin_referer( 'tiny_talk_general' ) ) {
			update_option( 'tiny_talk_enabled', isset( $_POST['tiny_talk_enabled'] ) );
			echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Settings saved.', 'tiny-talk' ) . '</p></div>';
		}

		$enabled = get_option( 'tiny_talk_enabled', true );
		?>
		<div class="wrap tiny-talk-admin">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>

			<div class="tiny-talk-header">
				<p>
				<?php
					echo wp_kses(
						sprintf(
							/* translators: %1$s: opening link tag for Tiny Talk website, %2$s: closing link tag, %3$s: opening link tag for dashboard, %4$s: closing link tag */
							__( 'Configure your %1$sTiny Talk AI%2$s agent plugin settings. Add multiple agents and create rules to show different agents on different pages. For more configuration and settings visit your %3$sTiny Talk dashboard%4$s.', 'tiny-talk' ),
							'<a href="' . esc_url( TINY_TALK_WEBSITE_URL ) . '" target="_blank">',
							'</a>',
							'<a href="' . esc_url( TINY_TALK_DASHBOARD_URL ) . '" target="_blank">',
							'</a>'
						),
						array(
							'a' => array(
								'href'   => array(),
								'target' => array(),
							),
						)
					);
				?>
				</p>
			</div>

			<!-- General Settings -->
			<div class="tiny-talk-section">
				<h2><?php esc_html_e( 'General Settings', 'tiny-talk' ); ?></h2>
				<form method="post">
					<?php wp_nonce_field( 'tiny_talk_general' ); ?>
					<table class="form-table">
						<tr>
							<th scope="row"><?php esc_html_e( 'Enable Agent', 'tiny-talk' ); ?></th>
							<td>
								<label>
									<input type="checkbox" name="tiny_talk_enabled" value="1" <?php checked( $enabled ); ?> />
									<?php esc_html_e( 'Show agent messenger on your site', 'tiny-talk' ); ?>
								</label>
							</td>
						</tr>
					</table>
					<p class="submit">
						<input type="submit" name="tiny_talk_save_general" class="button button-secondary" value="<?php esc_attr_e( 'Save General Settings', 'tiny-talk' ); ?>" />
					</p>
				</form>
			</div>

			<!-- Agents Section -->
			<div class="tiny-talk-section">
				<h2><?php esc_html_e( 'Your Agents', 'tiny-talk' ); ?></h2>
				<p class="description">
				<?php
					echo wp_kses(
						sprintf(
							/* translators: %1$s: opening link tag for dashboard, %2$s: closing link tag */
							__( 'Add your Tiny Talk agents. Find your Agent ID in the %1$sTiny Talk dashboard%2$s under Settings → General.', 'tiny-talk' ),
							'<a href="' . esc_url( TINY_TALK_DASHBOARD_URL ) . '" target="_blank">',
							'</a>'
						),
						array(
							'a' => array(
								'href'   => array(),
								'target' => array(),
							),
						)
					);
				?>
				</p>

				<div id="tiny-talk-agents-container">
					<table class="wp-list-table widefat fixed striped" id="tiny-talk-agents-table">
						<thead>
							<tr>
								<th class="column-name"><?php esc_html_e( 'Agent Name', 'tiny-talk' ); ?></th>
								<th class="column-id"><?php esc_html_e( 'Agent ID', 'tiny-talk' ); ?></th>
								<th class="column-default"><?php esc_html_e( 'Default', 'tiny-talk' ); ?></th>
								<th class="column-actions"><?php esc_html_e( 'Actions', 'tiny-talk' ); ?></th>
							</tr>
						</thead>
						<tbody id="tiny-talk-agents-list">
							<!-- Populated by JavaScript -->
						</tbody>
					</table>

					<p>
						<button type="button" class="button" id="tiny-talk-add-agent">
							<?php esc_html_e( '+ Add Agent', 'tiny-talk' ); ?>
						</button>
						<button type="button" class="button button-primary" id="tiny-talk-save-agents">
							<?php esc_html_e( 'Save Agents', 'tiny-talk' ); ?>
						</button>
					</p>
				</div>
			</div>

			<!-- Rules Section -->
			<div class="tiny-talk-section">
				<h2><?php esc_html_e( 'Assignment Rules', 'tiny-talk' ); ?></h2>
				<p class="description"><?php esc_html_e( 'Create rules to show different agents on different pages. Rules are evaluated from top to bottom - the first matching rule wins.', 'tiny-talk' ); ?></p>

				<div id="tiny-talk-rules-container">
					<table class="wp-list-table widefat fixed striped" id="tiny-talk-rules-table">
						<thead>
							<tr>
								<th class="column-order">#</th>
								<th class="column-condition"><?php esc_html_e( 'Condition', 'tiny-talk' ); ?></th>
								<th class="column-agent"><?php esc_html_e( 'Show Agent', 'tiny-talk' ); ?></th>
								<th class="column-actions"><?php esc_html_e( 'Actions', 'tiny-talk' ); ?></th>
							</tr>
						</thead>
						<tbody id="tiny-talk-rules-list">
							<!-- Populated by JavaScript -->
						</tbody>
					</table>

					<p>
						<button type="button" class="button" id="tiny-talk-add-rule">
							<?php esc_html_e( '+ Add Rule', 'tiny-talk' ); ?>
						</button>
						<button type="button" class="button button-primary" id="tiny-talk-save-rules">
							<?php esc_html_e( 'Save Rules', 'tiny-talk' ); ?>
						</button>
					</p>
				</div>
			</div>

			<!-- Usage Section -->
			<div class="tiny-talk-section">
				<h2><?php esc_html_e( 'Usage', 'tiny-talk' ); ?></h2>

				<h3><?php esc_html_e( 'Shortcodes', 'tiny-talk' ); ?></h3>
				<p><?php esc_html_e( 'Embed the messenger interface using shortcodes:', 'tiny-talk' ); ?></p>
				<code>[tinytalk]</code> - <?php esc_html_e( 'Uses default or rule-matched agent', 'tiny-talk' ); ?><br><br>
				<code>[tinytalk agent="Sales Agent"]</code> - <?php esc_html_e( 'Uses specific agent by name', 'tiny-talk' ); ?><br><br>
				<code>[tinytalk agent_id="abc-123"]</code> - <?php esc_html_e( 'Uses specific agent by ID', 'tiny-talk' ); ?><br><br>
				<code>[tinytalk width="100%" height="500" type="iframe" styled="yes"]</code> - <?php esc_html_e( 'Embed as styled iframe with custom dimensions', 'tiny-talk' ); ?><br><br>
				<code>[tinytalk width="100%" height="500" type="iframe" styled="no"]</code> - <?php esc_html_e( 'Embed as plain iframe with custom dimensions', 'tiny-talk' ); ?>

				<h3><?php esc_html_e( 'Page Override', 'tiny-talk' ); ?></h3>
				<p><?php esc_html_e( 'You can override the agent for specific pages using the "Tiny Talk" meta box when editing a page or post.', 'tiny-talk' ); ?></p>
			</div>
		</div>
		<?php
	}
}
