// @ts-nocheck

/**
 * Tiny Talk Admin JavaScript
 */

(function($) {
    'use strict';

    // Shorthand for i18n strings
    const i18n = tinyTalkAdmin.i18n;

    // State
    let agents = tinyTalkAdmin.agents || [];
    let rules = tinyTalkAdmin.rules || [];
    let defaultAgent = tinyTalkAdmin.defaultAgent || '';

    // Initialize
    $(document).ready(function() {
        renderAgents();
        renderRules();
        bindEvents();
        initSortableRules();
    });

    /**
     * Initialize sortable on rules table
     */
    function initSortableRules() {
        $('#tiny-talk-rules-list').sortable({
            handle: '.column-order',
            placeholder: 'sortable-placeholder',
            update: function() {
                // Update data-index after reorder
                $('#tiny-talk-rules-list tr').each(function(i) {
                    $(this).attr('data-index', i);
                });
            }
        });
    }

    /**
     * Bind event handlers
     */
    function bindEvents() {
        // Agents
        $('#tiny-talk-add-agent').on('click', addAgent);
        $('#tiny-talk-save-agents').on('click', saveAgents);
        $(document).on('click', '.tiny-talk-delete-agent', deleteAgent);
        $(document).on('change', '.agent-default-radio', setDefaultAgent);

        // Rules
        $('#tiny-talk-add-rule').on('click', addRule);
        $('#tiny-talk-save-rules').on('click', saveRules);
        $(document).on('click', '.tiny-talk-delete-rule', deleteRule);
        $(document).on('change', '.rule-condition-type', updateConditionValue);
    }

    /**
     * Render agents table
     */
    function renderAgents() {
        const $tbody = $('#tiny-talk-agents-list');
        $tbody.empty();

        if (agents.length === 0) {
            $tbody.append(
                '<tr class="tiny-talk-empty-row">' +
                '<td colspan="4">' + escapeHtml(i18n.emptyAgents) + '</td>' +
                '</tr>'
            );
            return;
        }

        agents.forEach(function(agent, index) {
            const isDefault = agent.id === defaultAgent;
            const row = `
                <tr data-index="${index}">
                    <td class="column-name">
                        <input type="text" class="agent-name" value="${escapeHtml(agent.name)}"
                               placeholder="${escapeHtml(i18n.agentNamePlaceholder)}" />
                    </td>
                    <td class="column-id">
                        <input type="text" class="agent-id" value="${escapeHtml(agent.id)}"
                               placeholder="${escapeHtml(i18n.agentIdPlaceholder)}" />
                    </td>
                    <td class="column-default">
                        <input type="radio" name="default_agent" class="agent-default-radio"
                               value="${escapeHtml(agent.id)}" ${isDefault ? 'checked' : ''} />
                    </td>
                    <td class="column-actions">
                        <span class="tiny-talk-delete-agent dashicons dashicons-trash" title="${escapeHtml(i18n.confirmDelete)}"></span>
                    </td>
                </tr>
            `;
            $tbody.append(row);
        });
    }

    /**
     * Render rules table
     */
    function renderRules() {
        const $tbody = $('#tiny-talk-rules-list');
        $tbody.empty();

        if (rules.length === 0) {
            $tbody.append(
                '<tr class="tiny-talk-empty-row">' +
                '<td colspan="4">' + escapeHtml(i18n.emptyRules) + '</td>' +
                '</tr>'
            );
            return;
        }

        rules.forEach(function(rule, index) {
            const row = `
                <tr data-index="${index}">
                    <td class="column-order" title="${escapeHtml(i18n.dragToReorder)}"><span class="dashicons dashicons-menu"></span></td>
                    <td class="column-condition">
                        <div class="rule-condition-builder">
                            ${buildConditionTypeSelect(rule.condition_type)}
                            ${buildConditionValueInput(rule.condition_type, rule.condition_value)}
                        </div>
                    </td>
                    <td class="column-agent">
                        ${buildAgentSelect(rule.agent_id)}
                    </td>
                    <td class="column-actions">
                        <span class="tiny-talk-delete-rule dashicons dashicons-trash" title="${escapeHtml(i18n.confirmDelete)}"></span>
                    </td>
                </tr>
            `;
            $tbody.append(row);
        });

        initSortableRules();
    }

    /**
     * Build condition type select
     */
    function buildConditionTypeSelect(selectedType) {
        const types = [
            { value: 'post_type', label: i18n.conditionPostType },
            { value: 'taxonomy', label: i18n.conditionTaxonomy },
            { value: 'url_pattern', label: i18n.conditionUrlPattern },
            { value: 'page_template', label: i18n.conditionPageTemplate },
            { value: 'user_role', label: i18n.conditionUserRole },
            { value: 'page_id', label: i18n.conditionPageId }
        ];

        let html = '<select class="rule-condition-type">';
        types.forEach(function(type) {
            const selected = type.value === selectedType ? 'selected' : '';
            html += `<option value="${type.value}" ${selected}>${escapeHtml(type.label)}</option>`;
        });
        html += '</select>';

        return html;
    }

    /**
     * Build condition value input based on type
     */
    function buildConditionValueInput(conditionType, value) {
        value = value || '';

        switch (conditionType) {
            case 'post_type':
                return buildPostTypeSelect(value);

            case 'taxonomy':
                return buildTaxonomySelect(value);

            case 'user_role':
                return buildUserRoleSelect(value);

            case 'url_pattern':
                return `<input type="text" class="condition-value" value="${escapeHtml(value)}" placeholder="${escapeHtml(i18n.placeholderUrl)}" />`;

            case 'page_template':
                return `<input type="text" class="condition-value" value="${escapeHtml(value)}" placeholder="${escapeHtml(i18n.placeholderTemplate)}" />`;

            case 'page_id':
                return `<input type="text" class="condition-value" value="${escapeHtml(value)}" placeholder="${escapeHtml(i18n.placeholderPageId)}" />`;

            default:
                return `<input type="text" class="condition-value" value="${escapeHtml(value)}" />`;
        }
    }

    /**
     * Build post type select
     */
    function buildPostTypeSelect(selectedValue) {
        let html = '<select class="condition-value">';
        html += `<option value="">${escapeHtml(i18n.select)}</option>`;

        tinyTalkAdmin.postTypes.forEach(function(pt) {
            const selected = pt.value === selectedValue ? 'selected' : '';
            html += `<option value="${escapeHtml(pt.value)}" ${selected}>${escapeHtml(pt.label)}</option>`;
        });

        html += '</select>';
        return html;
    }

    /**
     * Build taxonomy/term select
     */
    function buildTaxonomySelect(selectedValue) {
        let html = '<select class="condition-value">';
        html += `<option value="">${escapeHtml(i18n.select)}</option>`;

        tinyTalkAdmin.taxonomies.forEach(function(tax) {
            html += `<optgroup label="${escapeHtml(tax.label)}">`;
            tax.terms.forEach(function(term) {
                const selected = term.value === selectedValue ? 'selected' : '';
                html += `<option value="${escapeHtml(term.value)}" ${selected}>${escapeHtml(term.label)}</option>`;
            });
            html += '</optgroup>';
        });

        html += '</select>';
        return html;
    }

    /**
     * Build user role select
     */
    function buildUserRoleSelect(selectedValue) {
        const roles = [
            { value: 'guest', label: i18n.roleGuest },
            { value: 'subscriber', label: i18n.roleSubscriber },
            { value: 'customer', label: i18n.roleCustomer },
            { value: 'contributor', label: i18n.roleContributor },
            { value: 'author', label: i18n.roleAuthor },
            { value: 'editor', label: i18n.roleEditor },
            { value: 'administrator', label: i18n.roleAdministrator }
        ];

        let html = '<select class="condition-value">';
        html += `<option value="">${escapeHtml(i18n.select)}</option>`;

        roles.forEach(function(role) {
            const selected = role.value === selectedValue ? 'selected' : '';
            html += `<option value="${role.value}" ${selected}>${escapeHtml(role.label)}</option>`;
        });

        html += '</select>';
        return html;
    }

    /**
     * Build agent select dropdown
     */
    function buildAgentSelect(selectedAgentId) {
        let html = '<select class="rule-agent-select">';
        html += `<option value="">${escapeHtml(i18n.selectAgent)}</option>`;

        agents.forEach(function(agent) {
            const selected = agent.id === selectedAgentId ? 'selected' : '';
            html += `<option value="${escapeHtml(agent.id)}" ${selected}>${escapeHtml(agent.name)}</option>`;
        });

        html += '</select>';
        return html;
    }

    /**
     * Add a new agent
     */
    function addAgent() {
        agents.push({
            id: '',
            name: ''
        });
        renderAgents();

        // Focus the new row
        $('#tiny-talk-agents-list tr:last-child .agent-name').focus();
    }

    /**
     * Delete an agent
     */
    function deleteAgent(e) {
        if (!confirm(i18n.confirmDelete)) {
            return;
        }

        const $row = $(e.target).closest('tr');
        const index = $row.data('index');

        // Check if this agent is used in rules
        const agentId = agents[index].id;
        const usedInRules = rules.some(r => r.agent_id === agentId);

        if (usedInRules) {
            alert(i18n.agentUsedInRules);
            return;
        }

        agents.splice(index, 1);

        // Update default if needed
        if (defaultAgent === agentId) {
            defaultAgent = agents.length > 0 ? agents[0].id : '';
        }

        renderAgents();
    }

    /**
     * Set default agent
     */
    function setDefaultAgent(e) {
        defaultAgent = $(e.target).val();
    }

    /**
     * Save agents via AJAX
     */
    function saveAgents() {
        // Validate and collect data from form
        let hasErrors = false;
        agents = [];

        $('#tiny-talk-agents-list tr:not(.tiny-talk-empty-row)').each(function() {
            const $row = $(this);
            const $name = $row.find('.agent-name');
            const $id = $row.find('.agent-id');
            const name = $name.val().trim();
            const id = $id.val().trim();

            const isValidUuid = /^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i.test(id);

            $name.css('border-color', name ? '' : '#d63638');
            $id.css('border-color', (id && isValidUuid) ? '' : '#d63638');

            if (!name || !id || !isValidUuid) {
                hasErrors = true;
            } else {
                agents.push({ name: name, id: id });
            }
        });

        if (hasErrors) {
            showNotice(i18n.validationAgentId, 'error');
            return;
        }

        // Get selected default from the row's actual agent ID input (not the stale radio value)
        var $checkedRadio = $('input[name="default_agent"]:checked');
        if ($checkedRadio.length) {
            defaultAgent = $checkedRadio.closest('tr').find('.agent-id').val().trim();
        } else if (agents.length > 0) {
            defaultAgent = agents[0].id;
        } else {
            defaultAgent = '';
        }

        // AJAX save
        const $btn = $('#tiny-talk-save-agents');
        $btn.prop('disabled', true).text(i18n.saving);

        $.ajax({
            url: tinyTalkAdmin.ajaxUrl,
            method: 'POST',
            data: {
                action: 'tiny_talk_save_agents',
                nonce: tinyTalkAdmin.nonce,
                agents: JSON.stringify(agents),
                default_agent: defaultAgent
            },
            success: function(response) {
                if (response.success) {
                    // Resync local state with what the server actually saved
                    agents = response.data.agents || [];
                    defaultAgent = response.data.defaultAgent || '';
                    renderAgents();
                    renderRules();
                    showNotice(i18n.saved, 'success');
                } else {
                    showNotice(response.data.message || i18n.error, 'error');
                }
            },
            error: function() {
                showNotice(i18n.error, 'error');
            },
            complete: function() {
                $btn.prop('disabled', false).text(i18n.saveAgents);
            }
        });
    }

    /**
     * Add a new rule
     */
    function addRule() {
        if (agents.length === 0) {
            alert(i18n.addAgentFirst);
            return;
        }

        rules.push({
            condition_type: 'post_type',
            condition_value: '',
            agent_id: ''
        });
        renderRules();
    }

    /**
     * Delete a rule
     */
    function deleteRule(e) {
        if (!confirm(i18n.confirmDelete)) {
            return;
        }

        const $row = $(e.target).closest('tr');
        const index = $row.data('index');

        rules.splice(index, 1);
        renderRules();
    }

    /**
     * Update condition value input when type changes
     */
    function updateConditionValue(e) {
        const $select = $(e.target);
        const $row = $select.closest('tr');
        const newType = $select.val();
        const index = $row.data('index');

        // Update the rule
        rules[index].condition_type = newType;
        rules[index].condition_value = '';

        // Re-render just this row's condition value
        const $builder = $row.find('.rule-condition-builder');
        $builder.find('.condition-value').remove();
        $builder.append(buildConditionValueInput(newType, ''));
    }

    /**
     * Save rules via AJAX
     */
    function saveRules() {
        // Collect data from form
        rules = [];
        $('#tiny-talk-rules-list tr:not(.tiny-talk-empty-row)').each(function() {
            const $row = $(this);
            const conditionType = $row.find('.rule-condition-type').val();
            const conditionValue = $row.find('.condition-value').val();
            const agentId = $row.find('.rule-agent-select').val();

            if (conditionType && agentId) {
                rules.push({
                    condition_type: conditionType,
                    condition_value: conditionValue || '',
                    agent_id: agentId
                });
            }
        });

        // AJAX save
        const $btn = $('#tiny-talk-save-rules');
        $btn.prop('disabled', true).text(i18n.saving);

        $.ajax({
            url: tinyTalkAdmin.ajaxUrl,
            method: 'POST',
            data: {
                action: 'tiny_talk_save_rules',
                nonce: tinyTalkAdmin.nonce,
                rules: JSON.stringify(rules)
            },
            success: function(response) {
                if (response.success) {
                    // Resync local state with what the server actually saved
                    rules = response.data.rules || [];
                    renderRules();
                    showNotice(i18n.saved, 'success');
                } else {
                    showNotice(response.data.message || i18n.error, 'error');
                }
            },
            error: function() {
                showNotice(i18n.error, 'error');
            },
            complete: function() {
                $btn.prop('disabled', false).text(i18n.saveRules);
            }
        });
    }

    /**
     * Show admin notice
     */
    function showNotice(message, type) {
        const $notice = $('<div class="notice is-dismissible"></div>')
            .addClass(type === 'error' ? 'notice-error' : 'notice-success')
            .append($('<p>').text(message));

        $('.tiny-talk-admin h1').after($notice);

        // Auto dismiss after 3 seconds
        setTimeout(function() {
            $notice.fadeOut(function() {
                $(this).remove();
            });
        }, 3000);
    }

    /**
     * Escape HTML
     */
    function escapeHtml(text) {
        if (!text) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

})(jQuery);
