<?php

namespace WPMinimize\TimeFix\Traits;

defined( 'ABSPATH' ) || exit;

trait AppointmentsTrait {
	public static function get_appointment_statuses() {
		return apply_filters( 'wpm_timefix_get_appointment_statuses', [
			'pending'   => esc_html__( 'Pending', 'timefix' ),
			'confirmed' => esc_html__( 'Confirmed', 'timefix' ),
			'cancelled' => esc_html__( 'Cancelled', 'timefix' ),
			'completed' => esc_html__( 'Completed', 'timefix' ),
			'on_hold'   => esc_html__( 'On Hold', 'timefix' )
		] );
	}

	public static function get_appointment_status_title( $status = 'pending' ) {
		$statuses = self::get_appointment_statuses();

		return $statuses[ $status ];
	}

	public static function get_appointment_by_id( $appointment_id ) {
		global $wpdb;

		$appointment_id = absint( $appointment_id );

		$appointments_table         = $wpdb->prefix . "wpm_timefix_appointments";
		$appointment_services_table = $wpdb->prefix . "wpm_timefix_appointment_services";
		$customers_table            = $wpdb->prefix . "wpm_timefix_customers";
		$orders_table               = $wpdb->prefix . "wpm_timefix_orders";

		$query = $wpdb->prepare( "SELECT 
            a.id AS appointment_id,
            a.status AS appointment_status,
            a.created_at AS appointment_created_at,
            a.additional_notes,
            s.appointment_date,
            s.start_time,
            s.end_time,
            s.service_id,
            s.staff_id,
            c.id AS customer_id,
            c.name,
            c.email,
            c.phone,
            o.payment_method,
            o.total AS order_total FROM {$appointments_table} a
                JOIN {$customers_table} c ON a.customer_id = c.id
                JOIN {$orders_table} o ON o.object_id = a.id
                JOIN {$appointment_services_table} s ON s.appointment_id = a.id
                WHERE a.id=%d", $appointment_id );

		return $wpdb->get_row( $query );
	}

	public static function set_appointment_status_for_wc_order( $status ) {
		$new_status = '';
		switch ( $status ) {
			case 'refunded':
			case 'cancelled':
				$new_status = 'cancelled';
				break;
			case 'completed':
				$new_status = 'confirmed';
				break;
			case 'checkout-draft':
			case 'failed':
			case 'pending':
				$new_status = 'pending';
				break;
			case 'processing':
				$new_status = 'completed';
				break;
			case 'on-hold':
				$new_status = 'on_hold';
				break;
			default:
				$new_status = 'pending';
		}

		return $new_status;
	}

	/**
	 * @param $appointment_id
	 * @param $status
	 *
	 * @return bool|int|\mysqli_result|null
	 */
	public static function update_appointment_status( $appointment_id, $status ) {
		global $wpdb;

		$appointments_table = $wpdb->prefix . "wpm_timefix_appointments";

		$data = [
			'status'     => self::set_appointment_status_for_wc_order( $status ),
			'updated_at' => current_time( 'mysql' )
		];

		$where = [
			'id' => $appointment_id
		];

		return $wpdb->update( $appointments_table, $data, $where );
	}

	/**
	 * @param $customer
	 * @param $appointment
	 * @param $services
	 * @param $order
	 *
	 * @return false|int
	 */
	public static function insert_appointment_data( $customer = [], $appointment = [], $services = [], $order = [] ) {
		global $wpdb;

		$wpdb->query( 'START TRANSACTION' );

		try {
			$user_id        = $customer['user_id'];
			$customer_name  = $customer['name'];
			$customer_email = $customer['email'];
			$phone          = $customer['phone'];

			$customer_id = self::get_or_create_customer( $user_id, $customer_name, $customer_email, $phone );

			// Insert appointment data
			$appointments_table = $wpdb->prefix . "wpm_timefix_appointments";

			$appointment_data = [
				'customer_id'      => $customer_id,
				'status'           => 'pending',
				'additional_notes' => $appointment['note']
			];

			$wpdb->insert( $appointments_table, $appointment_data );
			$appointment_id = $wpdb->insert_id;

			// Insert appointment services data
			$appointment_services_table = $wpdb->prefix . "wpm_timefix_appointment_services";

			$wpdb->insert( $appointment_services_table, [
				'appointment_id'   => $appointment_id,
				'service_id'       => $services['service_id'],
				'staff_id'         => $services['staff_id'],
				'appointment_date' => $services['appointment_date'],
				'start_time'       => $services['start_time'],
				'end_time'         => $services['end_time'],
				'price'            => $services['price']
			] );

			// Insert appointment order
			$orders_table = $wpdb->prefix . "wpm_timefix_orders";

			$wpdb->insert( $orders_table, [
				'object_id'             => $appointment_id,
				'object_type'           => 'appointment',
				'customer_id'           => $customer_id,
				'woo_order_id'          => $order['wc_order_id'] ?? '',
				'transaction_reference' => $order['key'] ?? '',
				'subtotal'              => $order['subtotal'],
				'total'                 => $order['total'],
				'payment_method'        => $order['payment_method'],
				'payment_status'        => $order['status']
			] );

			$wpdb->query( 'COMMIT' );

			do_action( 'wpm_timefix_appointment_submission_success', $appointment_id, [
				'customer_name'    => $customer_name,
				'customer_email'   => $customer_email,
				'appointment_date' => $services['appointment_date'],
				'appointment_time' => $services['start_time'] . '-' . $services['end_time'],
				'service_id'       => $services['service_id'],
				'service_name'     => get_the_title( $services['service_id'] ),
				'staff_id'         => $services['staff_id']
			] );

			return $appointment_id;
		} catch ( \Exception $e ) {
			$wpdb->query( 'ROLLBACK' );

			return false;
		}
	}
}
