<?php

namespace WPMinimize\TimeFix\Payments\WooCommerce;

use WPMinimize\TimeFix\Helpers\Functions;
use WPMinimize\TimeFix\Models\Service;

class ProductCompatibility {

	public function __construct() {
		// Override the product class for service post type
		add_filter( 'woocommerce_product_class', [ $this, 'product_class' ], 10, 4 );
		// Disable quantity input box in cart for service products
		add_filter( 'woocommerce_cart_item_quantity', [ $this, 'disable_quantity_box' ], 10, 3 );
		// Add custom meta data to order line item for service booking info
		add_action( 'woocommerce_checkout_create_order_line_item', [ $this, 'order_item_line' ], 10, 4 );
		// Customize the checkout fields
		add_filter( 'woocommerce_checkout_fields', [ __CLASS__, 'custom_checkout_fields' ] );
		// Create appointment when order created
		add_action( 'woocommerce_checkout_order_created', [ __CLASS__, 'create_appointment' ], 10 );
		// Change appointment status when order status changed
		add_action( 'woocommerce_order_status_changed', [ __CLASS__, 'update_appointment_status' ], 10, 3 );
	}

	/**
	 * Update appointment status when WooCommerce updated status
	 *
	 * @param int    $order_id
	 * @param string $old_status
	 * @param string $new_status
	 */
	public static function update_appointment_status( $order_id, $old_status, $new_status ) {
		$wc_order       = wc_get_order( $order_id );
		$appointment_id = absint( $wc_order->get_meta( 'wpm_timefix_appointment_id' ) );

		if ( $appointment_id ) {
			Functions::update_appointment_status( $appointment_id, $new_status );
		}
	}

	/**
	 * @param \WC_Order $wc_order
	 *
	 * @return false|void
	 */
	public static function create_appointment( $wc_order ) {
		if ( ! $wc_order ) {
			return false;
		}

		$service_id = absint( $wc_order->get_meta( 'wpm_timefix_appointment_id' ) );

		if ( $service_id ) {
			return false;
		}

		$wc_items = $wc_order->get_items();

		if ( ! $wc_items ) {
			return false;
		}

		$service_data = [];

		foreach ( $wc_items as $item ) {
			if ( isset( $item->legacy_values ) ) {
				$values       = $item->legacy_values;
				$booking_data = $values['wpm_timefix_booking_data'] ?? [];
				if ( empty( $booking_data ) ) {
					continue;
				}
				$service_id = ! empty( $booking_data['service_id'] ) ? absint( $booking_data['service_id'] ) : 0;
				if ( wpm_timefix()->service_post_type !== get_post_type( $service_id ) ) {
					continue;
				}
				$staff_id = ! empty( $booking_data['staff_id'] ) ? absint( $booking_data['staff_id'] ) : 0;
				if ( wpm_timefix()->staff_post_type !== get_post_type( $staff_id ) ) {
					continue;
				}
				$booking_date  = isset( $booking_data['date'] ) ? sanitize_text_field( $booking_data['date'] ) : '';
				$time_slot     = isset( $booking_data['time'] ) ? sanitize_text_field( $booking_data['time'] ) : '';
				$time_slot_arr = explode( '-', $time_slot );
				$start_time    = isset( $time_slot_arr[0] ) ? trim( $time_slot_arr[0] ) : '';
				$end_time      = isset( $time_slot_arr[1] ) ? trim( $time_slot_arr[1] ) : '';
				// Organize information
				$service_data['appointment_date'] = date( 'Y-m-d', strtotime( $booking_date ) );
				$service_data['start_time']       = $start_time;
				$service_data['end_time']         = $end_time;
				$service_data['service_id']       = $service_id;
				$service_data['staff_id']         = $staff_id;
				$service_data['price']            = $values['wpm_timefix_service_price'] ?? 0;
			}
		}

		if ( empty( $service_data ) ) {
			return false;
		}

		$service = wpm_timefix()->factory->get_service( $service_id );

		if ( ! is_a( $service, Service::class ) ) {
			return false;
		}

		$customer = [
			'user_id' => $wc_order->get_customer_id(),
			'name'    => $wc_order->get_billing_first_name(),
			'email'   => $wc_order->get_billing_email(),
			'phone'   => $wc_order->get_billing_phone()
		];

		$customer_note = method_exists( $wc_order, 'get_customer_note' ) ? $wc_order->get_customer_note() : $wc_order->customer_note;

		$appointment = [
			'note' => $customer_note
		];

		$order = [
			'subtotal'       => $wc_order->get_subtotal(),
			'total'          => $wc_order->get_total(),
			'payment_method' => 'woocommerce',
			'wc_order_id'    => $wc_order->get_id(),
			'key'            => $wc_order->get_order_key(),
			'status'         => $wc_order->get_status()
		];

		$appointment_id = Functions::insert_appointment_data( $customer, $appointment, $service_data, $order );

		if ( $appointment_id ) {
			$wc_order->add_meta_data( 'wpm_timefix_appointment_id', $appointment_id );
			$wc_order->save();
		}
	}

	/**
	 * Filter the product class used when loading a product object.
	 *
	 * @param string $classname    The default product class.
	 * @param string $product_type The product type.
	 * @param string $post_type    The post type.
	 * @param int    $product_id   The product ID.
	 *
	 * @return string Custom product class if it's a service product.
	 */
	public function product_class( $classname, $product_type, $post_type, $product_id ) {
		if ( wpm_timefix()->service_post_type == get_post_type( $product_id ) ) {
			$classname = ServiceProduct::class;
		}

		return $classname;
	}

	/**
	 * Filter the quantity display in the cart.
	 * Replaces the quantity input with plain text for service products.
	 *
	 * @param string $product_quantity The quantity HTML.
	 * @param string $cart_item_key    The cart item key.
	 * @param array  $cart_item        The cart item data array.
	 *
	 * @return string Modified quantity HTML for service products.
	 */
	public function disable_quantity_box( $product_quantity, $cart_item_key, $cart_item ) {
		return ( get_class( $cart_item['data'] ) === ServiceProduct::class ) ? sprintf( '<span style="text-align: center; display: block">%s</span>', $cart_item['quantity'] ) : $product_quantity;
	}

	/**
	 * Add custom meta data to the WooCommerce order line item.
	 * Includes service ID, staff ID/name, and booking date/time.
	 *
	 * @param \WC_Order_Item_Product $item          Order item object.
	 * @param string                 $cart_item_key Cart item key.
	 * @param array                  $values        Cart item values.
	 * @param \WC_Order              $order         Order object.
	 */
	public function order_item_line( $item, $cart_item_key, $values, $order ) {
		if ( wpm_timefix()->service_post_type === get_post_type( $values['product_id'] ) ) {
			if ( isset( $values['wpm_timefix_booking_data'] ) ) {
				$item->add_meta_data( 'Service ID', $values['wpm_timefix_booking_data']['service_id'] ?? '' );
				$staff_id = $values['wpm_timefix_booking_data']['staff_id'] ?? '';
				if ( $staff_id ) {
					$item->add_meta_data( 'Staff Name', get_the_title( $staff_id ) );
					$item->add_meta_data( 'Staff ID', $staff_id );
				}
				$item->add_meta_data( 'Booking Date', $values['wpm_timefix_booking_data']['date'] ?? '' );
				$item->add_meta_data( 'Booking Time', $values['wpm_timefix_booking_data']['time'] ?? '' );
			}
		}
	}

	/**
	 * Customize WooCommerce checkout fields.
	 * Change the placeholder text for the order comment field.
	 *
	 * @param array $fields Checkout fields.
	 *
	 * @return array Modified checkout fields.
	 */
	public static function custom_checkout_fields( $fields ) {
		$fields['order']['order_comments']['placeholder'] = esc_html__( 'Additional booking note', 'timefix' );

		return $fields;
	}
}
