<?php

namespace WPMinimize\TimeFix\Helpers;

defined( 'ABSPATH' ) || exit;

use WPMinimize\TimeFix\Traits\DateTrait;
use DateTimeImmutable;
use DateTimeZone;

class Utility {
	use DateTrait;

	/**
	 * Converts php DateTime format to Javascript Moment format.
	 *
	 * @param string $phpFormat
	 *
	 * @return string
	 */
	public static function dateFormatPHPToMoment( $phpFormat ) {
		$replacements = [
			'A' => 'A',      // for the sake of escaping below
			'a' => 'a',      // for the sake of escaping below
			'B' => '',       // Swatch internet time (.beats), no equivalent
			'c' => 'YYYY-MM-DD[T]HH:mm:ssZ', // ISO 8601
			'D' => 'ddd',
			'd' => 'DD',
			'e' => 'zz',     // deprecated since version 1.6.0 of moment.js
			'F' => 'MMMM',
			'G' => 'H',
			'g' => 'h',
			'H' => 'HH',
			'h' => 'hh',
			'I' => '',       // Daylight Saving Time? => moment().isDST();
			'i' => 'mm',
			'j' => 'D',
			'L' => '',       // Leap year? => moment().isLeapYear();
			'l' => 'dddd',
			'M' => 'MMM',
			'm' => 'MM',
			'N' => 'E',
			'n' => 'M',
			'O' => 'ZZ',
			'o' => 'YYYY',
			'P' => 'Z',
			'r' => 'ddd, DD MMM YYYY HH:mm:ss ZZ', // RFC 2822
			'S' => 'o',
			's' => 'ss',
			'T' => 'z',      // deprecated since version 1.6.0 of moment.js
			't' => '',       // days in the month => moment().daysInMonth();
			'U' => 'X',
			'u' => 'SSSSSS', // microseconds
			'v' => 'SSS',    // milliseconds (from PHP 7.0.0)
			'W' => 'W',      // for the sake of escaping below
			'w' => 'e',
			'Y' => 'YYYY',
			'y' => 'YY',
			'Z' => '',       // time zone offset in minutes => moment().zone();
			'z' => 'DDD',
		];

		// Converts escaped characters.
		foreach ( $replacements as $from => $to ) {
			$replacements[ '\\' . $from ] = '[' . $from . ']';
		}

		return strtr( $phpFormat, $replacements );
	}

	/**
	 * Format a time supplied as string to a format from a format.
	 *
	 * @param string $value
	 * @param null   $to
	 * @param null   $from
	 *
	 * @return string
	 * @since 1.0
	 *
	 */
	public static function formatDate( $value, $to = null, $from = null ) {

		$to   = is_null( $to ) ? Functions::date_format() : $to;
		$from = is_null( $from ) ? Functions::date_format() : $from;

		if ( strlen( $value ) > 0 ) {

			return self::createFromFormat( $from, $value )->format( $to );

		} else {

			return $value;
		}
	}

	/**
	 * Format a time supplied as string to a format from a format.
	 *
	 * @param string $value
	 * @param null   $to
	 * @param null   $from
	 *
	 * @return string
	 * @since 1.0
	 *
	 */
	public static function formatTime( $value, $to = null, $from = null ) {

		$to   = is_null( $to ) ? Functions::time_format() : $to;
		$from = is_null( $from ) ? Functions::time_format() : $from;

		if ( strlen( $value ) > 0 ) {
			try {
				return self::createFromFormat( $from, $value )->format( $to );
			} catch ( \Exception $e ) {
				return '';
			}

		} else {

			return $value;
		}
	}

	/**
	 * Format a time supplied as string to a format from a format.
	 *
	 * @param string             $value
	 * @param null| DateTimeZone $timeZone
	 *
	 * @return DateTimeImmutable
	 * @since 1.0
	 *
	 */
	public static function sanitizedDateObj( $value, $timeZone = null ) {
		$timeZone = empty( $timeZone ) ? wp_timezone() : $timeZone;

		if ( $value ) {
			try {
				return new DateTimeImmutable( $value, $timeZone );
			} catch ( \Exception $e ) {
				return null;
			}
		}

		return null;

	}

}
