<?php

namespace WPMinimize\TimeFix\Helpers;

defined( 'ABSPATH' ) || exit;

class Installer {
	public static function activate() {

		if ( ! is_blog_installed() ) {
			return;
		}

		// Check if we are not already running this routine.
		if ( 'yes' === get_transient( 'wpm_timefix_installing' ) ) {
			return;
		}

		// If we made it till here nothing is running yet, lets set the transient now.
		set_transient( 'wpm_timefix_installing', 'yes', MINUTE_IN_SECONDS * 10 );

		self::create_tables();
		self::create_options();

		delete_transient( 'wpm_timefix_installing' );

		Functions::update_permalink();
	}

	private static function create_options() {
		$options = [
			'general_settings'         => [
				'duration' => [
					'value' => 5,
					'label' => '5 Mins'
				],
				'interval' => 0,
				'status'   => 'pending'
			],
			'permalink_settings'       => [
				'service_single_base'  => 'timefix_service',
				'service_archive_base' => 'timefix_services',
				'staff_single_base'    => 'timefix_staff',
				'staff_archive_base'   => 'timefix_staffs',
			],
			'email_settings'           => [
				'notify_admin' => [ 'appointment_submission' ],
				'notify_user'  => [ 'appointment_submission', 'appointment_confirmation', 'appointment_cancellation' ],
				'from_name'    => get_option( 'blogname' ),
				'from_email'   => get_option( 'admin_email' ),
				'admin_email'  => get_option( 'admin_email' )
			],
			'payment_settings'         => [
				'currency'            => 'USD',
				'currency_position'   => 'left',
				'thousands_separator' => ',',
				'decimal_separator'   => '.',
				'payment_enable'      => 1,
				'payment_methods'     => [ 'offline' ]
			],
			'offline_payment_settings' => [
				'title'       => 'Pay Later',
				'description' => 'Make your payment cash at venue, when take service.'
			]
		];

		foreach ( $options as $key => $value ) {
			$option_name = 'wpm_timefix_' . trim( $key );
			if ( ! get_option( $option_name ) ) {
				update_option( $option_name, $value );
			}
		}
	}

	private static function create_tables() {
		global $wpdb;

		$wpdb->hide_errors();

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		dbDelta( self::get_table_schema() );
	}

	/**
	 * @return array
	 */
	static function get_table_schema() {
		global $wpdb;

		$collate = '';

		if ( $wpdb->has_cap( 'collation' ) ) {
			$collate = $wpdb->get_charset_collate();
		}

		$service_meta_table_name    = $wpdb->prefix . "wpm_timefix_service_meta";
		$staff_meta_table_name      = $wpdb->prefix . "wpm_timefix_staff_meta";
		$appointments_table_name    = $wpdb->prefix . "wpm_timefix_appointments";
		$appointment_meta_table     = $wpdb->prefix . "wpm_timefix_appointment_meta";
		$appointment_services_table = $wpdb->prefix . "wpm_timefix_appointment_services";
		$orders_table               = $wpdb->prefix . "wpm_timefix_orders";
		$customers_table            = $wpdb->prefix . "wpm_timefix_customers";
		$table_schema               = [];

		if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $appointments_table_name ) ) !== $appointments_table_name ) {
			$table_schema[] = self::appointments_table_schema( $appointments_table_name, $collate );
		}

		if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $appointment_meta_table ) ) !== $appointment_meta_table ) {
			$table_schema[] = self::appointment_meta_table_schema( $appointment_meta_table, $collate );
		}

		if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $appointment_services_table ) ) !== $appointment_services_table ) {
			$table_schema[] = self::appointment_services_table_schema( $appointment_services_table, $collate );
		}

		if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $orders_table ) ) !== $orders_table ) {
			$table_schema[] = self::orders_table_schema( $orders_table, $collate );
		}

		if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $customers_table ) ) !== $customers_table ) {
			$table_schema[] = self::customers_table_schema( $customers_table, $collate );
		}

		if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $service_meta_table_name ) ) !== $service_meta_table_name ) {
			$table_schema[] = "CREATE TABLE $service_meta_table_name (
                      meta_id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                      service_id BIGINT(20) UNSIGNED NOT NULL,
                      meta_key varchar(191) NOT NULL,
                      meta_value longtext DEFAULT NULL,
                      PRIMARY KEY (meta_id)
                      ) $collate;";
		}

		if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $staff_meta_table_name ) ) !== $staff_meta_table_name ) {
			$table_schema[] = "CREATE TABLE $staff_meta_table_name (
                      meta_id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                      staff_id BIGINT(20) UNSIGNED NOT NULL,
                      meta_key varchar(191) NOT NULL,
                      meta_value longtext DEFAULT NULL,
                      PRIMARY KEY (meta_id)
                      ) $collate;";
		}

		return $table_schema;
	}

	public static function appointments_table_schema( $table_name, $collate ) {

		return "CREATE TABLE $table_name (
                      id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                      customer_id BIGINT(20) UNSIGNED DEFAULT NULL,
                      total_price double NULL,
                      additional_notes longtext DEFAULT NULL,
                      info text DEFAULT NULL,
                      status ENUM('pending', 'confirmed', 'cancelled', 'completed', 'on_hold') DEFAULT 'pending',
                      created_at timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                      updated_at timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                      PRIMARY KEY (id)
                      ) $collate;";
	}

	public static function appointment_services_table_schema( $table_name, $collate ) {

		return "CREATE TABLE $table_name (
                      id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                      appointment_id BIGINT(20) UNSIGNED NOT NULL,
                      service_id BIGINT(20) UNSIGNED NOT NULL,
                      staff_id BIGINT(20) UNSIGNED DEFAULT NULL,
                      location_id BIGINT(20) UNSIGNED DEFAULT NULL,
                      service_extras text DEFAULT NULL,
                      appointment_date DATE NOT NULL,
                      start_time TIME DEFAULT NULL,
                      end_time TIME DEFAULT NULL,
                      price double NULL,
                      PRIMARY KEY (id)
                      ) $collate;";
	}

	public static function orders_table_schema( $table_name, $collate ) {

		return "CREATE TABLE $table_name (
                      id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                      object_id BIGINT(20) UNSIGNED NOT NULL,
                      object_type ENUM('appointment', 'membership') NOT NULL,
                      customer_id BIGINT(20) UNSIGNED DEFAULT NULL,
                      woo_order_id BIGINT(20) UNSIGNED DEFAULT NULL,
                      subtotal double NULL,
                      total double NULL,
                      payment_method VARCHAR(50),
                      payment_status ENUM('paid', 'partially_paid', 'unpaid', 'failed', 'cancelled', 'completed', 'on_hold', 'pending') DEFAULT 'unpaid',
                      transaction_reference VARCHAR(100),
                      coupon_code VARCHAR(100),
                      coupon_discount double NULL,
                      tax_total double NULL,
                      info text DEFAULT NULL,
                      created_at timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                      updated_at timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                      PRIMARY KEY (id)
                      ) $collate;";
	}

	public static function customers_table_schema( $table_name, $collate ) {

		return "CREATE TABLE $table_name (
                      id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                      wp_user_id INT DEFAULT NULL,
                      name varchar(191) NOT NULL,
                      email varchar(100) DEFAULT NULL,
                      phone varchar(20) DEFAULT NULL,
                      info text DEFAULT NULL,
                      created_at timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                      updated_at timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                      PRIMARY KEY (id)
                      ) $collate;";
	}

	public static function appointment_meta_table_schema( $table_name, $collate ) {
		return "CREATE TABLE $table_name (
                      meta_id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                      appointment_id BIGINT(20) UNSIGNED NOT NULL,
                      meta_key varchar(191) NOT NULL,
                      meta_value longtext DEFAULT NULL,
                      PRIMARY KEY (meta_id)
                      ) $collate;";
	}

	public static function deactivate() {
		Functions::update_permalink();
	}
}