<?php

namespace WPMinimize\TimeFix\Helpers;

use WPMinimize\TimeFix\Payments\OfflineGateway;
use WPMinimize\TimeFix\Payments\PaymentGatewayInterface;
use WPMinimize\TimeFix\Payments\WooCommerce\WooCommerceGateway;

class GatewayManager {
	protected static $gateways = [];

	public static function register_gateways() {
		self::$gateways = [
			'offline'     => new OfflineGateway(),
			'woocommerce' => new WooCommerceGateway(),
		];

		/**
		 * Allow adding more gateways via hook
		 */
		self::$gateways = apply_filters( 'wpm_timefix_payment_gateways', self::$gateways );
	}

	/**
	 * Get a single gateway instance by ID
	 */
	public static function get_gateway( $id ): ?PaymentGatewayInterface {
		return self::$gateways[ $id ] ?? null;
	}

	/**
	 * Return all registered gateways
	 */
	public static function get_all(): array {
		return self::$gateways;
	}

	/**
	 * Get all enabled gateways.
	 */
	public static function get_enabled_gateways(): array {
		return array_filter(
			self::$gateways,
			fn( PaymentGatewayInterface $gateway ) => $gateway->is_enabled()
		);
	}

	/**
	 * Get all enabled gateways as renderable array.
	 */
	public static function get_enabled_gateways_info(): array {
		$enabled = self::get_enabled_gateways();
		$data    = [];

		foreach ( $enabled as $gateway ) {
			$data[ $gateway->get_id() ] = [
				'title'        => $gateway->get_title(),
				'description'  => $gateway->get_description(),
				'instructions' => $gateway->get_instructions(),
			];
		}

		return $data;
	}

	/**
	 * Check if WooCommerce payment is enabled.
	 *
	 * @return bool
	 */
	public static function is_woocommerce_payment_enabled(): bool {
		$gateway = self::get_gateway( 'woocommerce' );

		return $gateway && $gateway->is_enabled();
	}
}
