<?php

namespace WPMinimize\TimeFix\Helpers;

defined( 'ABSPATH' ) || exit;

use WPMinimize\TimeFix\Traits\AppointmentsTrait;
use WPMinimize\TimeFix\Traits\CustomerTrait;
use WPMinimize\TimeFix\Traits\OrderTrait;
use WPMinimize\TimeFix\Traits\ServiceTrait;
use WPMinimize\TimeFix\Traits\SettingsTrait;
use WPMinimize\TimeFix\Traits\StaffTrait;

class Functions {
    use SettingsTrait;
    use StaffTrait;
    use AppointmentsTrait;
    use OrderTrait;
    use CustomerTrait;
    use ServiceTrait;

    /**
     * @param        $template_name
     * @param string $template_path
     * @param string $default_path
     *
     * @return mixed|void
     */
    public static function locate_template( $template_name, $template_path = '', $default_path = '' ) {
        $template_name = $template_name . '.php';

        if ( ! $template_path ) {
            $template_path = wpm_timefix()->get_template_path();
        }

        if ( ! $default_path ) {
            $default_path = wpm_timefix()->plugin_path() . '/templates/';
        }

        $template_files[] = trailingslashit( $template_path ) . $template_name;

        $template = locate_template( apply_filters( 'wpm_timefix_locate_template_files', $template_files, $template_name, $template_path, $default_path ) );

        // Get default template/.
        if ( ! $template ) {
            $template = trailingslashit( $default_path ) . $template_name;
        }

        return apply_filters( 'wpm_timefix_locate_template', $template, $template_name );
    }

    /**
     * Template Content
     *
     * @param string $template_name Template name
     * @param array  $args          Arguments. (default: array)
     * @param string $template_path Template path. (default: '')
     * @param string $default_path  Default path. (default: '')
     */
    public static function get_template( $template_name, $args = null, $template_path = '', $default_path = '' ) {
        if ( ! empty( $args ) && is_array( $args ) ) {
            extract( $args, EXTR_SKIP );
        }

        $located = self::locate_template( $template_name, $template_path, $default_path );

        if ( ! file_exists( $located ) ) {
            return;
        }

        // Allow 3rd party plugin filter template file from their plugin.
        $located = apply_filters( 'wpm_timefix_get_template', $located, $template_name, $args );

        do_action( 'wpm_timefix_before_template_part', $template_name, $located, $args );

        include $located;

        do_action( 'wpm_timefix_after_template_part', $template_name, $located, $args );
    }

    public static function get_week_days() {
        global $wp_locale;
        $weekStart = apply_filters( 'wpm_timefix_start_of_week', get_option( 'start_of_week' ) );
        $weekday   = $wp_locale->weekday;
        for ( $i = 0; $i < $weekStart; $i ++ ) {

            $day = array_slice( $weekday, 0, 1, true );
            unset( $weekday[ $i ] );

            $weekday = $weekday + $day;
        }

        return $weekday;
    }

    /**
     * Time Format.
     *
     * @return string
     */
    public static function time_format() {
        return apply_filters( 'wpm_timefix_time_format', get_option( 'time_format' ) );
    }

    /**
     * Date Format.
     *
     * @return string
     */
    public static function date_format() {
        return apply_filters( 'wpm_timefix_date_format', get_option( 'date_format' ) );
    }

    public static function get_social_profile_list() {
        $profiles = [
                'facebook'  => esc_html__( 'Facebook', 'timefix' ),
                'x-twitter' => esc_html__( 'Twitter', 'timefix' ),
                'youtube'   => esc_html__( 'Youtube', 'timefix' ),
                'instagram' => esc_html__( 'Instagram', 'timefix' ),
                'linkedin'  => esc_html__( 'LinkedIn', 'timefix' ),
                'tiktok'    => esc_html__( 'Tiktok', 'timefix' )
        ];

        return apply_filters( 'wpm_timefix_social_profiles_list', $profiles );
    }

    /**
     * @return mixed|null
     */
    public static function date_range_picker_locale() {
        return apply_filters( 'wpm_timefix_daterangepicker_locale', [
                'format'      => Utility::dateFormatPHPToMoment( Functions::date_format() ),
                'applyLabel'  => esc_html__( 'Apply', 'timefix' ),
                'cancelLabel' => esc_html__( 'Cancel', 'timefix' ),
                'fromLabel'   => esc_html__( 'From', 'timefix' ),
                'toLabel'     => esc_html__( 'To', 'timefix' ),
                'daysOfWeek'  => [
                        esc_html__( 'Su', 'timefix' ),
                        esc_html__( 'Mo', 'timefix' ),
                        esc_html__( 'Tu', 'timefix' ),
                        esc_html__( 'We', 'timefix' ),
                        esc_html__( 'Th', 'timefix' ),
                        esc_html__( 'Fr', 'timefix' ),
                        esc_html__( 'Sa', 'timefix' )
                ],
                'monthNames'  => [
                        esc_html__( 'January', 'timefix' ),
                        esc_html__( 'February', 'timefix' ),
                        esc_html__( 'March', 'timefix' ),
                        esc_html__( 'April', 'timefix' ),
                        esc_html__( 'May', 'timefix' ),
                        esc_html__( 'June', 'timefix' ),
                        esc_html__( 'July', 'timefix' ),
                        esc_html__( 'August', 'timefix' ),
                        esc_html__( 'September', 'timefix' ),
                        esc_html__( 'October', 'timefix' ),
                        esc_html__( 'November', 'timefix' ),
                        esc_html__( 'December', 'timefix' ),
                ],
                'firstDay'    => 1
        ] );
    }

    public static function sanitize( $value, $type = null ) {
        $original_value = $value;
        $sanitize_value = null;

        $new_bhs = [];

        switch ( $type ) {
            case 'textarea':
                $sanitize_value = sanitize_textarea_field( wp_unslash( $value ) );
                break;
            case 'html_textarea':
                $sanitize_value = wp_kses_post( $value );
                break;
            case 'checkbox':
                $sanitize_value = array_map( 'esc_attr', is_array( $value ) ? $value : [] );
                break;
            case 'url':
                $sanitize_value = esc_url_raw( $value );
                break;
            case 'date':
                $sanitize_value = Utility::formatDate( $value, 'Y-m-d' );
                break;
            case 'weekly_schedule':
                foreach ( self::get_week_days() as $day_key => $day ) {
                    if ( ! empty( $value[ $day_key ] ) ) {
                        $bh = $value[ $day_key ];
                        if ( isset( $bh['open'] ) && filter_var( $bh['open'], FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE ) ) {
                            $new_bhs[ $day_key ]['open'] = true;
                            if ( isset( $bh['times'] ) && is_array( $bh['times'] ) && ! empty( $bh['times'] ) ) {
                                $new_times = [];
                                foreach ( $bh['times'] as $time ) {
                                    if ( ! empty( $time['start'] ) && ! empty( $time['end'] ) ) {
                                        $start = Utility::formatTime( $time['start'], 'H:i' );
                                        $end   = Utility::formatTime( $time['end'], 'H:i' );
                                        if ( $start && $end ) {
                                            $new_times[] = [ 'start' => $start, 'end' => $end ];
                                        }
                                    }
                                }
                                if ( ! empty( $new_times ) ) {
                                    $new_bhs[ $day_key ]['times'] = $new_times;
                                }
                            }
                        } else {
                            $new_bhs[ $day_key ]['open'] = false;
                        }
                    }
                }
                $sanitize_value = $new_bhs;
                break;
            case 'special_schedule':
                $temp_count = 0;
                $temp_keys  = [];
                foreach ( $value as $sh_key => $sbh ) {
                    if ( ! empty( $sbh['date'] ) && ! isset( $temp_keys[ $sbh['date'] ] ) && $date = Utility::formatDate( $sbh['date'], 'Y-m-d' ) ) {
                        $temp_keys[] = $new_bhs[ $temp_count ]['date'] = $date;
                        if ( filter_var( $sbh['open'], FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE ) ) {
                            $new_bhs[ $temp_count ]['open'] = true;
                            if ( isset( $sbh['times'] ) && is_array( $sbh['times'] ) && ! empty( $sbh['times'] ) ) {
                                $new_times = [];
                                foreach ( $sbh['times'] as $time ) {
                                    if ( ! empty( $time['start'] ) && ! empty( $time['end'] ) ) {
                                        $start = Utility::formatTime( $time['start'], 'H:i' );
                                        $end   = Utility::formatTime( $time['end'], 'H:i' );
                                        if ( $start && $end ) {
                                            $new_times[] = [ 'start' => $start, 'end' => $end ];
                                        }
                                    }
                                }
                                if ( ! empty( $new_times ) ) {
                                    $new_bhs[ $temp_count ]['times'] = $new_times;
                                }
                            }
                        } else {
                            $new_bhs[ $temp_count ]['open'] = false;
                        }
                    }
                    $temp_count ++;
                }
                $sanitize_value = $new_bhs;
                break;
            default:
                $sanitize_value = sanitize_text_field( wp_unslash( $value ) );
                break;
        }

        return apply_filters( 'wpm_timefix_sanitize', $sanitize_value, $original_value, $type );
    }

    /**
     * Output the pagination.
     */
    public static function pagination() {
        global $paged;

        $range     = 1;
        $showItems = ( $range * 2 ) + 1;

        if ( get_query_var( 'paged' ) ) {
            $paged = get_query_var( 'paged' );
        } else if ( get_query_var( 'page' ) ) {
            $paged = get_query_var( 'page' );
        } else {
            $paged = 1;
        }

        $paged = absint( $paged );

        if ( ! isset( $pages ) ) {
            global $wp_query;
            $pages = $wp_query->max_num_pages;

            if ( ! $pages ) {
                $pages = 1;
            }
        }

        self::get_template( 'global/pagination', compact( 'paged', 'showItems', 'pages' ) );
    }

    public static function verify_nonce() {
        $nonce     = isset( $_REQUEST[ wpm_timefix()->nonceId ] ) ? sanitize_text_field( $_REQUEST[ wpm_timefix()->nonceId ] ) : null; /* phpcs:ignore WordPress.Security.NonceVerification.Recommended */
        $nonceText = wpm_timefix()->nonceText;
        if ( wp_verify_nonce( $nonce, $nonceText ) ) {
            return true;
        }

        return false;
    }

    public static function get_thousands_separator() {
        $currency_settings = self::get_option( 'wpm_timefix_payment_settings' );

        return isset( $currency_settings['thousands_separator'] ) ? stripslashes( $currency_settings['thousands_separator'] ) : ',';
    }

    public static function get_decimal_separator() {
        $currency_settings = self::get_option( 'wpm_timefix_payment_settings' );

        return isset( $currency_settings['decimal_separator'] ) ? stripslashes( $currency_settings['decimal_separator'] ) : '.';
    }

    public static function get_price_decimals() {
        return absint( apply_filters( 'wpm_timefix_get_price_decimals', 2 ) );
    }

    public static function trim_zeros( $price ) {
        $decimal_separator = self::get_decimal_separator();

        return preg_replace( '/' . preg_quote( $decimal_separator, '/' ) . '0++$/', '', $price );
    }

    public static function get_currency_symbol( $currency = '' ) {
        if ( ! $currency ) {
            $currency = self::get_currency();
        }

        $symbols         = Options::get_currency_symbols();
        $currency_symbol = isset( $symbols[ $currency ] ) ? $symbols[ $currency ] : '';

        return apply_filters( 'wpm_timefix_get_currency_symbol', $currency_symbol, $currency );
    }

    public static function get_currency() {
        $currency = self::get_option_item( 'wpm_timefix_payment_settings', 'currency' );

        return apply_filters( 'wpm_timefix_get_currency', $currency );
    }

    public static function get_price_format() {
        $currency_settings = self::get_option_item( 'wpm_timefix_payment_settings', 'currency_position' );
        $currency_pos      = ! empty( $currency_settings ) ? $currency_settings : 'left';
        $format            = '%1$s%2$s';

        switch ( $currency_pos ) {
            case 'left':
                $format = '%1$s%2$s';
                break;
            case 'right':
                $format = '%2$s%1$s';
                break;
            case 'left_space':
                $format = '%1$s&nbsp;%2$s';
                break;
            case 'right_space':
                $format = '%2$s&nbsp;%1$s';
                break;
        }

        return apply_filters( 'wpm_timefix_get_price_format', $format, $currency_pos );
    }

    public static function get_formatted_price( $price ) {
        $thousands_sep = self::get_thousands_separator();
        $decimal_sep   = self::get_decimal_separator();
        $decimals      = self::get_price_decimals();

        $original_price = $price;

        $price = (float) $price;

        $unformatted_price = $price;
        $negative          = $price < 0;

        $price = apply_filters( 'wpm_timefix_raw_payment_price', $negative ? $price * - 1 : $price, $original_price );

        $price = apply_filters( 'wpm_timefix_formatted_payment_price',
                number_format( $price, $decimals, $decimal_sep, $thousands_sep ),
                $price,
                $decimals,
                $decimal_sep,
                $thousands_sep,
                $original_price );

        if ( apply_filters( 'wpm_timefix_payment_price_trim_zeros', false ) && $decimals > 0 ) {
            $price = self::trim_zeros( $price );
        }

        return apply_filters( 'wpm_timefix_get_payment_formatted_price', $price, $unformatted_price, $decimals, $decimal_sep, $thousands_sep );
    }

    public static function get_price_html( $price ) {
        $original_price  = $price;
        $formatted_price = self::get_formatted_price( $original_price );
        $currency_symbol = self::get_currency_symbol( self::get_currency() );
        $price_format    = self::get_price_format();

        $formatted_payment_price_html = apply_filters(
                'wpm_timefix_formatted_payment_price_html',
                sprintf( $price_format, '<span class="timefix-price-currencySymbol">' . $currency_symbol . '</span>', $formatted_price ),
                $price_format,
                $currency_symbol,
                $formatted_price,
                $original_price
        );
        $payment_price_meta_html      = apply_filters( 'wpm_timefix_payment_price_meta_html', '', $price_format, $currency_symbol, $formatted_price, $original_price );
        $payment_price_meta_html      = $payment_price_meta_html ? apply_filters( 'wpm_timefix_payment_price_meta_wrap_html',
                sprintf( '<span class="timefix-payment-price-meta">%s</span>', $payment_price_meta_html ),
                $payment_price_meta_html ) : null;
        $payment_price_html_format    = apply_filters( 'wpm_timefix_payment_price_amount_html_format', '<span class="timefix-price-amount amount">%1$s</span>%2$s' );
        $payment_price_html           = sprintf( $payment_price_html_format, $formatted_payment_price_html, $payment_price_meta_html );

        return apply_filters( 'wpm_timefix_get_payment_formatted_price_html',
                $payment_price_html,
                $formatted_price,
                $original_price,
                $currency_symbol,
                $price_format );
    }

    /**
     * @param $query
     *
     * @return int|void
     */
    public static function count_rows( $query ) {
        global $wpdb;

        $results = $wpdb->get_results( $query );

        if ( ! is_wp_error( $results ) ) {
            return count( $results );
        }

        return 0;
    }

    /**
     * @param $schedule
     *
     * @return void
     */
    public static function display_schedule( $schedule ) {
        $daysOfWeek = self::get_week_days();
        ?>
        <div class="timefix-weekly-schedule">
            <?php
            foreach ( $daysOfWeek as $key => $day ) {
                $day_open   = isset( $schedule[ $key ] ) && ! empty( $schedule[ $key ]['open'] );
                $slot_exist = $day_open && isset( $schedule[ $key ]['times'] );
                ?>
                <div class="timefix-day-schedule <?php echo esc_attr( $slot_exist ? 'day-open' : 'day-closed' ); ?>">
                    <h5 class="week-day"><?php echo esc_html( $day ); ?></h5>
                    <?php
                    if ( $slot_exist ) {
                        $times = $schedule[ $key ]['times'];
                        foreach ( $times as $time ) {
                            ?>
                            <div class="day-hours">
                                <span><?php echo esc_html( Utility::formatTime( $time['start'], null, 'H:i' ) ); ?></span>
                                <span class="time-separator">-</span>
                                <span><?php echo esc_html( Utility::formatTime( $time['end'], null, 'H:i' ) ); ?></span>
                            </div>
                            <?php
                        }
                    } else {
                        // If the day is closed
                        ?>
                        <div class="day-hours"><?php esc_html_e( 'Closed', 'timefix' ); ?></div>
                        <?php
                    }
                    ?>
                </div>
                <?php
            }
            ?>
        </div>
        <?php
    }

    /**
     * @param $post_id
     * @param $post_type
     *
     * @return bool
     */
    public static function is_post_type_published( $post_id, $post_type ) {
        $post = get_post( absint( $post_id ) );

        if ( $post && $post->post_type === $post_type && $post->post_status === 'publish' ) {
            return true;
        }

        return false;
    }

    /**
     * @return bool
     */
    public static function is_timefix_pro() {
        return false;
    }

    /**
     * WooCommerce is activated
     *
     * @return boolean
     */
    public static function is_woocommerce_activated() {
        if ( ! function_exists( 'is_plugin_active' ) ) {
            include_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        return is_plugin_active( 'woocommerce/woocommerce.php' );
    }

    public static function update_permalink() {
        update_option( 'wpm_timefix_needs_flush', 1 );
    }

}