<?php

namespace WPMinimize\TimeFix\Frontend;

defined( 'ABSPATH' ) || exit;

use WPMinimize\TimeFix\Common\ServiceHandler;
use WPMinimize\TimeFix\Common\StaffHandler;
use WPMinimize\TimeFix\Helpers\Functions;

class AjaxHooks {
	public static function init() {
		add_action( 'wp_ajax_timefix_staff_time_slots', [ __CLASS__, 'get_time_slots' ] );
		add_action( 'wp_ajax_nopriv_timefix_staff_time_slots', [ __CLASS__, 'get_time_slots' ] );
		add_action( 'wp_ajax_timefix_submit_appointment_form', [ __CLASS__, 'save_appointment_data' ] );
		add_action( 'wp_ajax_nopriv_timefix_submit_appointment_form', [ __CLASS__, 'save_appointment_data' ] );
		add_action( 'wp_ajax_timefix_load_services_by_category', [ __CLASS__, 'load_services_by_category' ] );
		add_action( 'wp_ajax_nopriv_timefix_load_services_by_category', [ __CLASS__, 'load_services_by_category' ] );
		add_action( 'wp_ajax_timefix_load_staffs_by_service', [ __CLASS__, 'load_staffs_by_service' ] );
		add_action( 'wp_ajax_nopriv_timefix_load_staffs_by_service', [ __CLASS__, 'load_staffs_by_service' ] );
		add_action( 'wp_ajax_timefix_appointment_wc_checkout', [ __CLASS__, 'appointment_wc_checkout' ] );
		add_action( 'wp_ajax_nopriv_timefix_appointment_wc_checkout', [ __CLASS__, 'appointment_wc_checkout' ] );
	}

	public static function appointment_wc_checkout() {
		$msg          = esc_html__( 'Something wrong!', 'timefix' );
		$success      = false;
		$redirect_url = '';

		if ( apply_filters( 'timefix_appointment_form_remove_nonce', true ) || Functions::verify_nonce() ) {
			$service_id = isset( $_POST['service_id'] ) ? absint( $_POST['service_id'] ) : 0;
			$service    = wpm_timefix()->factory->get_service( $service_id );

			if ( is_object( $service ) ) {
				$staff_id      = isset( $_POST['staff_id'] ) ? absint( $_POST['staff_id'] ) : '';
				$booking_date  = isset( $_POST['booking_date'] ) ? sanitize_text_field( $_POST['booking_date'] ) : '';
				$time_slot     = isset( $_POST['booking_time'] ) ? sanitize_text_field( $_POST['booking_time'] ) : '';
				$service_price = 200;

				$booking_data = [
					'service_id' => $service_id,
					'staff_id'   => $staff_id,
					'date'       => $booking_date,
					'time'       => $time_slot,
				];

				$product_id = $service->get_id();

				WC()->cart->empty_cart();

				WC()->cart->add_to_cart( $product_id, 1, 0, [], [
					'wpm_timefix_booking_data'  => $booking_data,
					'wpm_timefix_service_price' => $service_price,
				] );

				$redirect_url = wc_get_checkout_url();
				$success      = true;
			} else {
				$msg = apply_filters( 'timefix_appointment_error_message', esc_html__( "Service not found!", "timefix" ), $_REQUEST );
			}
		} else {
			$msg = apply_filters( 'timefix_appointment_session_error_message', esc_html__( "Session Error!", "timefix" ), $_REQUEST );
		}

		$response = [
			'success'  => $success,
			'message'  => $msg,
			'redirect' => $redirect_url
		];

		wp_send_json( $response );
	}

	public static function load_staffs_by_service() {
		if ( ! Functions::verify_nonce() ) {
			return false;
		}

		$service_id = absint( $_POST['service_id'] );

		if ( ! $service_id ) {
			wp_send_json_error( esc_html__( 'Service not found!', 'timefix' ) );
		}

		$staff_ids = StaffHandler::get_staffs_by_service_id( $service_id );

		$options = [];

		foreach ( $staff_ids as $staff_id ) {
			$options[ $staff_id ] = get_the_title( $staff_id );
		}

		wp_send_json_success( $options );
	}

	public static function load_services_by_category() {
		if ( ! Functions::verify_nonce() ) {
			return false;
		}

		$category_id = absint( $_POST['category_id'] );

		$options = ServiceHandler::get_services_by_category_id( $category_id );

		wp_send_json_success( $options );
	}

	public static function get_time_slots() {
		if ( ! Functions::verify_nonce() ) {
			return false;
		}

		$service_id = isset( $_POST['service_id'] ) ? absint( $_POST['service_id'] ) : '';

		if ( ! $service_id ) {
			wp_send_json_error( esc_html__( 'Service not found!', 'timefix' ) );
		}

		if ( wpm_timefix()->service_post_type !== get_post_type( $service_id ) ) {
			wp_send_json_error( esc_html__( 'Wrong service id!', 'timefix' ) );
		}

		$staff_id = isset( $_POST['staff_id'] ) ? absint( $_POST['staff_id'] ) : '';

		if ( ! $staff_id ) {
			wp_send_json_error( esc_html__( 'Staff not found!', 'timefix' ) );
		}

		if ( wpm_timefix()->staff_post_type !== get_post_type( $staff_id ) ) {
			wp_send_json_error( esc_html__( 'Wrong staff id!', 'timefix' ) );
		}

		$booking_date = isset( $_POST['booking_date'] ) ? sanitize_text_field( $_POST['booking_date'] ) : '';

		if ( empty( $booking_date ) ) {
			wp_send_json_error( esc_html__( 'Booking date empty!', 'timefix' ) );
		}

		if ( strtotime( $booking_date ) < strtotime( "now" ) ) {
			wp_send_json_error( esc_html__( 'Older date not allowed!', 'timefix' ) );
		}

		$service = wpm_timefix()->factory->get_service( $service_id );

		if ( ! $service ) {
			wp_send_json_error( esc_html__( 'Error to create service object!', 'timefix' ) );
		}

		$duration = $service->get_duration();

		if ( ! $duration ) {
			$duration = Functions::get_option_item( 'wpm_timefix_general_settings', 'duration', 15 );
		}

		$interval = $service->get_interval();

		if ( '' === $interval ) {
			$interval = Functions::get_option_item( 'wpm_timefix_general_settings', 'time_interval', 0 );
		}

		$time_slots = Functions::get_staff_time_slots( $staff_id, $booking_date, $duration, $interval );

		wp_send_json_success( $time_slots );
	}

	public static function save_appointment_data() {
		global $wpdb;
		$msg     = esc_html__( 'Something wrong!', 'timefix' );
		$success = false;

		if ( apply_filters( 'timefix_appointment_form_remove_nonce', true ) || Functions::verify_nonce() ) {
			$service_id = isset( $_POST['service_id'] ) ? absint( $_POST['service_id'] ) : 0;

			$service = wpm_timefix()->factory->get_service( $service_id );

			if ( is_object( $service ) ) {
				// Begin transaction
				$wpdb->query( 'START TRANSACTION' );

				try {
					$user_id = isset( $_POST['user_id'] ) ? absint( $_POST['user_id'] ) : '';

					$customer_name  = isset( $_POST['customer_name'] ) ? sanitize_text_field( $_POST['customer_name'] ) : '';
					$customer_email = isset( $_POST['customer_email'] ) ? sanitize_email( $_POST['customer_email'] ) : '';
					$phone          = isset( $_POST['customer_phone'] ) ? sanitize_text_field( $_POST['customer_phone'] ) : '';

					$customer_id = Functions::get_or_create_customer( $user_id, $customer_name, $customer_email, $phone );

					if ( ! $customer_id ) {
						throw new \Exception( __( 'Failed to insert customer!', 'timefix' ) );
					}
					// Insert appointment data
					$appointments_table = $wpdb->prefix . "wpm_timefix_appointments";

					$additional_note = isset( $_POST['customer_note'] ) ? sanitize_textarea_field( $_POST['customer_note'] ) : '';

					$appointment_data = [
						'customer_id'      => $customer_id,
						'status'           => 'pending',
						'additional_notes' => $additional_note
					];

					$wpdb->insert( $appointments_table, $appointment_data );

					$appointment_id = $wpdb->insert_id;

					if ( ! $appointment_id ) {
						throw new \Exception( __( 'Failed to insert appointment!', 'timefix' ) );
					}
					// Insert appointment services data
					$appointment_services_table = $wpdb->prefix . "wpm_timefix_appointment_services";

					$staff_id      = isset( $_POST['staff_id'] ) ? absint( $_POST['staff_id'] ) : '';
					$booking_date  = isset( $_POST['booking_date'] ) ? sanitize_text_field( $_POST['booking_date'] ) : '';
					$time_slot     = isset( $_POST['booking_time'] ) ? sanitize_text_field( $_POST['booking_time'] ) : '';
					$time_slot_arr = explode( '-', $time_slot );
					$start_time    = isset( $time_slot_arr[0] ) ? trim( $time_slot_arr[0] ) : '';
					$end_time      = isset( $time_slot_arr[1] ) ? trim( $time_slot_arr[1] ) : '';
					$service_price = $service->get_price();

					$appointment_services_data = [
						'appointment_id'   => $appointment_id,
						'service_id'       => $service_id,
						'staff_id'         => $staff_id,
						'appointment_date' => date( 'Y-m-d', strtotime( $booking_date ) ),
						'start_time'       => $start_time,
						'end_time'         => $end_time,
						'price'            => $service_price
					];

					$wpdb->insert( $appointment_services_table, $appointment_services_data );

					if ( ! $wpdb->insert_id ) {
						throw new \Exception( __( 'Failed to insert appointment services!', 'timefix' ) );
					}
					// Insert appointment order
					$orders_table = $wpdb->prefix . "wpm_timefix_orders";

					$orders_table_data = [
						'object_id'      => $appointment_id,
						'object_type'    => 'appointment',
						'customer_id'    => $customer_id,
						'subtotal'       => $service_price,
						'total'          => $service_price,
						'payment_method' => 'offline',
						'payment_status' => 'unpaid'
					];

					$wpdb->insert( $orders_table, $orders_table_data );

					if ( ! $wpdb->insert_id ) {
						throw new \Exception( __( 'Failed to insert orders!', 'timefix' ) );
					}

					// If everything is successful, commit the transaction
					$wpdb->query( 'COMMIT' );
					$success = true;
					$msg     = apply_filters( 'timefix_appointment_success_message', esc_html__( "Thank you for choosing our service. You will receive a confirmation email shortly.", "timefix" ), $_REQUEST );
					do_action( 'wpm_timefix_appointment_submission_success', $appointment_id, [
						'customer_name'    => $customer_name,
						'customer_email'   => $customer_email,
						'appointment_date' => $booking_date,
						'appointment_time' => $start_time . '-' . $end_time,
						'service_id'       => $service_id,
						'service_name'     => $service->get_the_title(),
						'staff_id'         => $staff_id
					] );
				} catch ( \Exception $e ) {
					// Rollback on error
					$wpdb->query( 'ROLLBACK' );
					$msg = sprintf( esc_html__( "Error to complete appointment! %s", "timefix" ), $e->getMessage() );
				}
			} else {
				$msg = apply_filters( 'timefix_appointment_error_message', esc_html__( "Service not found!", "timefix" ), $_REQUEST );
			}
		} else {
			$msg = apply_filters( 'timefix_appointment_session_error_message', esc_html__( "Session Error!", "timefix" ), $_REQUEST );
		}

		$response = [
			'success'  => $success,
			'message'  => $msg,
			'title'    => esc_html__( 'Your appointment has been successfully booked!', 'timefix' ),
			'btn_text' => esc_html__( 'Add New Appointment', 'timefix' )
		];

		wp_send_json( $response );
	}
}