<?php

namespace WPMinimize\TimeFix\Emails;

use WPMinimize\TimeFix\Helpers\Functions;

abstract class EmailBase {
	protected $to;
	protected $cc = [];
	protected $bcc = [];
	protected $reply_to = '';
	protected $attachments = [];
	protected $subject;
	protected $heading;
	protected $placeholders = [];
	protected $template_slug = '';

	public function __construct( $placeholders = [] ) {
		$this->placeholders  = $this->get_template_data( $placeholders );
		$this->template_slug = $this->get_template_name();
		$this->to            = $this->get_default_recipient( $this->placeholders );

		if ( empty( $this->to ) ) {
			$this->to = get_option( 'admin_email' );
		}
	}

	public function set_recipient( $email ) {
		$this->to = sanitize_email( $email );
	}

	public function set_cc( $cc ) {
		if ( is_array( $cc ) ) {
			$this->cc = array_map( 'sanitize_email', $cc );
		} else {
			$this->cc[] = sanitize_email( $cc );
		}
	}

	public function set_bcc( $bcc ) {
		$this->bcc = is_array( $bcc ) ? array_map( 'sanitize_email', $bcc ) : [ sanitize_email( $bcc ) ];
	}

	public function set_reply_to( $email ) {
		$this->reply_to = sanitize_email( $email );
	}

	public function add_attachment( $file_path ) {
		if ( file_exists( $file_path ) ) {
			$this->attachments[] = $file_path;
		}
	}

	public function set_placeholders( array $placeholders ) {
		$this->placeholders = $this->get_template_data( $placeholders );
	}

	public function send() {
		do_action( 'wpm_timefix_before_email_send', $this );

		$subject = $this->replace_placeholders( $this->get_default_subject() );
		$subject = apply_filters( 'wpm_timefix_email_subject', $subject, $this );

		$heading = $this->replace_placeholders( $this->get_default_heading() );

		$headers = [ 'Content-Type: text/html; charset=UTF-8' ];

		if ( ! empty( $this->cc ) ) {
			$headers[] = 'Cc: ' . implode( ',', $this->cc );
		}

		if ( ! empty( $this->bcc ) ) {
			$headers[] = 'Bcc: ' . implode( ',', $this->bcc );
		}

		if ( ! empty( $this->reply_to ) ) {
			$headers[] = 'Reply-To: ' . $this->reply_to;
		}

		$body = $this->get_template_body( $heading );

		$result = wp_mail( $this->to, $subject, $body, $headers, $this->attachments );

		if ( ! $result ) {
			do_action( 'wpm_timefix_email_log', [
				'email'    => $this,
				'to'       => $this->to,
				'subject'  => $subject,
				'headers'  => $headers,
				'body'     => $body,
				'template' => $this->template_slug,
			] );
		}

		do_action( 'wpm_timefix_after_email_send', $this );

		return $result;
	}

	protected function replace_placeholders( $text ) {
		foreach ( $this->placeholders as $key => $value ) {
			if ( is_array( $value ) || is_object( $value ) ) {
				$value = '';
			}
			$text = str_replace( '{' . $key . '}', $value, $text );
		}

		return $text;
	}

	protected function get_template_body( $heading ) {
		ob_start();

		$data = array_merge( $this->placeholders, [
			'email_heading' => $heading,
		] );

		Functions::get_template( $this->template_slug, $data );

		return ob_get_clean();
	}

	// Methods child class must implement
	abstract public function get_id();

	abstract public function get_default_subject();

	abstract public function get_default_heading();

	abstract public function get_template_name();

	abstract public function get_template_data( $data );

	abstract public function get_default_recipient( $data );
}
