<?php

namespace WPMinimize\TimeFix\Api\V1;

defined( 'ABSPATH' ) || exit;

use WP_Error;
use WP_REST_Request;
use WP_REST_Server;
use WPMinimize\TimeFix\Helpers\Functions;

class SettingsApi {

	public function register_routes() {
		register_rest_route( 'wpm-timefix/v1', 'general-settings', [
			[
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => [ $this, 'get_general_settings' ],
				'permission_callback' => [ $this, 'permission_check' ]
			],
			[
				'methods'             => WP_REST_Server::EDITABLE,
				'callback'            => [ $this, 'update_general_settings' ],
				'permission_callback' => [ $this, 'permission_check' ],
			]
		] );
		register_rest_route( 'wpm-timefix/v1', 'permalink-settings', [
			[
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => [ $this, 'get_permalink_settings' ],
				'permission_callback' => [ $this, 'permission_check' ]
			],
			[
				'methods'             => WP_REST_Server::EDITABLE,
				'callback'            => [ $this, 'update_permalink_settings' ],
				'permission_callback' => [ $this, 'permission_check' ],
			]
		] );
		register_rest_route( 'wpm-timefix/v1', 'payment-settings', [
			[
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => [ $this, 'get_payment_settings' ],
				'permission_callback' => [ $this, 'permission_check' ]
			],
			[
				'methods'             => WP_REST_Server::EDITABLE,
				'callback'            => [ $this, 'update_payment_settings' ],
				'permission_callback' => [ $this, 'permission_check' ],
			]
		] );
		register_rest_route( 'wpm-timefix/v1', 'offline-payment-settings', [
			[
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => [ $this, 'get_offline_payment_settings' ],
				'permission_callback' => [ $this, 'permission_check' ]
			],
			[
				'methods'             => WP_REST_Server::EDITABLE,
				'callback'            => [ $this, 'update_offline_payment_settings' ],
				'permission_callback' => [ $this, 'permission_check' ],
			]
		] );
		register_rest_route( 'wpm-timefix/v1', 'email-settings', [
			[
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => [ $this, 'get_email_settings' ],
				'permission_callback' => [ $this, 'permission_check' ]
			],
			[
				'methods'             => WP_REST_Server::EDITABLE,
				'callback'            => [ $this, 'update_email_settings' ],
				'permission_callback' => [ $this, 'permission_check' ],
			]
		] );
	}

	public function get_general_settings() {
		$settings = get_option( 'wpm_timefix_general_settings' );

		wp_send_json( $settings );
	}

	public function get_permalink_settings() {
		$settings = get_option( 'wpm_timefix_permalink_settings' );

		wp_send_json( $settings );
	}

	public function get_payment_settings() {
		$settings = get_option( 'wpm_timefix_payment_settings' );

		wp_send_json( $settings );
	}

	public function get_offline_payment_settings() {
		$settings = get_option( 'wpm_timefix_offline_payment_settings' );

		wp_send_json( $settings );
	}

	public function get_email_settings() {
		$settings = get_option( 'wpm_timefix_email_settings' );

		wp_send_json( $settings );
	}

	public function update_general_settings( $request ) {

		$nonce = $request->get_header( 'X-WP-Nonce' );

		if ( ! wp_verify_nonce( $nonce, 'wp_rest' ) ) {
			return new WP_Error( 'rest_forbidden', __( 'Invalid nonce.', 'timefix' ), array( 'status' => 403 ) );
		}

		if ( ! $request->get_param( 'settings' ) ) {
			$response = [
				'status'        => "error",
				'error'         => 'BADREQUEST',
				'code'          => '400',
				'error_message' => esc_html__( 'Settings data not found.', "timefix" )
			];
			wp_send_json( $response, 400 );
		}

		$success = false;
		$message = esc_html__( 'Something wrong!', 'timefix' );

		$options = $request->get_param( 'settings' );

		if ( is_array( $options ) ) {
			update_option( 'wpm_timefix_general_settings', $options );
			$success = true;
			$message = esc_html__( 'Updated general settings successfully.', 'timefix' );
		}

		$response = [
			'success' => $success,
			'msg'     => $message
		];

		return rest_ensure_response( $response );
	}

	public function update_permalink_settings( $request ) {

		$nonce = $request->get_header( 'X-WP-Nonce' );

		if ( ! wp_verify_nonce( $nonce, 'wp_rest' ) ) {
			return new WP_Error( 'rest_forbidden', __( 'Invalid nonce.', 'timefix' ), array( 'status' => 403 ) );
		}

		if ( ! $request->get_param( 'settings' ) ) {
			$response = [
				'status'        => "error",
				'error'         => 'BADREQUEST',
				'code'          => '400',
				'error_message' => esc_html__( 'Settings data not found.', "timefix" )
			];
			wp_send_json( $response, 400 );
		}

		$success = false;
		$message = esc_html__( 'Something wrong!', 'timefix' );

		$options = $request->get_param( 'settings' );

		$sanitized_options = [];

		$sanitized_options['service_single_base']  = ! empty( $options['singleServiceBase'] ) ? sanitize_text_field( $options['singleServiceBase'] ) : 'timefix_service';
		$sanitized_options['service_archive_base'] = ! empty( $options['archiveServiceBase'] ) ? sanitize_text_field( $options['archiveServiceBase'] ) : 'timefix_services';
		$sanitized_options['staff_single_base']    = ! empty( $options['singleStaffBase'] ) ? sanitize_text_field( $options['singleStaffBase'] ) : 'timefix_staff';
		$sanitized_options['staff_archive_base']   = ! empty( $options['archiveStaffBase'] ) ? sanitize_text_field( $options['archiveStaffBase'] ) : 'timefix_staffs';

		if ( is_array( $sanitized_options ) ) {
			update_option( 'wpm_timefix_permalink_settings', $sanitized_options );
			Functions::update_permalink();
			$success = true;
			$message = esc_html__( 'Updated permalink settings successfully.', 'timefix' );
		}

		$response = [
			'success' => $success,
			'msg'     => $message
		];

		return rest_ensure_response( $response );
	}

	public function update_payment_settings( $request ) {

		$nonce = $request->get_header( 'X-WP-Nonce' );

		if ( ! wp_verify_nonce( $nonce, 'wp_rest' ) ) {
			return new WP_Error( 'rest_forbidden', __( 'Invalid nonce.', 'timefix' ), array( 'status' => 403 ) );
		}

		if ( ! $request->get_param( 'settings' ) ) {
			$response = [
				'status'        => "error",
				'error'         => 'BADREQUEST',
				'code'          => '400',
				'error_message' => esc_html__( 'Settings data not found.', "timefix" )
			];
			wp_send_json( $response, 400 );
		}

		$success = false;
		$message = esc_html__( 'Something wrong!', 'timefix' );

		$options = $request->get_param( 'settings' );

		if ( is_array( $options ) ) {
			update_option( 'wpm_timefix_payment_settings', $options );
			$success = true;
			$message = esc_html__( 'Updated payment settings successfully.', 'timefix' );
		}

		$response = [
			'success' => $success,
			'msg'     => $message
		];

		return rest_ensure_response( $response );
	}

	public function update_offline_payment_settings( $request ) {
		$nonce = $request->get_header( 'X-WP-Nonce' );

		if ( ! wp_verify_nonce( $nonce, 'wp_rest' ) ) {
			return new WP_Error( 'rest_forbidden', __( 'Invalid nonce.', 'timefix' ), array( 'status' => 403 ) );
		}

		if ( ! $request->get_param( 'settings' ) ) {
			$response = [
				'status'        => "error",
				'error'         => 'BADREQUEST',
				'code'          => '400',
				'error_message' => esc_html__( 'Settings data not found.', "timefix" )
			];
			wp_send_json( $response, 400 );
		}

		$success = false;
		$message = esc_html__( 'Something wrong!', 'timefix' );

		$options = $request->get_param( 'settings' );

		if ( is_array( $options ) ) {
			update_option( 'wpm_timefix_offline_payment_settings', $options );
			$success = true;
			$message = esc_html__( 'Updated pay on-site settings successfully.', 'timefix' );
		}

		$response = [
			'success' => $success,
			'msg'     => $message
		];

		return rest_ensure_response( $response );
	}

	public function update_email_settings( $request ) {

		$nonce = $request->get_header( 'X-WP-Nonce' );

		if ( ! wp_verify_nonce( $nonce, 'wp_rest' ) ) {
			return new WP_Error( 'rest_forbidden', __( 'Invalid nonce.', 'timefix' ), array( 'status' => 403 ) );
		}

		if ( ! $request->get_param( 'settings' ) ) {
			$response = [
				'status'        => "error",
				'error'         => 'BADREQUEST',
				'code'          => '400',
				'error_message' => esc_html__( 'Settings data not found.', "timefix" )
			];
			wp_send_json( $response, 400 );
		}

		$success = false;
		$message = esc_html__( 'Something wrong!', 'timefix' );

		$options = $request->get_param( 'settings' );

		if ( is_array( $options ) ) {
			update_option( 'wpm_timefix_email_settings', $options );
			$success = true;
			$message = esc_html__( 'Updated email settings successfully.', 'timefix' );
		}

		$response = [
			'success' => $success,
			'msg'     => $message
		];

		return rest_ensure_response( $response );
	}

	/**
	 * @param WP_REST_Request $request
	 *
	 * @return bool
	 */
	public function permission_check( WP_REST_Request $request ) {

		return true;

		if ( ! is_user_logged_in() ) {
			return false;
		}

		$current_user = wp_get_current_user();

		if ( ! in_array( 'administrator', $current_user->roles ) ) {
			return false;
		}

		return true;
	}
}