<?php

namespace WPMinimize\TimeFix\Api\V1;

use WP_REST_Request;
use WPMinimize\TimeFix\Helpers\Functions;

defined( 'ABSPATH' ) || exit;

class PaymentsApi {

	public function register_routes() {
		register_rest_route( 'wpm-timefix/v1', 'payments', [
			'methods'             => 'GET',
			'callback'            => [ $this, 'get_payments_data' ],
			'permission_callback' => [ $this, 'permission_check' ]
		] );
		register_rest_route( 'wpm-timefix/v1', 'payment/(?P<id>\d+)', [
			'methods'             => 'GET',
			'callback'            => [ $this, 'get_single_payment' ],
			'permission_callback' => [ $this, 'permission_check' ],
			'args'                => [
				'id' => [
					'validate_callback' => function ( $param ) {
						return is_numeric( $param );
					}
				]
			]
		] );
		register_rest_route( 'wpm-timefix/v1', 'payment/(?P<id>\d+)', [
			'methods'             => 'POST',
			'callback'            => [ $this, 'update_payment_status' ],
			'permission_callback' => [ $this, 'permission_check' ],
			'args'                => [
				'id' => [
					'validate_callback' => function ( $param ) {
						return is_numeric( $param );
					}
				]
			]
		] );
	}

	public function get_single_payment( WP_REST_Request $request ) {
		global $wpdb;

		$id = (int) $request['id'];

		$orders_table             = "{$wpdb->prefix}wpm_timefix_orders";
		$appointments_table       = "{$wpdb->prefix}wpm_timefix_appointments";
		$appointment_services_tbl = "{$wpdb->prefix}wpm_timefix_appointment_services";
		$customers_table          = "{$wpdb->prefix}wpm_timefix_customers";
		$wp_posts_table           = $wpdb->posts;

		// Fetch the order
		$row = $wpdb->get_row( $wpdb->prepare( "
        SELECT o.id AS order_id, o.created_at AS date, o.payment_status AS order_status,
               o.subtotal, o.tax_total, o.total, o.transaction_reference, o.payment_method,
               o.info AS order_info, o.woo_order_id,
               a.id AS appointment_id, a.total_price, a.status AS appointment_status,
               c.id AS customer_id, c.wp_user_id, c.name AS customer_name, c.email, c.phone
        FROM $orders_table o
        LEFT JOIN $appointments_table a
            ON a.id = o.object_id AND o.object_type = 'appointment'
        LEFT JOIN $customers_table c
            ON c.id = o.customer_id
        WHERE o.id = %d
        LIMIT 1
    ", $id ) );

		if ( ! $row ) {
			return new \WP_Error( 'order_not_found', __( 'Payment not found', 'timefix' ), [ 'status' => 404 ] );
		}

		// Get appointment service items with names from wp_posts
		$items = $wpdb->get_results( $wpdb->prepare( "
        SELECT asv.service_id, asv.staff_id, asv.price, asv.appointment_date, asv.start_time, asv.end_time,
               s.post_title AS service_name,
               st.post_title AS staff_name
        FROM $appointment_services_tbl asv
        LEFT JOIN $wp_posts_table s
            ON s.ID = asv.service_id
        LEFT JOIN $wp_posts_table st
            ON st.ID = asv.staff_id
        WHERE asv.appointment_id = %d
    ", $row->appointment_id ) );

		$item_list = [];
		foreach ( $items as $item ) {
			$item_list[] = [
				'sku'         => 'SKU' . str_pad( $item->service_id, 3, '0', STR_PAD_LEFT ),
				'serviceId'   => $item->service_id,
				'serviceName' => $item->service_name,
				'staffId'     => $item->staff_id,
				'staffName'   => $item->staff_name,
				'serviceDate' => $item->appointment_date,
				'startTime'   => $item->start_time,
				'endTime'     => $item->end_time,
				'qty'         => 1,
				'price'       => (float) $item->price,
			];
		}

		// Build single order response
		$payment = [
			'id'             => (int) $row->order_id,
			'date'           => $row->date,
			'status'         => $row->order_status,
			'subTotal'       => (float) $row->subtotal,
			'taxAmount'      => (float) $row->tax_total,
			'total'          => (float) $row->total,
			'transactionRef' => $row->transaction_reference,
			'wooOrderId'     => $row->woo_order_id,
			'method'         => $row->payment_method,
			'currencySymbol' => '$',
			'info'           => $row->order_info,
			'appointment'    => [
				'id'         => (int) $row->appointment_id,
				'totalPrice' => (float) $row->total_price,
				'status'     => $row->appointment_status
			],
			'customer'       => [
				'id'         => (int) $row->customer_id,
				'wp_user_id' => (int) $row->wp_user_id,
				'name'       => $row->customer_name,
				'email'      => $row->email,
				'phone'      => $row->phone
			],
			'items'          => $item_list
		];

		return rest_ensure_response( $payment );
	}

	public function get_payments_data( WP_REST_Request $request ) {
		global $wpdb;

		$orders_table             = "{$wpdb->prefix}wpm_timefix_orders";
		$appointments_table       = "{$wpdb->prefix}wpm_timefix_appointments";
		$appointment_services_tbl = "{$wpdb->prefix}wpm_timefix_appointment_services";
		$customers_table          = "{$wpdb->prefix}wpm_timefix_customers";
		$wp_posts_table           = $wpdb->posts;

		// Pagination parameters
		$page     = max( 1, intval( $request->get_param( 'page' ) ) );
		$per_page = max( 1, intval( $request->get_param( 'per_page' ) ?: 10 ) );
		$offset   = ( $page - 1 ) * $per_page;

		// Filter parameters
		$status      = sanitize_text_field( $request->get_param( 'status' ) );
		$customer_id = intval( $request->get_param( 'customer_id' ) );
		$service_id  = intval( $request->get_param( 'service_id' ) );
		$date_from   = sanitize_text_field( $request->get_param( 'date_from' ) );
		$date_to     = sanitize_text_field( $request->get_param( 'date_to' ) );

		// WHERE clauses
		$where  = "WHERE 1=1";
		$params = [];

		if ( ! empty( $status ) ) {
			$where    .= " AND o.payment_status = %s";
			$params[] = $status;
		}

		if ( $customer_id ) {
			$where    .= " AND o.customer_id = %d";
			$params[] = $customer_id;
		}

		if ( $service_id ) {
			$where    .= " AND EXISTS (
            SELECT 1 
            FROM $appointment_services_tbl asv 
            WHERE asv.appointment_id = a.id 
              AND asv.service_id = %d
        )";
			$params[] = $service_id;
		}

		if ( ! empty( $date_from ) && ! empty( $date_to ) ) {
			$date_from = date( 'Y-m-d', strtotime( $date_from ) );
			$date_to   = date( 'Y-m-d', strtotime( $date_to ) );

			$where    .= " AND DATE(o.created_at) BETWEEN %s AND %s";
			$params[] = $date_from;
			$params[] = $date_to;
		}

		// Count query for pagination
		$count_sql = "
        SELECT COUNT(*)
        FROM $orders_table o
        LEFT JOIN $appointments_table a
            ON a.id = o.object_id
        LEFT JOIN $customers_table c
            ON c.id = o.customer_id
        $where
    ";

		$total_items = $wpdb->get_var( $wpdb->prepare( $count_sql, $params ) );

		// Get orders linked to appointments
		$sql = "
        SELECT o.id AS order_id, o.created_at AS date, o.payment_status AS order_status,
               o.subtotal, o.tax_total, o.total, o.transaction_reference, o.payment_method,
               o.info AS order_info,
               a.id AS appointment_id, a.total_price, a.status AS appointment_status,
               c.id AS customer_id, c.wp_user_id, c.name AS customer_name, c.email, c.phone
        FROM $orders_table o
        LEFT JOIN $appointments_table a
            ON a.id = o.object_id AND o.object_type = 'appointment'
        LEFT JOIN $customers_table c
            ON c.id = o.customer_id
        $where
        ORDER BY o.created_at DESC
        LIMIT %d OFFSET %d
    ";

		$query_params = array_merge( $params, [ $per_page, $offset ] );
		$results      = $wpdb->get_results( $wpdb->prepare( $sql, $query_params ) );

		$payments = [];

		foreach ( $results as $row ) {
			// Get appointment service items with names from wp_posts
			$items = $wpdb->get_results( $wpdb->prepare( "
            SELECT asv.service_id, asv.staff_id, asv.price,
                   s.post_title AS service_name,
                   st.post_title AS staff_name
            FROM $appointment_services_tbl asv
            LEFT JOIN $wp_posts_table s
                ON s.ID = asv.service_id
            LEFT JOIN $wp_posts_table st
                ON st.ID = asv.staff_id
            WHERE asv.appointment_id = %d
        ", $row->appointment_id ) );

			$item_list = [];
			foreach ( $items as $item ) {
				$item_list[] = [
					'sku'         => 'SKU' . str_pad( $item->service_id, 3, '0', STR_PAD_LEFT ),
					'serviceId'   => $item->service_id,
					'serviceName' => $item->service_name,
					'staffId'     => $item->staff_id,
					'staffName'   => $item->staff_name,
					'qty'         => 1,
					'price'       => (float) $item->price,
				];
			}

			// Build response object
			$payments[] = [
				'id'             => (int) $row->order_id,
				'date'           => $row->date,
				'status'         => $row->order_status,
				'subTotal'       => (float) $row->subtotal,
				'taxAmount'      => (float) $row->tax_total,
				'total'          => (float) $row->total,
				'transactionRef' => $row->transaction_reference,
				'method'         => $row->payment_method,
				'currencySymbol' => '$', // get it based on settings
				'info'           => $row->order_info,
				'appointment'    => [
					'id'         => (int) $row->appointment_id,
					'totalPrice' => (float) $row->total_price,
					'status'     => $row->appointment_status
				],
				'customer'       => [
					'id'         => (int) $row->customer_id,
					'wp_user_id' => (int) $row->wp_user_id,
					'name'       => $row->customer_name,
					'email'      => $row->email,
					'phone'      => $row->phone
				],
				'items'          => $item_list
			];
		}

		return rest_ensure_response( [
			'data'  => $payments,
			'total' => intval( $total_items ),
		] );
	}

	public function update_payment_status( WP_REST_Request $request ) {
		global $wpdb;

		$id     = (int) $request['id'];
		$status = sanitize_text_field( $request->get_param( 'status' ) );

		$orders_table       = "{$wpdb->prefix}wpm_timefix_orders";
		$appointments_table = "{$wpdb->prefix}wpm_timefix_appointments";

		// Get order and linked appointment
		$order = $wpdb->get_row( $wpdb->prepare( "
        SELECT o.id AS order_id, o.object_id, o.object_type
        FROM $orders_table o
        WHERE o.id = %d
        LIMIT 1
    ", $id ) );

		if ( ! $order ) {
			return new \WP_Error( 'order_not_found', esc_html__( 'Payment not found', 'timefix' ), [ 'status' => 404 ] );
		}

		// Update the order status
		$wpdb->update(
			$orders_table,
			[ 'payment_status' => $status ],
			[ 'id' => $order->order_id ],
			[ '%s' ],
			[ '%d' ]
		);

		// If linked to an appointment, update appointment status
		if ( $order->object_type === 'appointment' && $order->object_id ) {
			// Map order status to appointment status
			$appointment_status = $this->map_order_status_to_appointment_status( $status );

			if ( $appointment_status ) {
				$wpdb->update(
					$appointments_table,
					[ 'status' => $appointment_status ],
					[ 'id' => $order->object_id ],
					[ '%s' ],
					[ '%d' ]
				);
			}
		}

		return rest_ensure_response( [
			'success' => true,
			'message' => esc_html__( 'Order status updated successfully', 'timefix' ),
		] );
	}

	/**
	 * Map order status to appointment status
	 */
	private function map_order_status_to_appointment_status( $order_status ) {
		$map = [
			'paid'           => 'confirmed',
			'partially_paid' => 'on_hold',
			'unpaid'         => 'pending',
			'pending'        => 'pending',
			'failed'         => 'on_hold',
			'cancelled'      => 'cancelled',
			'completed'      => 'completed',
		];

		return $map[ $order_status ] ?? null;
	}

	/**
	 * @param WP_REST_Request $request
	 *
	 * @return bool
	 */
	public function permission_check( WP_REST_Request $request ) {

		$nonce = $request->get_header( 'X-WP-Nonce' );

		if ( ! $nonce ) {
			return false;
		}

		if ( ! is_user_logged_in() ) {
			return false;
		}

		$current_user = wp_get_current_user();

		if ( ! in_array( 'administrator', $current_user->roles ) ) {
			return false;
		}

		return true;
	}

}