<?php

namespace WPMinimize\TimeFix\Api\V1;

use WP_Error;
use WP_REST_Request;

defined( 'ABSPATH' ) || exit;

class CustomersApi {

	private $table_name;

	public function __construct() {
		global $wpdb;
		$this->table_name = $wpdb->prefix . 'wpm_timefix_customers';
	}

	public function register_routes() {
		register_rest_route( 'wpm-timefix/v1', 'customers', [
			'methods'             => 'GET',
			'callback'            => [ $this, 'get_customers' ],
			'permission_callback' => [ $this, 'permission_check' ],
			'args'                => array(
				'page'     => array(
					'default'           => 1,
					'sanitize_callback' => 'absint',
				),
				'per_page' => array(
					'default'           => 4,
					'sanitize_callback' => 'absint',
				),
				'name'     => array(
					'default'           => '',
					'sanitize_callback' => 'sanitize_text_field',
				),
				'email'    => array(
					'default'           => '',
					'sanitize_callback' => 'sanitize_text_field',
				)
			)
		] );
		register_rest_route( 'wpm-timefix/v1', 'customers', array(
			'methods'             => 'POST',
			'callback'            => array( $this, 'create_customer' ),
			'permission_callback' => array( $this, 'permission_check' ),
			'args'                => array(
				'name'  => array(
					'required'          => true,
					'sanitize_callback' => 'sanitize_text_field',
				),
				'email' => array(
					'required'          => true,
					'sanitize_callback' => 'sanitize_email',
				),
				'phone' => array(
					'required'          => true,
					'sanitize_callback' => 'sanitize_text_field',
				),
			),
		) );
		register_rest_route( 'wpm-timefix/v1', '/customers/(?P<id>\d+)', array(
			'methods'             => 'PUT',
			'callback'            => array( $this, 'update_customer' ),
			'permission_callback' => array( $this, 'permission_check' ),
			'args'                => array(
				'name'  => array(
					'required'          => true,
					'sanitize_callback' => 'sanitize_text_field',
				),
				'email' => array(
					'required'          => true,
					'sanitize_callback' => 'sanitize_email',
				),
				'phone' => array(
					'required'          => true,
					'sanitize_callback' => 'sanitize_text_field',
				),
			),
		) );
		register_rest_route( 'wpm-timefix/v1', '/customers/(?P<id>\d+)', array(
			'methods'             => 'DELETE',
			'callback'            => array( $this, 'delete_customer' ),
			'permission_callback' => array( $this, 'permission_check' ),
		) );
	}

	public function delete_customer( $request ) {
		global $wpdb;

		$id = (int) $request['id'];

		// Check if customer exists
		$customer = $wpdb->get_row( $wpdb->prepare(
			"SELECT * FROM {$this->table_name} WHERE id = %d",
			$id
		), ARRAY_A );

		if ( ! $customer ) {
			return new WP_Error( 'customer_not_found', esc_html__( 'Customer not found', 'timefix' ), array( 'status' => 404 ) );
		}

		// Delete customer
		$result = $wpdb->delete(
			$this->table_name,
			array( 'id' => $id ),
			array( '%d' )
		);

		if ( $result === false ) {
			return new WP_Error( 'db_error', esc_html__( 'Failed to delete customer', 'timefix' ), array( 'status' => 500 ) );
		}

		return rest_ensure_response( array(
			'message' => esc_html__( 'Customer deleted successfully', 'timefix' ),
			'id'      => $id
		) );
	}

	public function update_customer( $request ) {
		global $wpdb;

		$id    = (int) $request['id'];
		$name  = $request['name'];
		$email = $request['email'];
		$phone = $request['phone'];

		// Check if customer exists
		$existing = $wpdb->get_row( $wpdb->prepare(
			"SELECT * FROM {$this->table_name} WHERE id = %d",
			$id
		), ARRAY_A );

		if ( ! $existing ) {
			return new WP_Error( 'customer_not_found', esc_html__( 'Customer not found', 'timefix' ), array( 'status' => 404 ) );
		}

		// Validate email
		if ( ! is_email( $email ) ) {
			return new WP_Error( 'invalid_email', esc_html__( 'Invalid email address', 'timefix' ), array( 'status' => 400 ) );
		}

		// Check if email exists for other customers
		$email_exists = $wpdb->get_var( $wpdb->prepare(
			"SELECT id FROM {$this->table_name} WHERE email = %s AND id != %d",
			$email, $id
		) );

		if ( $email_exists ) {
			return new WP_Error( 'email_exists', esc_html__( 'Email already exists', 'timefix' ), array( 'status' => 409 ) );
		}

		// Update customer
		$result = $wpdb->update(
			$this->table_name,
			array(
				'name'  => $name,
				'email' => $email,
				'phone' => $phone,
			),
			array( 'id' => $id ),
			array( '%s', '%s', '%s' ),
			array( '%d' )
		);

		if ( $result === false ) {
			return new WP_Error( 'db_error', esc_html__( 'Failed to update customer', 'timefix' ), array( 'status' => 500 ) );
		}

		// Update WordPress user if exists
		if ( $existing['wp_user_id'] ) {
			wp_update_user( array(
				'ID'           => $existing['wp_user_id'],
				'user_email'   => $email,
				'display_name' => $name,
				'first_name'   => explode( ' ', $name )[0],
				'last_name'    => isset( explode( ' ', $name )[1] ) ? explode( ' ', $name )[1] : '',
			) );
		}

		// Get updated customer
		$customer = $wpdb->get_row( $wpdb->prepare(
			"SELECT * FROM {$this->table_name} WHERE id = %d",
			$id
		), ARRAY_A );

		return rest_ensure_response( $customer );
	}

	public function create_customer( $request ) {
		global $wpdb;

		$name  = sanitize_text_field( $request['name'] );
		$email = sanitize_email( $request['email'] );
		$phone = sanitize_text_field( $request['phone'] );

		// Validate email
		if ( ! is_email( $email ) ) {
			return new WP_Error( 'invalid_email', 'Invalid email address', array( 'status' => 400 ) );
		}

		// Check if email already exists
		$existing = $wpdb->get_var( $wpdb->prepare(
			"SELECT id FROM {$this->table_name} WHERE email = %s",
			$email
		) );

		if ( $existing ) {
			return new WP_Error( 'email_exists', 'Email already exists', array( 'status' => 409 ) );
		}

		$wp_user_id = null;

		// Insert customer
		$result = $wpdb->insert(
			$this->table_name,
			array(
				'name'       => $name,
				'email'      => $email,
				'phone'      => $phone,
				'wp_user_id' => $wp_user_id,
			),
			array( '%s', '%s', '%s', '%d' )
		);

		if ( $result === false ) {
			return new WP_Error( 'db_error', 'Failed to create customer', array( 'status' => 500 ) );
		}

		$customer_id = $wpdb->insert_id;

		// Get the created customer
		$customer = $wpdb->get_row( $wpdb->prepare(
			"SELECT * FROM {$this->table_name} WHERE id = %d",
			$customer_id
		), ARRAY_A );

		return rest_ensure_response( $customer );
	}

	/**
	 * @param $request
	 *
	 * @return \WP_Error|\WP_HTTP_Response|\WP_REST_Response
	 */
	public function get_customers( $request ) {
		global $wpdb;

		$page     = $request['page'];
		$per_page = $request['per_page'];
		$offset   = ( $page - 1 ) * $per_page;

		$where_conditions = array( "1=1" );
		$where_values     = array();

		if ( ! empty( $request['name'] ) ) {
			$where_conditions[] = "name LIKE %s";
			$where_values[]     = '%' . $wpdb->esc_like( $request['name'] ) . '%';
		}

		if ( ! empty( $request['email'] ) ) {
			$where_conditions[] = "email LIKE %s";
			$where_values[]     = '%' . $wpdb->esc_like( $request['email'] ) . '%';
		}

		$where_clause = implode( ' AND ', $where_conditions );

		// Get total count for pagination
		$count_query = "SELECT COUNT(*) FROM {$this->table_name} WHERE {$where_clause}";
		if ( ! empty( $where_values ) ) {
			$count_query = $wpdb->prepare( $count_query, $where_values );
		}
		$total_items = $wpdb->get_var( $count_query );

		// Get customers
		$query        = "SELECT id, name, email, phone, wp_user_id, created_at FROM {$this->table_name} WHERE {$where_clause} ORDER BY created_at DESC LIMIT %d OFFSET %d";
		$query_values = array_merge( $where_values, array( $per_page, $offset ) );
		$customers    = $wpdb->get_results( $wpdb->prepare( $query, $query_values ), ARRAY_A );

		// Add WordPress user data if exists
		foreach ( $customers as &$customer ) {
			if ( $customer['wp_user_id'] ) {
				$wp_user                    = get_user_by( 'ID', $customer['wp_user_id'] );
				$customer['wordpress_user'] = $wp_user ? array(
					'id'           => $wp_user->ID,
					'username'     => $wp_user->user_login,
					'display_name' => $wp_user->display_name,
				) : null;
			} else {
				$customer['wordpress_user'] = null;
			}
		}

		$response = array(
			'customers'  => $customers,
			'pagination' => array(
				'total_items'  => (int) $total_items,
				'total_pages'  => ceil( $total_items / $per_page ),
				'current_page' => (int) $page,
				'per_page'     => (int) $per_page,
			),
		);

		return rest_ensure_response( $response );
	}

	/**
	 * @param WP_REST_Request $request
	 *
	 * @return bool
	 */
	public function permission_check( WP_REST_Request $request ) {

		$nonce = $request->get_header( 'X-WP-Nonce' );

		if ( ! $nonce ) {
			return false;
		}

		if ( ! is_user_logged_in() ) {
			return false;
		}

		$current_user = wp_get_current_user();

		if ( ! in_array( 'administrator', $current_user->roles ) ) {
			return false;
		}

		return true;
	}

}