<?php

namespace WPMinimize\TimeFix\Api\V1;

defined( 'ABSPATH' ) || exit;

use WP_Error;
use WP_REST_Request;
use WP_REST_Server;
use WPMinimize\TimeFix\Helpers\Functions;

class AppointmentsApi {

	public function register_routes() {
		register_rest_route( 'wpm-timefix/v1', 'appointments', [
			[
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => [ $this, 'get_appointments' ],
				'permission_callback' => [ $this, 'permission_check' ]
			]
		] );
		register_rest_route( 'wpm-timefix/v1', '/appointment/(?P<appointment_id>[\d]+)', [
			[
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => [ $this, 'get_single_appointment_info' ],
				'permission_callback' => [ $this, 'permission_check' ],
				'args'                => [
					'appointment_id' => [
						'required'    => true,
						'type'        => 'integer',
						'description' => esc_html__( 'Appointment id is required', 'timefix' ),
					]
				]
			],
			[
				'methods'             => WP_REST_Server::EDITABLE,
				'callback'            => [ $this, 'update_appointment' ],
				'permission_callback' => [ $this, 'permission_check' ],
				'args'                => [
					'appointment_id' => [
						'required'    => true,
						'type'        => 'integer',
						'description' => esc_html__( 'Appointment ID is required', 'timefix' ),
					]
				]
			],
			[
				'methods'             => WP_REST_Server::DELETABLE,
				'callback'            => [ $this, 'delete_appointment' ],
				'permission_callback' => [ $this, 'permission_check' ],
				'args'                => [
					'appointment_id' => [
						'required'    => true,
						'type'        => 'integer',
						'description' => esc_html__( 'Appointment ID is required', 'timefix' ),
					]
				]
			]
		] );
	}

	public function delete_appointment( WP_REST_Request $request ) {

		global $wpdb;

		if ( ! $request->get_param( 'appointment_id' ) ) {
			$response = [
				'status'        => "error",
				'error'         => 'BADREQUEST',
				'code'          => '400',
				'error_message' => esc_html__( 'Appointment ID not found.', "timefix" )
			];
			wp_send_json( $response, 400 );
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			$response = [
				'status'        => "error",
				'error'         => 'FORBIDDEN',
				'code'          => '403',
				'error_message' => "You are not permitted to delete."
			];
			wp_send_json( $response, 403 );
		}

		$success                = false;
		$message                = esc_html__( 'Something wrong!', 'timefix' );
		$appointment_id         = absint( $request->get_param( 'appointment_id' ) );
		$appointments_table     = $wpdb->prefix . "wpm_timefix_appointments";
		$appointment_meta_table = $wpdb->prefix . "wpm_timefix_appointment_meta";

		$wpdb->query( 'START TRANSACTION' );

		$meta_deleted        = $wpdb->delete( $appointment_meta_table, [ 'appointment_id' => $appointment_id ], [ '%d' ] );
		$appointment_deleted = $wpdb->delete( $appointments_table, [ 'id' => $appointment_id ], [ '%d' ] );

		if ( $meta_deleted !== false && $appointment_deleted !== false ) {
			$wpdb->query( 'COMMIT' );
			$success = true;
			$message = esc_html__( 'Appointment deleted successfully', 'timefix' );
		} else {
			$wpdb->query( 'ROLLBACK' );
		}

		$response = [
			'success' => $success,
			'msg'     => $message
		];

		return rest_ensure_response( $response );
	}

	public function get_appointments( WP_REST_Request $request ) {
		global $wpdb;

		$appointments_table         = $wpdb->prefix . "wpm_timefix_appointments";
		$appointment_services_table = $wpdb->prefix . "wpm_timefix_appointment_services";
		$customers_table            = $wpdb->prefix . "wpm_timefix_customers";
		$orders_table               = $wpdb->prefix . "wpm_timefix_orders";

		$page     = (int) $request->get_param( 'page' );
		$per_page = (int) $request->get_param( 'per_page' );
		$offset   = ( $page - 1 ) * $per_page;

		$status = $request->get_param( 'status' );

		$query = "SELECT 
            a.id AS appointment_id,
            a.status AS appointment_status,
            a.created_at AS appointment_created_at,
            s.appointment_date,
            s.start_time,
            s.end_time,
            s.service_id,
            s.staff_id,
            c.id AS customer_id,
            c.name,
            c.email,
            c.phone,
            o.payment_method,
            o.total AS order_total FROM {$appointments_table} a
                JOIN {$customers_table} c ON a.customer_id = c.id
                JOIN {$orders_table} o ON o.object_id = a.id
                JOIN {$appointment_services_table} s ON s.appointment_id = a.id";

		if ( ! empty( $status ) ) {
			$query .= " " . "WHERE a.status = '$status'";
		}

		$query .= " " . "GROUP BY a.id";

		$count_row = Functions::count_rows( $query );

		$query .= " " . "ORDER BY s.appointment_date DESC";

		$query .= " " . "LIMIT {$offset},{$per_page}";

		$results = $wpdb->get_results( $query );

		$response['max_posts'] = $count_row;

		if ( ! empty( $results ) ) {
			foreach ( $results as $appointment ) {
				$time_format = get_option( 'time_format' );

				$time = date( $time_format, strtotime( $appointment->start_time ) );

				if ( ! empty( $appointment->end_time ) ) {
					$time .= ' - ' . date( $time_format, strtotime( $appointment->end_time ) );
				}

				$info             = [ $appointment->name, $appointment->email, $appointment->phone ];
				$customer_details = implode( "\n", array_filter( $info ) );

				$response['posts'][] = array(
					'id'               => $appointment->appointment_id,
					'service_title'    => get_the_title( $appointment->service_id ),
					'staff_name'       => get_the_title( $appointment->staff_id ),
					'customer_id'      => $appointment->customer_id,
					'customer_info'    => $customer_details,
					'appointment_date' => \date( get_option( 'date_format' ), strtotime( $appointment->appointment_date ) ),
					'time'             => $time,
					'fee'              => Functions::get_price_html( $appointment->order_total ),
					'created_at'       => \date( 'F d, Y \a\t g:i a', strtotime( $appointment->appointment_created_at ) ),
					'status'           => Functions::get_appointment_status_title( $appointment->appointment_status )
				);
			}
		}

		return rest_ensure_response( $response );

	}

	public function get_single_appointment_info( WP_REST_Request $request ) {

		if ( ! $request->get_param( 'appointment_id' ) ) {
			$response = [
				'status'        => "error",
				'error'         => 'BADREQUEST',
				'code'          => '400',
				'error_message' => esc_html__( 'Appointment id not found.', "timefix" )
			];
			wp_send_json( $response, 400 );
		}

		global $wpdb;

		$appointment_id = absint( $request->get_param( 'appointment_id' ) );

		$appointments_table         = $wpdb->prefix . "wpm_timefix_appointments";
		$appointment_services_table = $wpdb->prefix . "wpm_timefix_appointment_services";
		$customers_table            = $wpdb->prefix . "wpm_timefix_customers";
		$orders_table               = $wpdb->prefix . "wpm_timefix_orders";

		$query = $wpdb->prepare( "SELECT 
            a.id AS appointment_id,
            a.status AS appointment_status,
            a.created_at AS appointment_created_at,
            a.additional_notes,
            s.appointment_date,
            s.start_time,
            s.end_time,
            s.service_id,
            s.staff_id,
            c.id AS customer_id,
            c.name,
            c.email,
            c.phone,
            o.payment_method,
            o.total AS order_total FROM {$appointments_table} a
                JOIN {$customers_table} c ON a.customer_id = c.id
                JOIN {$orders_table} o ON o.object_id = a.id
                JOIN {$appointment_services_table} s ON s.appointment_id = a.id
                WHERE a.id=%d", $appointment_id );

		$appointment = $wpdb->get_row( $query );

		$response = array();

		if ( ! empty( $appointment ) ) {
			$response = array(
				'id'             => $appointment->appointment_id,
				'service_title'  => get_the_title( $appointment->service_id ),
				'staff_name'     => get_the_title( $appointment->staff_id ),
				'name'           => $appointment->name,
				'phone'          => $appointment->phone,
				'email'          => $appointment->email,
				'customer_notes' => $appointment->additional_notes,
				'status'         => $appointment->appointment_status
			);
		}

		return rest_ensure_response( $response );
	}

	public function update_appointment( WP_REST_Request $request ) {
		global $wpdb;

		if ( ! current_user_can( 'manage_options' ) ) {
			$response = [
				'status'        => "error",
				'error'         => 'FORBIDDEN',
				'code'          => '403',
				'error_message' => esc_html__( "You are not permitted to update.", "timefix" )
			];
			wp_send_json( $response, 403 );
		}

		if ( ! $request->get_param( 'appointment_id' ) ) {
			$response = [
				'status'        => "error",
				'error'         => 'BADREQUEST',
				'code'          => '400',
				'error_message' => esc_html__( 'Appointment ID not found.', "timefix" )
			];
			wp_send_json( $response, 400 );
		}

		if ( ! $request->get_param( 'customer_id' ) ) {
			$response = [
				'status'        => "error",
				'error'         => 'BADREQUEST',
				'code'          => '400',
				'error_message' => esc_html__( 'Customer ID not found.', "timefix" )
			];
			wp_send_json( $response, 400 );
		}

		if ( ! $request->get_param( 'name' ) ) {
			$response = [
				'status'        => "error",
				'error'         => 'BADREQUEST',
				'code'          => '400',
				'error_message' => esc_html__( 'Name not found.', "timefix" )
			];
			wp_send_json( $response, 400 );
		}

		if ( ! $request->get_param( 'phone' ) ) {
			$response = [
				'status'        => "error",
				'error'         => 'BADREQUEST',
				'code'          => '400',
				'error_message' => esc_html__( 'Phone not found.', "timefix" )
			];
			wp_send_json( $response, 400 );
		}

		if ( ! $request->get_param( 'status' ) ) {
			$response = [
				'status'        => "error",
				'error'         => 'BADREQUEST',
				'code'          => '400',
				'error_message' => esc_html__( 'Appointment status not found.', "timefix" )
			];
			wp_send_json( $response, 400 );
		}

		$success = false;
		$message = esc_html__( 'Something wrong!', 'timefix' );

		$customer_name   = sanitize_text_field( $request->get_param( 'name' ) );
		$customer_email  = sanitize_email( $request->get_param( 'email' ) );
		$customer_phone  = sanitize_text_field( $request->get_param( 'phone' ) );
		$additional_note = sanitize_textarea_field( $request->get_param( 'customer_notes' ) );
		$status          = sanitize_text_field( $request->get_param( 'status' ) );
		$customer_id     = absint( $request->get_param( 'customer_id' ) );
		$appointment_id  = absint( $request->get_param( 'appointment_id' ) );

		// Begin transaction
		$wpdb->query( 'START TRANSACTION' );

		try {
			$appointments_table = $wpdb->prefix . "wpm_timefix_appointments";
			$customers_table    = $wpdb->prefix . "wpm_timefix_customers";

			// Update the customer
			$customer_updated = $wpdb->update(
				$customers_table,
				[
					'name'  => $customer_name,
					'phone' => $customer_phone,
					'email' => $customer_email
				],
				[ 'id' => $customer_id ],
				[ '%s', '%s', '%s' ],
				[ '%d' ]
			);

			// Update the appointment
			$appointment_updated = $wpdb->update(
				$appointments_table,
				[
					'status'           => $status,
					'additional_notes' => $additional_note
				],
				[
					'id'          => $appointment_id,
					'customer_id' => $customer_id
				],
				[ '%s', '%s' ],
				[ '%d', '%d' ]
			);

			if ( $customer_updated !== false && $appointment_updated !== false ) {
				$wpdb->query( 'COMMIT' );
				$success = true;
				$message = esc_html__( 'Appointment updated successfully.', 'timefix' );

				do_action( 'wpm_timefix_appointment_updated', $appointment_id, $status, [
					'appointment_id' => $appointment_id,
					'customer_id'    => $customer_id,
					'customer_email' => $customer_email
				] );

			} else {
				throw new \Exception( 'Update failed.' );
			}
		} catch ( \Exception $e ) {
			$wpdb->query( 'ROLLBACK' );
			$message = sprintf( esc_html__( 'Update failed! %s', 'timefix' ), $e->getMessage() );
		}

		$response = [
			'success' => $success,
			'msg'     => $message
		];

		return rest_ensure_response( $response );
	}

	/**
	 * @param WP_REST_Request $request
	 *
	 * @return bool
	 */
	public function permission_check( WP_REST_Request $request ) {

		$nonce = $request->get_header( 'X-WP-Nonce' );

		if ( ! $nonce ) {
			return false;
		}

		if ( ! is_user_logged_in() ) {
			return false;
		}

		$current_user = wp_get_current_user();

		if ( ! in_array( 'administrator', $current_user->roles ) ) {
			return false;
		}

		return true;
	}
}