<?php
if ( ! defined( 'ABSPATH' ) ) exit;


function xsuri_get_pin_meta_key( int $term_id = 0 ): string {
    if ( $term_id > 0 ) {
        return '_xsuri_pin_pos_cat_' . intval( $term_id );
    }
    return '_xsuri_pin_pos';
}

function xsuri_get_query_product_cat_term_id( WP_Query $query = null ): int {
    if ( ! $query || ! is_a( $query, 'WP_Query' ) ) return 0;
    $qo = $query->get_queried_object();
    if ( $qo && isset( $qo->taxonomy ) && $qo->taxonomy === 'product_cat' && isset( $qo->term_id ) ) {
        return intval( $qo->term_id );
    }
    $tax_query = $query->get( 'tax_query' );
    if ( is_array( $tax_query ) ) {
        foreach ( $tax_query as $cond ) {
            if ( ! is_array( $cond ) ) continue;
            if ( isset( $cond['taxonomy'] ) && $cond['taxonomy'] === 'product_cat' && isset( $cond['terms'] ) ) {
                $terms = $cond['terms'];
                if ( is_array( $terms ) && count( $terms ) === 1 ) {
                    return intval( array_values( $terms )[0] );
                } elseif ( ! is_array( $terms ) ) {
                    return intval( $terms );
                }
            }
        }
    }
    $slug = $query->get( 'product_cat' );
    if ( $slug ) {
        $term = get_term_by( 'slug', $slug, 'product_cat' );
        if ( $term && ! is_wp_error( $term ) ) {
            return intval( $term->term_id );
        }
    }
    return 0;
}

function xsuri_get_all_products( int $cat_id = 0 ): array {
    $meta_key = xsuri_get_pin_meta_key($cat_id);
    $base_args = [
        'post_type'      => 'product',
        'posts_per_page' => -1,
        'post_status'    => 'publish',
    ];
    if ($cat_id) {
        $base_args['tax_query'] = [[
            'taxonomy' => 'product_cat',
            'field'    => 'term_id',
            'terms'    => $cat_id,
        ]];
    }
    $pinned = get_posts($base_args + [
        'meta_key'       => $meta_key,
        'orderby'        => ['meta_value_num'=>'ASC','title'=>'ASC'],
        'order'          => 'ASC',
        'meta_query'     => [[ 'key'=>$meta_key,'value'=>0,'compare'=>'>','type'=>'NUMERIC' ]]
    ]);
    $unpinned = get_posts($base_args + [
        'meta_key'       => $meta_key,
        'orderby'        => ['meta_value_num'=>'ASC','title'=>'ASC'],
        'order'          => 'ASC',
        'meta_query'     => [[ 'key'=>$meta_key,'value'=>0,'compare'=>'=','type'=>'NUMERIC' ]]
    ]);
    $no_meta = get_posts($base_args + [
        'orderby'=>'title','order'=>'ASC',
        'meta_query'=>[[ 'key'=>$meta_key,'compare'=>'NOT EXISTS' ]]
    ]);
    $ids = array_merge( wp_list_pluck($pinned,'ID'), wp_list_pluck($unpinned,'ID') );
    $result = [];
    foreach ($pinned as $p) $result[] = $p;
    foreach ($unpinned as $p) if (!in_array($p->ID, wp_list_pluck($result,'ID'), true)) $result[] = $p;
    foreach ($no_meta as $p) if (!in_array($p->ID, $ids, true)) $result[] = $p;

    if ( ! false && count($result) > 100 ) {
        $result = array_slice($result, 0, 100);
    }

    if ( $cat_id === 0 ) {
        $order = get_option('xsuri_category_order');
        if ( is_array($order) && ! empty($order) ) {
            $seen = [];
            $posMapTop = [];
            $pos = 1;
            foreach ( $order as $tid ) {
                $tid = intval($tid);
                if ( $tid <= 0 ) continue;
                if ( isset($seen[$tid]) ) continue;
                $seen[$tid] = true;
                $term = get_term( $tid, 'product_cat' );
                if ( $term && ! is_wp_error($term) && intval($term->parent) === 0 ) {
                    $posMapTop[$tid] = $pos++;
                }
            }
            if ( ! empty($posMapTop) ) {
                $withIdx = [];
                foreach ( $result as $i => $p ) {
                    $terms = wp_get_post_terms( $p->ID, 'product_cat', [ 'fields' => 'ids' ] );
                    $rank = PHP_INT_MAX;
                    if ( is_array($terms) ) {
                        foreach ( $terms as $tid ) {
                            $tid = intval($tid);
                            if ( ! isset($posMapTop[$tid]) ) {
                                $anc = get_ancestors( $tid, 'product_cat', 'taxonomy' );
                                $top = $tid;
                                if ( is_array($anc) && ! empty($anc) ) {
                                    $top = intval( end($anc) );
                                }
                                if ( isset($posMapTop[$top]) && $posMapTop[$top] < $rank ) {
                                    $rank = $posMapTop[$top];
                                }
                            } else {
                                if ( $posMapTop[$tid] < $rank ) $rank = $posMapTop[$tid];
                            }
                        }
                    }
                    $withIdx[] = [ 'post' => $p, 'rank' => $rank, 'i' => $i ];
                }
                usort( $withIdx, function($a,$b){
                    if ( $a['rank'] !== $b['rank'] ) return $a['rank'] <=> $b['rank'];
                    return $a['i'] <=> $b['i'];
                });
                $result = [];
                foreach ( $withIdx as $row ) $result[] = $row['post'];
            }
        }
    }
    return $result;
}

function xsuri_dnd_get_color_settings(): array {
    $defaults = [
        'primary'   => '#008aff',
        'secondary' => '#2bbaff',
        'bg_soft'   => '#eaf5ff',
        'accent'    => '#005bb5',
        'radius'    => '18px',
    ];
    if ( ! false ) return $defaults;
    $saved = get_option( XSURI_DND_OPT_COLOR_SETTINGS, [] );
    return wp_parse_args( is_array($saved)?$saved:[], $defaults );
}