<?php
if ( ! defined( 'ABSPATH' ) ) exit;

add_action('wp_ajax_xsuri_move_category', function(){
    check_ajax_referer('xsuri_sort_products_nonce', 'nonce');
    if ( ! current_user_can('manage_woocommerce') ) {
        wp_send_json_error(['msg'=>__('Insufficient permissions.', 'tileflow-sorting-products')]);
    }
    $term_id = isset($_POST['term_id']) ? intval($_POST['term_id']) : 0;
    $new_parent = isset($_POST['new_parent_id']) ? intval($_POST['new_parent_id']) : 0;
    if ( $term_id <= 0 || $new_parent < 0 ) {
        wp_send_json_error(['msg'=>__('Invalid data.', 'tileflow-sorting-products')]);
    }
    if ( $new_parent === $term_id ) {
        wp_send_json_error(['msg'=>__('Invalid parent.', 'tileflow-sorting-products')]);
    }
    if ( $new_parent > 0 ) {
        $anc = get_ancestors( $new_parent, 'product_cat', 'taxonomy' );
        if ( in_array( $term_id, array_map('intval', $anc), true ) ) {
            wp_send_json_error(['msg'=>__('Cannot move into its own descendant.', 'tileflow-sorting-products')]);
        }
    }
    $res = wp_update_term( $term_id, 'product_cat', [ 'parent' => $new_parent ] );
    if ( is_wp_error($res) ) {
        wp_send_json_error(['msg'=>$res->get_error_message()]);
    }
    wp_send_json_success(['msg'=>__('Category parent changed.', 'tileflow-sorting-products')]);
});

add_action('wp_ajax_xsuri_save_category_order', function() {
    check_ajax_referer('xsuri_sort_products_nonce', 'nonce');
    if ( ! current_user_can('manage_woocommerce') ) {
        wp_send_json_error(['msg'=>__('Insufficient permissions.', 'tileflow-sorting-products')]);
    }
    if ( isset($_POST['cat_order']) && is_array($_POST['cat_order']) ) {
        $order = array_map('intval', $_POST['cat_order']);
        $seen = [];
        $norm = [];
        foreach ($order as $v) {
            if (!isset($seen[$v])) { $seen[$v] = true; $norm[] = $v; }
        }
        update_option( 'xsuri_category_order', $norm );
        wp_send_json_success(['msg'=>__('Category order saved.', 'tileflow-sorting-products')]);
    }
    wp_send_json_error(['msg'=>__('Invalid data.', 'tileflow-sorting-products')]);
});

add_action('wp_ajax_xsuri_get_products', function(){
    check_ajax_referer('xsuri_sort_products_nonce', 'nonce');

    if ( ! current_user_can('manage_woocommerce') ) {
        wp_send_json_error(['msg'=>__('Insufficient permissions.', 'tileflow-sorting-products')]);
    }
    
    // Używamy wp_unslash przy pobieraniu danych POST
    $cat_id   = isset($_POST['cat_id']) ? intval($_POST['cat_id']) : 0;
    $page     = isset($_POST['page']) ? max(1, intval($_POST['page'])) : 1;
    $per_page = isset($_POST['per_page']) ? max(1, min(200, intval($_POST['per_page']))) : 40;
    $search   = isset($_POST['search']) ? sanitize_text_field(wp_unslash($_POST['search'])) : '';
    $instock  = isset($_POST['instock']) ? (bool) intval($_POST['instock']) : false;

    $all = xsuri_get_all_products($cat_id);
    if ($search !== '') {
        $needle = mb_strtolower($search);
        $all = array_values(array_filter($all, function($p) use ($needle){
            return mb_strpos(mb_strtolower($p->post_title), $needle) !== false;
        }));
    }
    if ($instock) {
        $all = array_values(array_filter($all, function($p){
            return get_post_meta($p->ID, '_stock_status', true) !== 'outofstock';
        }));
    }
    $total   = count($all);
    $pages   = max(1, (int)ceil($total / $per_page));
    if ($page > $pages) $page = $pages;
    $offset  = ($page - 1) * $per_page;
    $slice   = array_slice($all, $offset, $per_page);

    ob_start();
    foreach ($slice as $product) {
        $pid   = $product->ID;
        
        $title = $product->post_title; 
        $img   = get_the_post_thumbnail($pid, [60,60], ['style'=>'vertical-align:middle']);
        $stock = get_post_meta($pid, '_stock_status', true);
        $is_out = ($stock === 'outofstock');
        $class = 'xsuri-product-item' . ($is_out ? ' outofstock' : '');
        
        echo '<li class="'.esc_attr($class).'" data-id="'.intval($pid).'">';
        
        
        if ( $img ) {
            echo wp_kses_post( $img );
        } else {
            echo '<span class="no-img">'.esc_html__('No image','tileflow-sorting-products').'</span>';
        }
        
       
        echo '<span class="xsuri-title'.($is_out?' outofstock':'').'">'.esc_html($title).'</span>';
        echo '</li>';
    }
    $html = ob_get_clean();
    wp_send_json_success([
        'html'      => $html,
        'total'     => $total,
        'page'      => $page,
        'pages'     => $pages,
        'per_page'  => $per_page,
        'has_more'  => $page < $pages,
    ]);
});

add_action('wp_ajax_xsuri_save_products_order', function() {
    check_ajax_referer('xsuri_sort_products_nonce', 'nonce');
    if ( ! current_user_can('manage_woocommerce') ) {
        wp_send_json_error(['msg'=>__('Insufficient permissions.', 'tileflow-sorting-products')]);
    }
    $raw_ids = isset($_POST['ids']) ? wp_unslash($_POST['ids']) : '';
    $ids = [];


    if ( is_array($raw_ids) ) {
        
        $ids = array_map('intval', $raw_ids);
    } elseif ( is_string($raw_ids) ) {
      
        $clean_string = sanitize_text_field($raw_ids);
        $parts = explode(',', $clean_string);
        foreach ($parts as $p) {
            $p = trim($p);
            if ($p !== '') {
                $ids[] = intval($p);
            }
        }
    }
    $seen = [];
    $ordered = [];
    foreach ($ids as $v) {
        if (!isset($seen[$v])) { $seen[$v]=true; $ordered[] = $v; }
    }
    $ids = $ordered;

    if ( empty($ids) ) {
        wp_send_json_error(['msg'=>__('No products.', 'tileflow-sorting-products')]);
    }

    if (count($ids) > 100 ) {
        $ids = array_slice($ids, 0, 100);
    }

    $cat_id   = isset($_POST['cat_id']) ? intval($_POST['cat_id']) : 0;
    $meta_key = xsuri_get_pin_meta_key($cat_id);

    foreach ($ids as $i => $pid) {
        update_post_meta($pid, $meta_key, $i+1);
    }

    $args = [
        'post_type'      => 'product',
        'posts_per_page' => -1,
        'post_status'    => 'publish',
        'fields'         => 'ids',
        'meta_query'     => [[ 'key'=>$meta_key,'value'=>0,'compare'=>'>','type'=>'NUMERIC' ]]
    ];
    if ($cat_id) {
        $args['tax_query'] = [[ 'taxonomy'=>'product_cat','field'=>'term_id','terms'=>$cat_id ]];
    }
    $currently_pinned_ids = get_posts($args);
    $to_unpin = array_diff($currently_pinned_ids, $ids);
    foreach ($to_unpin as $pid) {
        update_post_meta($pid, $meta_key, 0);
    }

    if ( function_exists('wc_delete_product_transients') ) {
        wc_delete_product_transients();
    }

    $msg = false ? __('Order saved!', 'tileflow-sorting-products') : __('Saved (Possibly truncated to 100).', 'tileflow-sorting-products');
    wp_send_json_success(['msg'=>$msg]);
});
