<?php

namespace TidioLiveChat\Http\Client;

if (!defined('WPINC')) {
    die('File loaded directly. Exiting.');
}

use TidioLiveChat\Http\Exception\ErrorResponseException;
use TidioLiveChat\Http\Exception\HttpClientException;
use TidioLiveChat\Http\Exception\UnauthorizedResponseException;
use TidioLiveChat\Http\HttpClient;
use TidioLiveChat\Logs\Logger;
use TidioLiveChat\Utils\Url;
use function curl_error;
use function json_encode;

class CurlHttpClient implements HttpClient
{
    /**
     * @var Logger
     */
    private $logger;
    /**
     * @var string
     */
    private $apiUrl;
    /**
     * @var string[]
     */
    private $headers;

    /**
     * @param Logger $logger
     * @param string $apiUrl
     * @param string[] $additionalHeaders
     */
    public function __construct($logger, $apiUrl, $additionalHeaders = [])
    {
        $this->logger = $logger;
        $this->apiUrl = $apiUrl;
        $this->headers = array_merge(
            [
                'Content-Type: application/json',
                'Accept: application/json'
            ],
            $additionalHeaders
        );
    }

    /**
     * @inerhitDoc
     */
    public function sendPostRequest($path, $data = [])
    {
        $url = Url::build($this->apiUrl, $path);
        $ch = curl_init();

        curl_setopt($ch, CURLOPT_HEADER, 1);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, $this->headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_USERAGENT, 'Tidio WordPress plugin ' . TIDIOCHAT_VERSION);

        $response = curl_exec($ch);
        $responseInfo = curl_getinfo($ch);
        $curlError = curl_error($ch);
        curl_close($ch);

        if ($curlError) {
            $this->logger->error($curlError);
        }

        $responseData = $this->parseResponseData($response, $responseInfo);
        $this->log($responseInfo, $url, $responseData);
        $this->validateResponse($responseData, $responseInfo);

        return $responseData;
    }

    /**
     * @inerhitDoc
     */
    public function sendGetRequest($path)
    {
        $url = Url::build($this->apiUrl, $path);
        $ch = curl_init();

        curl_setopt($ch, CURLOPT_HEADER, 1);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $this->headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_USERAGENT, 'Tidio WordPress plugin ' . TIDIOCHAT_VERSION);

        $response = curl_exec($ch);
        $responseInfo = curl_getinfo($ch);
        $curlError = curl_error($ch);
        curl_close($ch);

        if ($curlError) {
            $this->logger->error($curlError);
        }

        $responseData = $this->parseResponseData($response, $responseInfo);
        $this->log($responseInfo, $url, $responseData);
        $this->validateResponse($responseData, $responseInfo);

        return $responseData;
    }

    /**
     * @param string|false $response
     * @param array<string, mixed> $responseInfo
     * @return array<string, mixed>
     */
    private function parseResponseData($response, $responseInfo)
    {
        $headerSize = $responseInfo['header_size'];
        $responseBody = substr($response, $headerSize);
        $responseData = json_decode($responseBody, true);

        if (false === $responseData) {
            return [];
        }

        return $responseData;
    }

    /**
     * @param array<string, mixed> $responseData
     * @param array<string, mixed> $responseInfo
     * @throws HttpClientException
     */
    private function validateResponse($responseData, $responseInfo)
    {
        $statusCode = $responseInfo['http_code'];
        if ($statusCode === 401) {
            throw new UnauthorizedResponseException();
        }

        if ($statusCode < 200 || $statusCode >= 300) {
            throw ErrorResponseException::withResponse($statusCode, $responseData);
        }
    }

    /**
     * @param array<string, mixed> $responseInfo
     * @param string $url
     * @param array<string, mixed>|null $responseData
     * @return void
     */
    private function log($responseInfo, $url, $responseData)
    {
        if (isset($responseInfo['http_code'])) {
            $statusCode = $responseInfo['http_code'];

            if ($statusCode > 299) {
                $data = [
                    'statusCode' => $statusCode,
                    'path' => $url,
                    'response' => $responseData,
                ];
                $this->logger->error('Invalid api response: ' . json_encode($data));
            }
        }
    }
}
