<?php
/**
 * Plugin Name:	Tida URL Screenshot
 * Description:	This plugin is adding the tool on your website that takes a screenshot from a URL.
 * Version: 	1.2.3
 * Author: 		Tida Web
 * Author URI: 	https://tidaweb.com
 * License:		GPL-2.0-or-later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: tida-url-screenshot
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if(!class_exists('Tida_URL_Screenshot'))
{
    class Tida_URL_Screenshot
    {
        /**
         * Construct the plugin object
         */
        public function __construct()
        {
            define( 'TIDA_URL_SCREENSHOT_VERSION', '1.2.3' );
            define( 'TIDA_URL_SCREENSHOT_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
            define( 'TIDA_URL_SCREENSHOT_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
            
            // Initialize Settings
            add_action( 'wp_enqueue_scripts', array(&$this, 'header_enqueue_scripts'));
            
            // tida_screenshot shortcode & ajax call
            add_shortcode( 'tida_screenshot', array(&$this, 'tida_screenshot_shortcode'));
            add_action( 'wp_ajax_get_tida_url_screenshot', array(&$this, 'ajax_get_tida_url_screenshot'));
            add_action( 'wp_ajax_nopriv_get_tida_url_screenshot', array(&$this, 'ajax_get_tida_url_screenshot'));
            
            // Add screenshot button to post type & ajax call
            add_filter( 'the_content' , array( &$this, 'add_screenshot_button_to_content' ) );
            add_action( 'wp_ajax_get_tida_post_screenshot', array(&$this, 'ajax_get_tida_post_screenshot'));
            add_action( 'wp_ajax_nopriv_get_tida_post_screenshot', 	array(&$this, 'ajax_get_tida_post_screenshot'));
            
            // initialize classes
            self::initialize_classes();
            
            do_action('tida_url_screenshot_plugin_hooks');
        	
        } // END public function __construct()
        
        /**
         * Initialize classes of Tida URL Screenshot.
         *
         * @return void
         */
        public static function initialize_classes()
        {
            // include settings page
            require_once('includes/admin/class-settings.php');
            
            // include api services
            require_once('includes/class-api.php');
            require_once('includes/class-abstract-api.php');
            require_once('includes/class-api-flash.php');
            require_once('includes/class-jsonlink.php');
            require_once('includes/class-screenshot-layer.php');
            require_once('includes/class-screenshot-machine.php');
            require_once('includes/class-screenshot-one.php');
        }
        
        /**
         * Enqueue necessary styles and scripts in the header.
         *
         * @return void
         */
        public function header_enqueue_scripts()
        {
            $post = get_post( get_the_ID() );
            if(isset( $post ) && !empty( $post ) && ( has_shortcode($post->post_content, 'tida_screenshot') || is_single() ) )
            {
            	wp_enqueue_style('tida-screenshot', plugins_url( '/assets/css/tida-screenshot.css', __FILE__ ), array(), TIDA_URL_SCREENSHOT_VERSION );
            	wp_register_script('tida-screenshot', plugins_url( '/assets/js/tida-screenshot.js', __FILE__ ), array('jquery'), TIDA_URL_SCREENSHOT_VERSION, true );
            
            	wp_localize_script( 'tida-screenshot', 'tida_screenshot_params', [
            		'ajax_url' => admin_url('admin-ajax.php'),
            		'please_text' => __('Please Wait...', 'tida-url-screenshot')
            	]);
            
            	wp_enqueue_script('tida-screenshot');
            }
        }
        
        /**
         * Handle the Tida Screenshot shortcode.
         *
         * @return string The HTML output of the screenshot form.
         */
        public function tida_screenshot_shortcode()
        {
            ob_start();
            $tida_url_screenshot_settings = get_option('tida_url_screenshot_settings');
            ?>
            <form class="tida-screenshot-form" method="post" action="">
            	<?php wp_nonce_field('tida_url_screenshot_nonce', 'tida_url_screenshot_nonce'); ?>
                <input type="text" <?php if(!empty($tida_url_screenshot_settings['input_class'])) echo 'class="'.esc_attr( $tida_url_screenshot_settings['input_class'] ).'"'; ?> name="url" placeholder="<?php echo esc_attr__('Enter URL...', 'tida-url-screenshot'); ?>" />
                <input type="submit" <?php if(!empty($tida_url_screenshot_settings['button_class'])) echo 'class="'.esc_attr( $tida_url_screenshot_settings['button_class'] ).'"'; ?> name="check" value="<?php echo esc_attr__('Take Screenshot', 'tida-url-screenshot'); ?>" />
            </form>
            <div class="screenshot-msg"></div>
            <div class="screenshot-result"><img src="" /></div>
            <?php
            
            $output = ob_get_contents();
            ob_end_clean();
            return apply_filters('tida_screenshot_shortcode_filter', $output);
        }
        
        /**
         * Add a screenshot button to the post content.
         *
         * @param string $content The original post content.
         * @return string The modified content with the screenshot button.
         */
        public function add_screenshot_button_to_content($content)
        {
            $tida_url_screenshot_settings = get_option('tida_url_screenshot_settings');
            $enabled_post_types = ( isset( $tida_url_screenshot_settings['post_types'] ) && !empty( $tida_url_screenshot_settings['post_types'] ) ) ? array_map('sanitize_text_field', $tida_url_screenshot_settings['post_types'] ) : array();
            $button_position    = ( isset( $tida_url_screenshot_settings['button_position'] ) && !empty( $tida_url_screenshot_settings['button_position'] ) ) ? esc_attr( $tida_url_screenshot_settings['button_position'] ) : 'bottom';
        
            if ( is_singular( $enabled_post_types ) ) {
                $button_html = '<button class="tida-post-screenshot" data-pid="'. get_the_ID() .'">' . __('Take a screenshot from this page', 'tida-url-screenshot') . '</button>';
                $button_html .= '<div class="screenshot-msg"></div>';
                $button_html .= '<div class="screenshot-result"><img src="" /></div>';
        
                if ( 'top' === $button_position ) {
                    return $button_html . $content;
                } else {
                    return $content . $button_html;
                }
            }
        
            return $content;
        }
        
        /**
         * Ajax handler for getting the screenshot from API.
         *
         * @return void JSON encoded response with screenshot status
         */
        public function ajax_get_tida_url_screenshot()
        {
            // get screenshot from api
            $result = [];
            if( isset( $_POST['url'] ) && !empty( $_POST['url'] ) )
            {
                $url = sanitize_url( wp_unslash( $_POST['url'] ) );
                if(!is_null($url))
                {
                    $tida_url_screenshot_settings = get_option('tida_url_screenshot_settings');
                    if( !empty( $tida_url_screenshot_settings['api_service'] ) )
                    {
                        switch( $tida_url_screenshot_settings['api_service'] )
                        {
                            case "abstractapi":
                                $screenshot_img_src = AbstractAPI::prepare_data_for_api( $url );
                                break;
                            case "apiflash":
                                $screenshot_img_src = ApiFlashAPI::prepare_data_for_api( $url );
                                break;
                            case "jsonlink":
                                $screenshot_img_src = JsonLinkAPI::prepare_data_for_api( $url );
                                break;
                            case "screenshotlayer":
                                $screenshot_img_src = ScreenshotLayerAPI::prepare_data_for_api( $url );
                                break;
                            case "screenshotmachine":
                                $screenshot_img_src = ScreenShotMachineAPI::prepare_data_for_api( $url );
                                break;
                            case "screenshotone":
                                $screenshot_img_src = ScreenshotOneAPI::prepare_data_for_api( $url );
                                break;
                            default:
                                $screenshot_img_src = '';
                                break;
                        }
                    }
                }
            }
            
            if( !empty( $screenshot_img_src ) )
            {
                $result = [ 'status' => true, 'src' => $screenshot_img_src, 'msg' => '' ];
            }
            else
            {
                $result = [ 'status' => false, 'msg' => esc_attr__('Error retrieving URL screenshot', 'tida-url-screenshot') ];
            }
            
            echo wp_json_encode( $result );
            wp_die();
        }
        
        /**
         * Ajax handler for getting the screenshot of a post.
         *
         * @return void JSON encoded response with screenshot status
         */
        public function ajax_get_tida_post_screenshot()
        {
            // get screenshot from api
            $result = [];
            if( isset( $_POST['pid'] ) && !empty( $_POST['pid'] ) )
            {
                $post_id = absint( wp_unslash( $_POST['pid'] ) );
                if( !empty( $post_id ) && $post_id >  0)
                {
                    $post_info = get_post( $post_id );
                    if( $post_info )
                    {
                        $post_url = get_permalink( $post_info->ID );
                        $tida_url_screenshot_settings = get_option('tida_url_screenshot_settings');
                        if( !empty( $tida_url_screenshot_settings['api_service'] ) )
                        {
                            switch( $tida_url_screenshot_settings['api_service'] )
                            {
                                case "abstractapi":
                                    $screenshot_img_src = AbstractAPI::prepare_data_for_api( $post_url );
                                	break;
                                case "apiflash":
                                    $screenshot_img_src = ApiFlashAPI::prepare_data_for_api( $post_url );
                                	break;
                                case "jsonlink":
                                    $screenshot_img_src = JsonLinkAPI::prepare_data_for_api( $post_url );
                                	break;
                                case "screenshotlayer":
                                    $screenshot_img_src = ScreenshotLayerAPI::prepare_data_for_api( $post_url );
                                	break;
                                case "screenshotmachine":
                                    $screenshot_img_src = ScreenShotMachineAPI::prepare_data_for_api( $post_url );
                                	break;
                                case "screenshotone":
                                    $screenshot_img_src = ScreenshotOneAPI::prepare_data_for_api( $post_url );
                                	break;
                                default:
                                	$screenshot_img_src = '';
                                	break;
                            }
                        }
                    }
                }
            }
            
            if( !empty( $screenshot_img_src ) )
            {
                $result = [ 'status' => true, 'src' => $screenshot_img_src, 'msg' => '' ];
            }
            else
            {
                $result = [ 'status' => false, 'msg' => esc_attr__('Error retrieving URL screenshot', 'tida-url-screenshot') ];
            }
            
            echo wp_json_encode( $result );
            wp_die();
        }
        
        /**
         * Activate the plugin.
         *
         * @return void
         */
        public static function activate()
        {
            // Do nothing
        } // END public static function activate
        
        /**
         * Deactivate the plugin.
         *
         * @return void
         */
        public static function deactivate()
        {
            // Do nothing
        } // END public static function deactivate
    } // END class Tida_URL_Screenshot
} // END if(!class_exists('Tida_URL_Screenshot'))

if(class_exists('Tida_URL_Screenshot'))
{
    // instantiate the plugin class
    new Tida_URL_Screenshot();
    
    register_activation_hook( __FILE__, array( 'Tida_URL_Screenshot', 'activate' ) );
    register_deactivation_hook( __FILE__, array( 'Tida_URL_Screenshot', 'deactivate' ) );
}