<?php
/**
 * Settings Class
 * The class responsible for managing the settings in WordPress.
 * 
 * Package: Tida URL Screenshot
 * Author: Tida Web
 * Last Modified Time: 2025/01/25 01:40:42
 */

if (!defined('ABSPATH')) {
    exit;
} // Exit if accessed directly

class Tida_URL_Screenshot_Settings {
    
    /**
     * Holds the singleton instance of the Tida_URL_Screenshot_Settings class.
     *
     * @var Tida_URL_Screenshot_Settings|null
     */
    protected static $_instance = null;

    /**
     * Singleton instance of the Tida_URL_Screenshot_Settings class.
     * Ensures that only one instance of the class exists at any given time.
     * 
     * @return Tida_URL_Screenshot_Settings|null Returns the singleton instance of Tida_URL_Screenshot_Settings.
     */
    public static function instance() {
        if (is_null(self::$_instance)) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * Constructor method.
     * Adds actions to initialize settings, create the settings page, enqueue scripts, and handle AJAX requests.
     * 
     * @return void
     */
    public function __construct()
    {
        add_action('admin_enqueue_scripts', array(&$this, 'admin_enqueue_scripts') );
        add_action('admin_menu', array(&$this, 'add_settings_menu') );
        add_action('admin_init', array(&$this, 'settings_init') );
    }
    
    /**
     * Enqueues styles for the admin settings page.
     * 
     * @return void
     */
    public function admin_enqueue_scripts()
    {
        $screen = get_current_screen();
        if (isset($screen->id) && $screen->id === 'settings_page_tida-url-screenshot-settings') {
            wp_enqueue_style('tida-screenshot', TIDA_URL_SCREENSHOT_PLUGIN_URL . '/assets/css/tida-screenshot-admin.css', array(), TIDA_URL_SCREENSHOT_VERSION );
        }
    }

    /**
     * Creates the settings page under the admin menu.
     * 
     * @return void
     */
    public function add_settings_menu()
    {
        add_options_page(
            __( 'Tida URL Screenshot Settings', 'tida-url-screenshot' ),
            __( 'Tida URL Screenshot Settings', 'tida-url-screenshot' ),
            'manage_options',
            'tida-url-screenshot-settings',
            array( &$this, 'settings_callback' )
        );
    }

    /**
     * Initializes the settings and adds settings fields.
     * 
     * @return void
     */
    public function settings_init()
    {
        register_setting('tida_url_screenshot_settings', 'tida_url_screenshot_settings', array($this, 'sanitize'));

        add_settings_section(
            'tida_url_screenshot_settings_section',
            '',
            '',
            'tida-url-screenshot-settings'
        );

        // API Service
        add_settings_field(
            'api_service',
            __('API Service', 'tida-url-screenshot'),
            array($this, 'api_service_field_callback'),
            'tida-url-screenshot-settings',
            'tida_url_screenshot_settings_section'
        );

        // API Key
        add_settings_field(
            'api_key',
            __('API Key', 'tida-url-screenshot'),
            array($this, 'api_key_field_callback'),
            'tida-url-screenshot-settings',
            'tida_url_screenshot_settings_section'
        );

        // Input Class
        add_settings_field(
            'input_class',
            __('Input Class', 'tida-url-screenshot'),
            array($this, 'input_class_field_callback'),
            'tida-url-screenshot-settings',
            'tida_url_screenshot_settings_section'
        );

        // Button Class
        add_settings_field(
            'button_class',
            __('Button Class', 'tida-url-screenshot'),
            array($this, 'button_class_field_callback'),
            'tida-url-screenshot-settings',
            'tida_url_screenshot_settings_section'
        );
        
        // Enable on post types
        add_settings_field(
            'post_types',
            __('Post Types', 'tida-url-screenshot'),
            array($this, 'post_types_field_callback'),
            'tida-url-screenshot-settings',
            'tida_url_screenshot_settings_section'
        );
        
        // Button Position for post types
        add_settings_field(
            'button_position',
            __('Button Position', 'tida-url-screenshot'),
            array($this, 'button_position_field_callback'),
            'tida-url-screenshot-settings',
            'tida_url_screenshot_settings_section'
        );
    }

    /**
     * Displays the settings page.
     * 
     * @return void
     */
    public static function settings_callback()
    {
        ?>
        <div class="wrap">
            <div class="settings-header">
                <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
            </div>
            <div class="settings-body">
                <form method="post" action="options.php">
                    <?php
                    settings_fields('tida_url_screenshot_settings');
                    do_settings_sections('tida-url-screenshot-settings');
                    submit_button();
                    ?>
                </form>
            </div>
        </div>
        <?php
    }

    /**
     * Outputs the dropdown field for selecting API services.
     * 
     * @return void
     */
    public function api_service_field_callback()
    {
        global $wpdb;
        $settings = get_option('tida_url_screenshot_settings');
        ?>
        <select name="tida_url_screenshot_settings[api_service]">
            <option value=""><?php esc_html_e('Select the API service...', 'tida-url-screenshot'); ?></option>
            <?php
            $api_services_list = array();
            $api_services_list = apply_filters('tida_screenshot_api_list', $api_services_list );
            
            foreach ($api_services_list as $key => $item) {
                $selected = (isset($settings['api_service']) && $settings['api_service'] === $key) ? 'selected' : '';
                echo '<option value="' . esc_attr($key) . '" ' . esc_attr($selected) . '>' . esc_html($item) . '</option>';
            }
            ?>
        </select>
        <p class="description"><?php echo esc_attr__('Please select the service for taking the screenshot.', 'tida-url-screenshot'); ?></p>
        <?php
    }
    
    /**
     * Outputs the input field for the API key.
     * 
     * @return void
     */
    public function api_key_field_callback()
    {
        $settings = get_option('tida_url_screenshot_settings');
        ?>
        <input type="text" name="tida_url_screenshot_settings[api_key]" value="<?php echo esc_attr( $settings['api_key'] ?? ''); ?>" size="70" />
        <p class="description"><?php echo esc_attr__('API Key = Access Token = User Key = API Token', 'tida-url-screenshot'); ?></p>
        <?php
    }

    /**
     * Outputs the input field for the input class.
     * 
     * @return void
     */
    public function input_class_field_callback()
    {
        $settings = get_option('tida_url_screenshot_settings');
        ?>
        <input type="text" name="tida_url_screenshot_settings[input_class]" value="<?php echo esc_attr( $settings['input_class'] ?? ''); ?>" size="20" />
        <p class="description"><?php echo esc_attr__('Enter the name of input field class for custom style.', 'tida-url-screenshot'); ?></p>
        <?php
    }

    /**
     * Outputs the input field for the button class.
     * 
     * @return void
     */
    public function button_class_field_callback()
    {
        $settings = get_option('tida_url_screenshot_settings');
        ?>
        <input type="text" name="tida_url_screenshot_settings[button_class]" value="<?php echo esc_attr( $settings['button_class'] ?? ''); ?>" size="20" />
        <p class="description"><?php echo esc_attr__('Enter the name of button class for custom style.', 'tida-url-screenshot'); ?></p>
        <?php
    }
    
    /**
     * Outputs a list of checkboxes for selecting post types to enable screenshot functionality.
     *
     * @return void
     */
    public function post_types_field_callback()
    {
        global $wpdb;
        $settings = get_option('tida_url_screenshot_settings');
        ?>
        <?php foreach (self::get_post_types() as $key => $item): ?>
            <p>
                <label>
                    <input type="checkbox" name="tida_url_screenshot_settings[post_types][]" value="<?php echo esc_attr($key); ?>" <?php if (isset($settings['post_types']) && in_array($key, $settings['post_types'])) echo 'checked'; ?>> <?php echo esc_attr($item); ?>
                </label>
            </p>
        <?php endforeach; ?>
        <p class="description"><?php echo esc_attr__('Select one or more post types to enable screenshot functionality on them.', 'tida-url-screenshot'); ?></p>
        <?php
    }
    
    /**
     * Outputs a dropdown list for selecting the button position.
     *
     * @return void
     */
    public function button_position_field_callback()
    {
        global $wpdb;
        $settings = get_option('tida_url_screenshot_settings');
        ?>
        <select name="tida_url_screenshot_settings[button_position]">
            <option value=""><?php esc_html_e('Select the button position...', 'tida-url-screenshot'); ?></option>
            <?php
            $button_positions = array('top' => __('Top of Content', 'tida-url-screenshot'), 'bottom' => __('Bottom of Content', 'tida-url-screenshot'));
            foreach ($button_positions as $key => $item) {
                $selected = (isset($settings['button_position']) && $settings['button_position'] === $key) ? 'selected' : '';
                echo '<option value="' . esc_attr($key) . '" ' . esc_attr($selected) . '>' . esc_html($item) . '</option>';
            }
            ?>
        </select>
        <p class="description"><?php echo esc_attr__('Select where to display the screenshot button.', 'tida-url-screenshot'); ?></p>
        <?php
    }
    
    /**
     * Sanitizes the input from the settings fields.
     *
     * @param array $input The input from the settings form.
     * @return array The sanitized input values, returning an array of sanitized settings fields.
     */
    public function sanitize($input)
    {
        $sanitized_input = array();
        
        if (isset($input['api_service'])) {
            $sanitized_input['api_service'] = sanitize_text_field($input['api_service']);
        }
    
        if (isset($input['api_key'])) {
            $sanitized_input['api_key'] = sanitize_text_field($input['api_key']);
        }
    
        if (isset($input['input_class'])) {
            $sanitized_input['input_class'] = sanitize_text_field($input['input_class']);
        }
    
        if (isset($input['button_class'])) {
            $sanitized_input['button_class'] = sanitize_text_field($input['button_class']);
        }
    
        // Handle post types
        if (isset($input['post_types']) && is_array($input['post_types'])) {
            $valid_post_types = array_keys(self::get_post_types()); // Get valid post type keys
            $sanitized_input['post_types'] = array(); // Initialize sanitized input array
    
            foreach ($input['post_types'] as $post_type) {
                // Check if the current post type is valid and sanitize
                if (in_array($post_type, $valid_post_types)) {
                    $sanitized_input['post_types'][] = sanitize_text_field($post_type);
                }
            }
        }
    
        if (isset($input['button_position']) && in_array($input['button_position'], array('top', 'bottom'))) {
            $sanitized_input['button_position'] = sanitize_text_field($input['button_position']);
        }
    
        return $sanitized_input;
    }
    
    /**
     * Get all registered post types.
     *
     * This method retrieves an array of public post types registered in WordPress, 
     * returning the post type names and their corresponding labels.
     *
     * @return array An associative array of post type names as keys and post type labels as values.
     */
    private static function get_post_types()
    {
        $post_types = get_post_types( array('public' => true ), 'objects' );
        $options    = array();
        
        foreach ($post_types as $post_type) {
            if( $post_type->name == 'attachment' ) continue;
            $options[$post_type->name] = $post_type->label;
        }
        
        return $options;
    }
}

if (class_exists('Tida_URL_Screenshot_Settings')) {
    Tida_URL_Screenshot_Settings::instance();
}