<?php
/**
 * Plugin Name: TicketPayGo Lite
 * Plugin URI: https://wordpress.org/plugins/ticketpaygo-lite/
 * Description: Simple and free WordPress event ticketing plugin. Create events, sell tickets, and manage attendees from wp-admin. Perfect for small events and venues.
 * Version: 1.4.1
 * Author: TicketPayGo Team
 * Author URI: https://ticketpaygo.upkyk.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: ticketpaygo-lite
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.9
 * Requires PHP: 7.4
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Check if TicketPayGo Full version is active - if so, disable Lite
if (defined('TICKETPAYGO_VERSION') || class_exists('TicketPayGo')) {
    add_action('admin_notices', function() {
        echo '<div class="notice notice-warning is-dismissible">';
        echo '<p><strong>' . esc_html__('TicketPayGo Lite', 'ticketpaygo-lite') . ':</strong> ';
        echo esc_html__('The full version of TicketPayGo is already active. TicketPayGo Lite has been disabled to prevent conflicts. Please deactivate one of the plugins.', 'ticketpaygo-lite');
        echo '</p></div>';
    });
    return;
}

// Define plugin constants with unique prefix for Lite
define('TPGL_VERSION', '1.4.1');
define('TPGL_PLUGIN_URL', plugin_dir_url(__FILE__));
define('TPGL_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('TPGL_PLUGIN_FILE', __FILE__);
define('TPGL_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Legacy constants for compatibility with includes
define('TICKETPAYGO_VERSION', TPGL_VERSION);
define('TICKETPAYGO_PLUGIN_URL', TPGL_PLUGIN_URL);
define('TICKETPAYGO_PLUGIN_PATH', TPGL_PLUGIN_PATH);
define('TICKETPAYGO_PLUGIN_FILE', TPGL_PLUGIN_FILE);
define('TICKETPAYGO_PLUGIN_BASENAME', TPGL_PLUGIN_BASENAME);

/**
 * Main TicketPayGo Lite Class
 */
class TicketPayGo_Lite {
    
    /**
     * Single instance of the class
     */
    private static $instance = null;
    
    /**
     * Get single instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->init_hooks();
    }
    
    /**
     * Initialize hooks
     */
    private function init_hooks() {
        add_action('init', array($this, 'init'));
        
        // Activation and deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        
        // Admin hooks
        if (is_admin()) {
            add_action('admin_menu', array($this, 'add_admin_menu'));
            add_action('admin_enqueue_scripts', array($this, 'admin_enqueue_scripts'));
        }
        
        // Frontend hooks - only enqueue on pages with our shortcodes
        add_action('wp_enqueue_scripts', array($this, 'frontend_enqueue_scripts'));
        add_action('wp_ajax_tpgl_purchase', array($this, 'handle_ticket_purchase'));
        add_action('wp_ajax_nopriv_tpgl_purchase', array($this, 'handle_ticket_purchase'));
        
        // Shortcodes (Lite-specific to avoid conflicts with Full version)
        add_shortcode('ticketpaygo_lite_event', array($this, 'event_shortcode'));
        add_shortcode('ticketpaygo_lite_events', array($this, 'events_list_shortcode'));
        add_shortcode('ticketpaygo_lite_ticket', array($this, 'ticket_shortcode'));
        add_shortcode('ticketpaygo_lite_calendar', array($this, 'calendar_shortcode'));
        
        // REST API
        add_action('rest_api_init', array($this, 'register_rest_routes'));
        
        // Cron hooks for automated email reminders
        add_action('tpgl_daily_reminders', array($this, 'send_daily_reminders'));
        
        // Elementor integration
        add_action('elementor/loaded', array($this, 'init_elementor'));
        
        // Elementor detection notice
        add_action('admin_notices', array($this, 'elementor_integration_notice'));
    }
    
    /**
     * Send daily event reminders via cron
     */
    public function send_daily_reminders() {
        if (class_exists('TicketPayGo_Email')) {
            $email = new TicketPayGo_Email();
            $email->schedule_reminders();
        }
    }
    
    /**
     * Initialize plugin
     */
    public function init() {
        // Load required files
        $this->load_dependencies();
        
        // Initialize components
        $this->init_components();
    }
    
    /**
     * Load plugin dependencies
     */
    private function load_dependencies() {
        // Core files only - no premium modules
        $required_files = array(
            'includes/class-database.php',
            'includes/class-event.php',
            'includes/class-ticket.php',
            'includes/class-order.php',
            'includes/class-payment.php',
            'includes/class-qr-generator.php',
            'includes/class-email.php',
            'includes/class-admin.php',
            'includes/class-frontend.php',
            'includes/class-api.php',
            'includes/class-calendar-links.php'
        );
        
        foreach ($required_files as $file) {
            $file_path = TPGL_PLUGIN_PATH . $file;
            if (file_exists($file_path)) {
                require_once $file_path;
            }
        }
    }
    
    /**
     * Initialize plugin components
     */
    private function init_components() {
        // Only initialize components if classes exist
        if (class_exists('TicketPayGo_Database')) {
            new TicketPayGo_Database();
        }
        
        if (class_exists('TicketPayGo_Admin')) {
            new TicketPayGo_Admin();
        }
        
        if (class_exists('TicketPayGo_Frontend')) {
            new TicketPayGo_Frontend();
        }
        
        if (class_exists('TicketPayGo_Lite_API')) {
            new TicketPayGo_Lite_API();
        }
        
        // Check and create missing database tables on admin init
        if (is_admin() && class_exists('TicketPayGo_Database')) {
            if (!TicketPayGo_Database::tables_exist()) {
                TicketPayGo_Database::create_tables();
            }
        }
    }
    
    /**
     * Plugin activation
     */
    public function activate() {
        // Check if Full version is active
        if (is_plugin_active('ticketpaygo/ticketpaygo.php')) {
            deactivate_plugins(plugin_basename(__FILE__));
            wp_die(
                esc_html__('TicketPayGo Full version is already active. Please deactivate it before activating TicketPayGo Lite.', 'ticketpaygo-lite'),
                esc_html__('Plugin Activation Error', 'ticketpaygo-lite'),
                array('back_link' => true)
            );
        }
        
        // Load database class first
        require_once TPGL_PLUGIN_PATH . 'includes/class-database.php';
        
        // Create database tables with error handling
        try {
            $results = TicketPayGo_Database::create_tables();
            
            // Verify tables were created
            if (!TicketPayGo_Database::tables_exist()) {
                if ( defined('WP_DEBUG') && WP_DEBUG ) {
                    error_log('TicketPayGo Lite: Warning - Some tables may not have been created properly during activation'); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Only used when WP_DEBUG is enabled
                }
            }
        } catch (Exception $e) {
            if ( defined('WP_DEBUG') && WP_DEBUG ) {
                error_log('TicketPayGo Lite: Database creation failed during activation: ' . $e->getMessage()); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Only used when WP_DEBUG is enabled
            }
        }
        
        // Set default capabilities
        $role = get_role('administrator');
        if ($role) {
            $role->add_cap('manage_ticketpaygo_lite');
            $role->add_cap('edit_ticketpaygo_lite_events');
            $role->add_cap('delete_ticketpaygo_lite_events');
        }
        
        // Schedule daily email reminders cron job
        if (!wp_next_scheduled('tpgl_daily_reminders')) {
            wp_schedule_event(time(), 'daily', 'tpgl_daily_reminders');
        }
        
        // Flush rewrite rules
        flush_rewrite_rules();
        
        // Set plugin version
        update_option('tpgl_version', TPGL_VERSION);
        
        // Set database version
        update_option('tpgl_db_version', '1.0.0');
        
        // Set activation flag
        update_option('tpgl_activated', true);
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Clear scheduled cron jobs
        wp_clear_scheduled_hook('tpgl_daily_reminders');
        
        // Flush rewrite rules
        flush_rewrite_rules();
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            __('TicketPayGo Lite', 'ticketpaygo-lite'),
            __('TicketPayGo Lite', 'ticketpaygo-lite'),
            'manage_options',
            'ticketpaygo-lite',
            array($this, 'admin_dashboard'),
            'dashicons-tickets-alt',
            30
        );
        
        add_submenu_page(
            'ticketpaygo-lite',
            __('Dashboard', 'ticketpaygo-lite'),
            __('Dashboard', 'ticketpaygo-lite'),
            'manage_options',
            'ticketpaygo-lite',
            array($this, 'admin_dashboard')
        );
        
        add_submenu_page(
            'ticketpaygo-lite',
            __('Events', 'ticketpaygo-lite'),
            __('Events', 'ticketpaygo-lite'),
            'manage_options',
            'ticketpaygo-lite-events',
            array($this, 'admin_events')
        );
        
        add_submenu_page(
            'ticketpaygo-lite',
            __('Orders', 'ticketpaygo-lite'),
            __('Orders', 'ticketpaygo-lite'),
            'manage_options',
            'ticketpaygo-lite-orders',
            array($this, 'admin_orders')
        );
        
        add_submenu_page(
            'ticketpaygo-lite',
            __('Check-in', 'ticketpaygo-lite'),
            __('Check-in', 'ticketpaygo-lite'),
            'manage_options',
            'ticketpaygo-lite-checkin',
            array($this, 'admin_checkin')
        );
        
        add_submenu_page(
            'ticketpaygo-lite',
            __('Settings', 'ticketpaygo-lite'),
            __('Settings', 'ticketpaygo-lite'),
            'manage_options',
            'ticketpaygo-lite-settings',
            array($this, 'admin_settings')
        );
        
        add_submenu_page(
            'ticketpaygo-lite',
            __('Upgrade', 'ticketpaygo-lite'),
            __('Upgrade', 'ticketpaygo-lite'),
            'manage_options',
            'ticketpaygo-lite-upgrade',
            array($this, 'admin_upgrade')
        );
    }
    
    /**
     * Admin dashboard page
     */
    public function admin_dashboard() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'ticketpaygo-lite'));
        }
        include TPGL_PLUGIN_PATH . 'admin/dashboard.php';
    }
    
    /**
     * Admin events page
     */
    public function admin_events() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'ticketpaygo-lite'));
        }
        include TPGL_PLUGIN_PATH . 'admin/events.php';
    }
    
    /**
     * Admin orders page
     */
    public function admin_orders() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'ticketpaygo-lite'));
        }
        include TPGL_PLUGIN_PATH . 'admin/orders.php';
    }
    
    /**
     * Admin check-in page
     */
    public function admin_checkin() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'ticketpaygo-lite'));
        }
        include TPGL_PLUGIN_PATH . 'admin/checkin.php';
    }
    
    /**
     * Admin settings page
     */
    public function admin_settings() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'ticketpaygo-lite'));
        }
        include TPGL_PLUGIN_PATH . 'admin/settings.php';
    }
    
    /**
     * Admin upgrade page
     */
    public function admin_upgrade() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'ticketpaygo-lite'));
        }
        include TPGL_PLUGIN_PATH . 'admin/upgrade.php';
    }
    
    /**
     * Enqueue admin scripts and styles
     */
    public function admin_enqueue_scripts($hook) {
        // Only load on plugin pages
        if (strpos($hook, 'ticketpaygo-lite') === false) {
            return;
        }
        
        wp_enqueue_style('ticketpaygo-lite-admin', TPGL_PLUGIN_URL . 'assets/css/admin.css', array(), TPGL_VERSION);
        wp_enqueue_script('ticketpaygo-lite-admin', TPGL_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), TPGL_VERSION, true);
        
        // Enqueue color picker for settings page
        if ($hook === 'ticketpaygo-lite_page_ticketpaygo-lite-settings') {
            wp_enqueue_style('wp-color-picker');
            wp_enqueue_script('wp-color-picker');
            wp_enqueue_media();
        }
        
        wp_localize_script('ticketpaygo-lite-admin', 'ticketpaygo_lite_admin', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('ticketpaygo_lite_admin_nonce'),
            'strings' => array(
                'select_image' => __('Select Featured Image', 'ticketpaygo-lite'),
                'use_image' => __('Use this image', 'ticketpaygo-lite'),
                'invalid_dates' => __('End date must be after start date', 'ticketpaygo-lite')
            )
        ));
    }
    
    /**
     * Enqueue frontend scripts and styles
     */
    public function frontend_enqueue_scripts() {
        global $post;
        
        // Only enqueue on pages with our shortcodes
        if (!is_a($post, 'WP_Post')) {
            return;
        }
        
        $has_shortcode = has_shortcode($post->post_content, 'ticketpaygo_lite_event') ||
                         has_shortcode($post->post_content, 'ticketpaygo_lite_events') ||
                         has_shortcode($post->post_content, 'ticketpaygo_lite_ticket') ||
                         has_shortcode($post->post_content, 'ticketpaygo_lite_calendar');
        
        if (!$has_shortcode) {
            return;
        }
        
        // Enqueue Leaflet for maps (bundled locally for WordPress.org compliance)
        wp_enqueue_style('leaflet-css', TPGL_PLUGIN_URL . 'assets/css/leaflet.css', array(), '1.9.4');
        wp_enqueue_script('leaflet-js', TPGL_PLUGIN_URL . 'assets/js/leaflet.js', array(), '1.9.4', false);
        
        // Enqueue event card styles
        wp_enqueue_style('tpgl-event-cards', TPGL_PLUGIN_URL . 'assets/css/event-cards.css', array(), TPGL_VERSION);
        
        wp_enqueue_script('tpgl-frontend', TPGL_PLUGIN_URL . 'assets/js/frontend.js', array('jquery', 'leaflet-js'), TPGL_VERSION, true);
        wp_enqueue_style('tpgl-frontend', TPGL_PLUGIN_URL . 'assets/css/frontend.css', array('leaflet-css', 'tpgl-event-cards'), TPGL_VERSION);
        
        // Enqueue event modal script
        wp_enqueue_script('tpgl-event-modal', TPGL_PLUGIN_URL . 'assets/js/event-modal.js', array(), TPGL_VERSION, true);
        
        // Add dynamic styles for event list
        if (class_exists('TicketPayGo_Database')) {
            $card_primary_color = TicketPayGo_Database::get_setting('card_primary_color', '#0A4974');
            $card_button_color = TicketPayGo_Database::get_setting('card_button_color', '#42963B');
            $card_button_text_color = TicketPayGo_Database::get_setting('card_button_text_color', '#FFFFFF');
            $card_text_color = TicketPayGo_Database::get_setting('card_text_color', '#333333');
            $modal_button_color = TicketPayGo_Database::get_setting('modal_button_color', '#3b82f6');
            
            // Sanitize color values - only allow valid hex colors
            $card_primary_color = self::sanitize_hex_color($card_primary_color, '#0A4974');
            $card_button_color = self::sanitize_hex_color($card_button_color, '#42963B');
            $card_button_text_color = self::sanitize_hex_color($card_button_text_color, '#FFFFFF');
            $card_text_color = self::sanitize_hex_color($card_text_color, '#333333');
            $modal_button_color = self::sanitize_hex_color($modal_button_color, '#3b82f6');
            
            $custom_css = "
                .ticketpaygo-events-list .event-card-compact .event-overlay {
                    background: linear-gradient(to top, " . esc_attr($card_primary_color) . ", transparent);
                }
                .ticketpaygo-events-list .event-title {
                    color: white !important;
                }
                .ticketpaygo-events-list .event-quick-info {
                    color: white !important;
                }
                .ticketpaygo-events-list .event-quick-info svg {
                    color: white !important;
                }
                .ticketpaygo-events-list .event-details-panel {
                    border-top: 3px solid " . esc_attr($card_primary_color) . ";
                }
                .ticketpaygo-events-list .event-meta-item svg {
                    color: " . esc_attr($card_primary_color) . ";
                }
                .ticketpaygo-events-list .event-description {
                    color: " . esc_attr($card_text_color) . ";
                }
                .ticketpaygo-events-list .btn-get-tickets {
                    background: " . esc_attr($card_button_color) . " !important;
                    border-color: " . esc_attr($card_button_color) . " !important;
                    color: " . esc_attr($card_button_text_color) . " !important;
                }
                .ticketpaygo-events-list .btn-get-tickets:hover {
                    background: " . esc_attr($card_button_color) . "dd !important;
                    color: " . esc_attr($card_button_text_color) . " !important;
                    transform: translateY(-2px);
                }
                
                /* Event Card Template Styles */
                .ticketpaygo-card-button {
                    background: " . esc_attr($card_button_color) . " !important;
                    border-color: " . esc_attr($card_button_color) . " !important;
                    color: " . esc_attr($card_button_text_color) . " !important;
                }
                .ticketpaygo-card-button:hover {
                    background: " . esc_attr($card_button_color) . "dd !important;
                    color: " . esc_attr($card_button_text_color) . " !important;
                }
                .ticketpaygo-modal-content {
                    color: " . esc_attr($card_text_color) . ";
                }
                .ticketpaygo-modal-header h2 {
                    color: " . esc_attr($card_primary_color) . ";
                }
                .info-icon {
                    color: " . esc_attr($card_primary_color) . ";
                }
                .ticketpaygo-book-now {
                    background: " . esc_attr($card_button_color) . " !important;
                    border-color: " . esc_attr($card_button_color) . " !important;
                    color: " . esc_attr($card_button_text_color) . " !important;
                }
                .ticketpaygo-book-now:hover {
                    background: " . esc_attr($card_button_color) . "dd !important;
                    color: " . esc_attr($card_button_text_color) . " !important;
                }
                .ticket-type-item {
                    border-color: " . esc_attr($card_primary_color) . "33;
                }
                .ticket-type-item.selected {
                    border-color: " . esc_attr($card_primary_color) . ";
                    background: " . esc_attr($card_primary_color) . "08;
                }
                .ticket-type-price {
                    color: " . esc_attr($card_primary_color) . ";
                    font-weight: 600;
                }
                .qty-btn {
                    background: " . esc_attr($card_button_color) . ";
                    color: " . esc_attr($card_button_text_color) . ";
                }
                .qty-btn:hover {
                    background: " . esc_attr($card_button_color) . "dd;
                    color: " . esc_attr($card_button_text_color) . ";
                }
                .ticket-total {
                    background: " . esc_attr($card_primary_color) . ";
                }
                
                /* Checkout Modal Button */
                .btn-checkout-submit {
                    background: " . esc_attr($modal_button_color) . " !important;
                }
                .btn-checkout-submit:hover {
                    background: " . esc_attr($modal_button_color) . "dd !important;
                }
            ";
            wp_add_inline_style('tpgl-frontend', $custom_css);
        }
        
        wp_localize_script('tpgl-frontend', 'ticketpaygo', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('tpgl_nonce'),
            'currency_symbol' => get_option('tpgl_currency_symbol', '€')
        ));
    }
    
    /**
     * Event shortcode
     */
    public function event_shortcode($atts) {
        $atts = shortcode_atts(array(
            'id' => 0,
            'layout' => 'default'
        ), $atts);
        
        if (!$atts['id']) {
            return '<p>' . esc_html__('Event ID is required.', 'ticketpaygo-lite') . '</p>';
        }
        
        if (!class_exists('TicketPayGo_Event')) {
            return '<p>' . esc_html__('Event system not available.', 'ticketpaygo-lite') . '</p>';
        }
        
        $event = new TicketPayGo_Event($atts['id']);
        if (!$event->exists()) {
            return '<p>' . esc_html__('Event not found.', 'ticketpaygo-lite') . '</p>';
        }
        
        ob_start();
        include TPGL_PLUGIN_PATH . 'templates/event-' . sanitize_file_name($atts['layout']) . '.php';
        return ob_get_clean();
    }
    
    /**
     * Events list shortcode
     */
    public function events_list_shortcode($atts) {
        $atts = shortcode_atts(array(
            'limit' => 10,
            'layout' => 'grid',
            'status' => 'active',
            'upcoming' => 'true'
        ), $atts);
        
        if (!class_exists('TicketPayGo_Event')) {
            return '<p>' . esc_html__('Event system not available.', 'ticketpaygo-lite') . '</p>';
        }
        
        // Get events
        if ($atts['upcoming'] === 'true') {
            $events = TicketPayGo_Event::get_upcoming_events($atts['limit']);
        } else {
            $events = TicketPayGo_Event::get_all_events($atts['limit'], $atts['status']);
        }
        
        // Set variables for template
        $limit = $atts['limit'];
        $layout = $atts['layout'];
        
        ob_start();
        include TPGL_PLUGIN_PATH . 'templates/events-list.php';
        return ob_get_clean();
    }
    
    /**
     * Ticket shortcode
     */
    public function ticket_shortcode($atts) {
        $atts = shortcode_atts(array(
            'order_id' => 0,
            'ticket_id' => 0
        ), $atts);
        
        if (!$atts['order_id'] || !$atts['ticket_id']) {
            return '<p>' . esc_html__('Order ID and Ticket ID are required.', 'ticketpaygo-lite') . '</p>';
        }
        
        if (!class_exists('TicketPayGo_Ticket')) {
            return '<p>' . esc_html__('Ticket system not available.', 'ticketpaygo-lite') . '</p>';
        }
        
        $ticket = new TicketPayGo_Ticket($atts['ticket_id']);
        if (!$ticket->exists()) {
            return '<p>' . esc_html__('Ticket not found.', 'ticketpaygo-lite') . '</p>';
        }
        
        ob_start();
        include TPGL_PLUGIN_PATH . 'templates/ticket.php';
        return ob_get_clean();
    }
    
    /**
     * Calendar shortcode
     */
    public function calendar_shortcode($atts) {
        $atts = shortcode_atts(array(
            'view' => 'dayGridMonth',
            'category' => '',
            'filters' => 'true'
        ), $atts);
        
        // Enqueue FullCalendar (bundled locally for WordPress.org compliance)
        wp_enqueue_style(
            'fullcalendar',
            TPGL_PLUGIN_URL . 'assets/css/fullcalendar.min.css',
            array(),
            '6.1.20'
        );
        wp_enqueue_script(
            'fullcalendar',
            TPGL_PLUGIN_URL . 'assets/js/fullcalendar.min.js',
            array(),
            '6.1.20',
            true
        );

        // Enqueue Calendar View assets
        wp_enqueue_style(
            'ticketpaygo-calendar-view',
            TPGL_PLUGIN_URL . 'assets/css/calendar-view.css',
            array('fullcalendar'),
            TPGL_VERSION
        );
        wp_enqueue_script(
            'ticketpaygo-calendar-view',
            TPGL_PLUGIN_URL . 'assets/js/calendar-view.js',
            array('fullcalendar', 'jquery'),
            TPGL_VERSION,
            true
        );

        // Localize script
        wp_localize_script('ticketpaygo-calendar-view', 'ticketpaygo_calendar', array(
            'locale' => get_locale(),
            'events_url' => rest_url('ticketpaygo-lite/v1/events'), // Updated to match the registered namespace
            'strings' => array(
                'location_tba' => __('Location TBA', 'ticketpaygo-lite'),
                'book_tickets' => __('Book Tickets', 'ticketpaygo-lite'),
                'view_details' => __('View Details', 'ticketpaygo-lite'),
                'close' => __('Close', 'ticketpaygo-lite')
            )
        ));
        
        ob_start();
        include TPGL_PLUGIN_PATH . 'templates/calendar-view.php';
        return ob_get_clean();
    }
    
    /**
     * Handle ticket purchase AJAX
     */
    public function handle_ticket_purchase() {
        check_ajax_referer('tpgl_nonce', 'nonce');
        
        if ( ! isset($_POST['event_id'], $_POST['quantity'], $_POST['customer_data'], $_POST['payment_method']) ) {
            wp_send_json(array('success' => false, 'message' => __('Missing required fields.', 'ticketpaygo-lite')));
            return;
        }
        
        $event_id = absint( wp_unslash( $_POST['event_id'] ) );
        $quantity = absint( wp_unslash( $_POST['quantity'] ) );
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized on next line
        $customer_data_raw = wp_unslash( $_POST['customer_data'] );
        $customer_data = is_array($customer_data_raw) ? array_map('sanitize_text_field', $customer_data_raw) : sanitize_text_field($customer_data_raw);
        $payment_method = sanitize_key( wp_unslash( $_POST['payment_method'] ) );
        
        if (!class_exists('TicketPayGo_Order')) {
            wp_send_json(array('success' => false, 'message' => __('Order system not available.', 'ticketpaygo-lite')));
            return;
        }
        
        // Process the purchase
        $order = new TicketPayGo_Order();
        $result = $order->create_order($event_id, $quantity, $customer_data, $payment_method);
        
        wp_send_json($result);
    }
    
    /**
     * Initialize Elementor integration
     */
    public function init_elementor() {
        // Load Elementor loader
        require_once TPGL_PLUGIN_PATH . 'includes/elementor/class-elementor-loader.php';
        
        // Initialize Elementor integration
        TicketPayGo_Lite_Elementor_Loader::init();
    }
    
    /**
     * Display Elementor integration notice when Elementor is detected
     */
    public function elementor_integration_notice() {
        if (!is_admin()) {
            return;
        }
        
        if (defined('DOING_AJAX') && DOING_AJAX) {
            return;
        }
        
        // Check if Elementor is active
        if (!did_action('elementor/loaded')) {
            return;
        }
        
        $screen = get_current_screen();
        if (!$screen) {
            return;
        }
        
        // Only show on TicketPayGo Lite pages
        if (strpos($screen->id, 'ticketpaygo') === false) {
            return;
        }
        
        // Check if notice was dismissed
        $dismissed = get_option('tpgl_elementor_notice_dismissed', false);
        if ($dismissed) {
            return;
        }
        
        // Check if this is the first time showing (set a flag)
        $shown_before = get_option('tpgl_elementor_notice_shown', false);
        if ($shown_before) {
            return;
        }
        
        // Mark as shown
        update_option('tpgl_elementor_notice_shown', true);
        
        ?>
        <div class="notice notice-info is-dismissible tpgl-elementor-notice" style="border-left-color: #92003B; padding: 15px;">
            <div style="display: flex; align-items: flex-start; gap: 15px;">
                <div style="flex-shrink: 0;">
                    <svg width="48" height="48" viewBox="0 0 48 48" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <rect width="48" height="48" rx="8" fill="#92003B"/>
                        <path d="M16 14H20V34H16V14Z" fill="white"/>
                        <path d="M24 14H32V18H24V14Z" fill="white"/>
                        <path d="M24 22H32V26H24V22Z" fill="white"/>
                        <path d="M24 30H32V34H24V30Z" fill="white"/>
                    </svg>
                </div>
                <div style="flex-grow: 1;">
                    <h3 style="margin: 0 0 8px 0; font-size: 16px; color: #1d2327;">
                        🎉 <?php esc_html_e('Elementor Integration Detected!', 'ticketpaygo-lite'); ?>
                    </h3>
                    <p style="margin: 0 0 12px 0; color: #50575e;">
                        <?php esc_html_e('Great news! TicketPayGo Lite has detected Elementor on your site and automatically enabled custom widgets for you.', 'ticketpaygo-lite'); ?>
                    </p>
                    <p style="margin: 0 0 12px 0; color: #50575e;">
                        <?php esc_html_e('You can now use the following widgets in Elementor:', 'ticketpaygo-lite'); ?>
                    </p>
                    <ul style="margin: 0 0 12px 0; padding-left: 20px; color: #50575e;">
                        <li><strong><?php esc_html_e('Event Card', 'ticketpaygo-lite'); ?></strong> - <?php esc_html_e('Display a single event', 'ticketpaygo-lite'); ?></li>
                        <li><strong><?php esc_html_e('Events Grid', 'ticketpaygo-lite'); ?></strong> - <?php esc_html_e('Show multiple events in a grid or list layout', 'ticketpaygo-lite'); ?></li>
                    </ul>
                    <p style="margin: 0; color: #50575e;">
                        <?php 
                        printf(
                            wp_kses(
                                __('Find them in the Elementor panel under the <strong>TicketPayGo Lite</strong> category.', 'ticketpaygo-lite'),
                                array('strong' => array())
                            )
                        ); 
                        ?>
                    </p>
                </div>
            </div>
        </div>
        <script>
        jQuery(document).ready(function($) {
            $('.tpgl-elementor-notice').on('click', '.notice-dismiss', function() {
                $.ajax({
                    url: ajaxurl,
                    data: {
                        action: 'tpgl_dismiss_elementor_notice',
                        nonce: '<?php echo esc_js(wp_create_nonce('tpgl_elementor_nonce')); ?>'
                    }
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * Sanitize a hex color value
     * 
     * @param string $color The color value to sanitize
     * @param string $default The default color to return if invalid
     * @return string Sanitized hex color
     */
    private static function sanitize_hex_color($color, $default = '#000000') {
        // Remove any whitespace
        $color = trim($color);
        
        // Check if it's a valid hex color (3 or 6 characters, with or without #)
        if (preg_match('/^#?([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/', $color, $matches)) {
            // Ensure it starts with #
            return '#' . $matches[1];
        }
        
        // Return default if invalid
        return $default;
    }
}

// Initialize the plugin
function tpgl_init() {
    return TicketPayGo_Lite::get_instance();
}

// Start the plugin
add_action('plugins_loaded', 'tpgl_init', 20);

/**
 * AJAX handler for dismissing Elementor notice
 */
function tpgl_dismiss_elementor_notice() {
    if (!isset($_REQUEST['nonce']) || !wp_verify_nonce($_REQUEST['nonce'], 'tpgl_elementor_nonce')) {
        wp_send_json_error();
        return;
    }
    
    update_option('tpgl_elementor_notice_dismissed', true);
    wp_send_json_success();
}
add_action('wp_ajax_tpgl_dismiss_elementor_notice', 'tpgl_dismiss_elementor_notice');

// Helper functions
function tpgl_get_event($event_id) {
    if (!class_exists('TicketPayGo_Event')) {
        return null;
    }
    return new TicketPayGo_Event($event_id);
}

function tpgl_get_ticket($ticket_id) {
    if (!class_exists('TicketPayGo_Ticket')) {
        return null;
    }
    return new TicketPayGo_Ticket($ticket_id);
}

function tpgl_get_order($order_id) {
    if (!class_exists('TicketPayGo_Order')) {
        return null;
    }
    return new TicketPayGo_Order($order_id);
}
