<?php
/**
 * Modern Events List Template - 16:9 Card Grid
 * 
 * Available variables:
 * $events - Array of TicketPayGo_Event objects
 * $layout - Layout type (grid or list)
 * 
 * phpcs:disable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
 */

if (!defined('ABSPATH')) {
    exit;
}

$layout = isset($layout) ? $layout : 'grid';

// Get custom colors from settings
$card_primary_color = TicketPayGo_Database::get_setting('card_primary_color', '#0A4974');
$card_button_color = TicketPayGo_Database::get_setting('card_button_color', '#42963B');
$card_button_text_color = TicketPayGo_Database::get_setting('card_button_text_color', '#FFFFFF');
$card_text_color = TicketPayGo_Database::get_setting('card_text_color', '#333333');
?>

<div class="ticketpaygo-events-list layout-<?php echo esc_attr($layout); ?>">

    <?php if (empty($events)): ?>
        <div class="no-events">
            <div class="no-events-content">
                <svg width="80" height="80" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M19 3H18V1H16V3H8V1H6V3H5C3.89 3 3.01 3.9 3.01 5L3 19C3 20.1 3.89 21 5 21H19C20.1 21 21 20.1 21 19V5C21 3.9 20.1 3 19 3ZM19 19H5V8H19V19Z" fill="currentColor" opacity="0.3"/>
                </svg>
                <h3><?php esc_html_e('No Events Found', 'ticketpaygo-lite'); ?></h3>
                <p><?php esc_html_e('There are currently no events available. Check back soon for upcoming events!', 'ticketpaygo-lite'); ?></p>
            </div>
        </div>
    <?php else: ?>
        <div class="events-grid">
            <?php foreach ($events as $event): 
                // Handle both object and array data
                if (is_object($event) && method_exists($event, 'get_data')) {
                    $event_data = $event->get_data();
                } elseif (is_array($event)) {
                    $event_data = $event;
                } else {
                    continue; // Skip invalid events
                }
                
                if (empty($event_data) || !isset($event_data['id'])) {
                    continue; // Skip events without proper data
                }
                
                $event_id = $event_data['id'];
            ?>
                <div class="ticketpaygo-event-card-list" id="event-list-<?php echo esc_attr($event_id); ?>">
                    <!-- Compact Card View -->
                    <div class="event-card-compact" onclick="toggleEventDetails(<?php echo esc_attr($event_id); ?>);return false;">
                        <div class="event-image-container">
                            <?php if (!empty($event_data['featured_image'])): ?>
                                <img src="<?php echo esc_url($event_data['featured_image']); ?>" alt="<?php echo esc_attr($event_data['title']); ?>" class="event-image" />
                            <?php else: ?>
                                <div class="event-image-placeholder">
                                    <svg width="60" height="60" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <path d="M19 3H5C3.9 3 3 3.9 3 5V19C3 20.1 3.9 21 5 21H19C20.1 21 21 20.1 21 19V5C21 3.9 20.1 3 19 3ZM19 19H5V5H19V19Z" fill="currentColor"/>
                                        <path d="M13.96 12.17L11.06 14.47L9.23 12.28L6.5 15.5H17.5L13.96 12.17Z" fill="currentColor"/>
                                    </svg>
                                </div>
                            <?php endif; ?>
                            
                            <div class="event-overlay">
                                <h3 class="event-title"><?php echo esc_html($event_data['title']); ?></h3>
                                
                                <div class="event-quick-info">
                                    <div class="info-item">
                                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M19 3H18V1H16V3H8V1H6V3H5C3.89 3 3.01 3.9 3.01 5L3 19C3 20.1 3.89 21 5 21H19C20.1 21 21 20.1 21 19V5C21 3.9 20.1 3 19 3ZM19 19H5V8H19V19Z" fill="currentColor"/>
                                        </svg>
                                        <span><?php echo esc_html(date_i18n('M j, Y', strtotime($event_data['start_date']))); ?></span>
                                    </div>
                                    
                                    <?php if (!empty($event_data['location'])): ?>
                                        <div class="info-item">
                                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                <path d="M12 2C8.13 2 5 5.13 5 9C5 14.25 12 22 12 22S19 14.25 19 9C19 5.13 15.87 2 12 2ZM12 11.5C10.62 11.5 9.5 10.38 9.5 9S10.62 6.5 12 6.5S14.5 7.62 14.5 9S13.38 11.5 12 11.5Z" fill="currentColor"/>
                                            </svg>
                                            <span><?php echo esc_html($event_data['location']); ?></span>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <div class="info-item price">
                                        <?php 
                                        // Get currency symbol from settings
                                        $currency_symbol = TicketPayGo_Database::get_setting('currency_symbol', '€');
                                        $currency_position = TicketPayGo_Database::get_setting('currency_position', 'before');
                                        ?>
                                        <span class="price-value">
                                            <?php if ($event_data['price'] > 0): ?>
                                                <?php 
                                                $formatted_price = number_format($event_data['price'], 2);
                                                if ($currency_position === 'before') {
                                                    echo esc_html($currency_symbol . ' ' . $formatted_price);
                                                } else {
                                                    echo esc_html($formatted_price . ' ' . $currency_symbol);
                                                }
                                                ?>
                                            <?php else: ?>
                                                <?php esc_html_e('Free', 'ticketpaygo-lite'); ?>
                                            <?php endif; ?>
                                        </span>
                                    </div>
                                    
                                    <?php if ($event_data['max_tickets'] > 0): ?>
                                        <div class="info-item">
                                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                <path d="M22 10V6C22 4.9 21.1 4 20 4H4C2.9 4 2 4.9 2 6V10C3.1 10 4 10.9 4 12S3.1 14 2 14V18C2 19.1 2.9 20 4 20H20C21.1 20 22 19.1 22 18V14C20.9 14 20 13.1 20 12S20.9 10 22 10ZM12 11.5C10.62 11.5 9.5 10.38 9.5 9S10.62 6.5 12 6.5S14.5 7.62 14.5 9S13.38 11.5 12 11.5Z" fill="currentColor"/>
                                                <path d="M22 10V6C22 4.9 21.1 4 20 4H4C2.9 4 2 4.9 2 6V10C3.1 10 4 10.9 4 12S3.1 14 2 14V18C2 19.1 2.9 20 4 20H20C21.1 20 22 19.1 22 18V14C20.9 14 20 13.1 20 12S20.9 10 22 10ZM13 17.5H11V16.5H13V17.5ZM13 15.5H11V14.5H13V15.5ZM13 13.5H11V12.5H13V13.5ZM13 11.5H11V10.5H13V11.5ZM13 9.5H11V8.5H13V9.5ZM13 7.5H11V6.5H13V7.5Z" fill="currentColor"/>
                                            </svg>
                                            <span><?php echo intval($event_data['max_tickets']); ?> <?php esc_html_e('tickets', 'ticketpaygo-lite'); ?></span>
                                        </div>
                                    <?php endif; ?>
                                </div>
                                
                                <div class="expand-indicator">
                                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <path d="M7.41 8.59L12 13.17L16.59 8.59L18 10L12 16L6 10L7.41 8.59Z" fill="currentColor"/>
                                    </svg>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Expanded Details View -->
                    <div class="event-details-expanded" id="details-list-<?php echo esc_attr($event_id); ?>" style="display: none;">
                        <?php if (!empty($event_data['description'])): ?>
                            <div class="event-section">
                                <h4>
                                    <svg width="18" height="18" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <path d="M14 2H6C4.9 2 4.01 2.9 4.01 4L4 20C4 21.1 4.89 22 6 22H18C19.1 22 20 21.1 20 20V8L14 2ZM18 20H6V4H13V9H18V20Z" fill="currentColor"/>
                                    </svg>
                                    <?php esc_html_e('About This Event', 'ticketpaygo-lite'); ?>
                                </h4>
                                <div class="section-content">
                                    <?php echo wp_kses_post($event_data['description']); ?>
                                </div>
                            </div>
                        <?php endif; ?>
                        
                        <div class="event-details-grid">
                            <div class="detail-column">
                                <?php if (!empty($event_data['address'])): ?>
                                    <div class="event-section">
                                        <h4>
                                            <svg width="18" height="18" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                <path d="M12 2C8.13 2 5 5.13 5 9C5 14.25 12 22 12 22S19 14.25 19 9C19 5.13 15.87 2 12 2ZM12 11.5C10.62 11.5 9.5 10.38 9.5 9S10.62 6.5 12 6.5S14.5 7.62 14.5 9S13.38 11.5 12 11.5Z" fill="currentColor"/>
                                            </svg>
                                            <?php esc_html_e('Venue Details', 'ticketpaygo-lite'); ?>
                                        </h4>
                                        <div class="section-content">
                                            <p><?php echo esc_html($event_data['address']); ?></p>
                                        </div>
                                    </div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="detail-column">
                                <?php if (!empty($event_data['organizer_name'])): ?>
                                    <div class="event-section">
                                        <h4>
                                            <svg width="18" height="18" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                <path d="M12 12C14.21 12 16 10.21 16 8S14.21 4 12 4S8 5.79 8 8S9.79 12 12 12ZM12 14C9.33 14 4 15.34 4 18V20H20V18C20 15.34 14.67 14 12 14Z" fill="currentColor"/>
                                            </svg>
                                            <?php esc_html_e('Organizer', 'ticketpaygo-lite'); ?>
                                        </h4>
                                        <div class="section-content">
                                            <p>
                                                <strong><?php echo esc_html($event_data['organizer_name']); ?></strong>
                                                <?php if (!empty($event_data['organizer_email'])): ?>
                                                    <br><a href="mailto:<?php echo esc_attr($event_data['organizer_email']); ?>"><?php echo esc_html($event_data['organizer_email']); ?></a>
                                                <?php endif; ?>
                                                <?php if (!empty($event_data['organizer_phone'])): ?>
                                                    <br><?php echo esc_html($event_data['organizer_phone']); ?>
                                                <?php endif; ?>
                                            </p>
                                        </div>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div class="event-booking-section">
                            <button class="ticketpaygo-book-now" data-event-id="<?php echo esc_attr($event_id); ?>">
                                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M22 10V6C22 4.9 21.1 4 20 4H4C2.9 4 2 4.9 2 6V10C3.1 10 4 10.9 4 12S3.1 14 2 14V18C2 19.1 2.9 20 4 20H20C21.1 20 22 19.1 22 18V14C20.9 14 20 13.1 20 12S20.9 10 22 10Z" fill="currentColor"/>
                                </svg>
<?php esc_html_e('Get Tickets', 'ticketpaygo-lite'); ?>
                            </button>
                        </div>
                    </div>
                </div>
            <?php 
                // Include booking modal for this event
                include TICKETPAYGO_PLUGIN_PATH . 'templates/booking-modal.php';
            endforeach; ?>
        </div>
    <?php endif; ?>
</div>

<?php
// Include checkout modal (only once per page)
static $checkout_modal_loaded = false;
if (!$checkout_modal_loaded) {
    include TICKETPAYGO_PLUGIN_PATH . 'templates/checkout-modal.php';
    $checkout_modal_loaded = true;
}
?>

