<?php
/**
 * Event Card Template - Matches Live Preview Design
 * Click opens modal for booking
 * 
 * Available variables:
 * $event - TicketPayGo_Event object
 * 
 * phpcs:disable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
 */

if (!defined('ABSPATH')) {
    exit;
}

// Validate event object exists
if (!isset($event) || !is_object($event) || !method_exists($event, 'get_data')) {
    return;
}

$event_data = $event->get_data();

// Validate event data
if (empty($event_data) || !isset($event_data['id'])) {
    return;
}

$event_id = $event_data['id'];

// Get custom colors from settings
$card_primary_color = TicketPayGo_Database::get_setting('card_primary_color', '#0A4974');
$card_button_color = TicketPayGo_Database::get_setting('card_button_color', '#42963B');
$card_button_text_color = TicketPayGo_Database::get_setting('card_button_text_color', '#FFFFFF');
$card_text_color = TicketPayGo_Database::get_setting('card_text_color', '#333333');

// Default settings for Lite version (customization available in Full)
$settings = array(
    'button_text' => __('Get Tickets', 'ticketpaygo-lite'),
    'primary_color' => $card_primary_color
);

// Format price - get lowest ticket price
$currency_symbol = TicketPayGo_Database::get_setting('currency_symbol', '€');
$currency_position = TicketPayGo_Database::get_setting('currency_position', 'before');

// Get the lowest ticket price from ticket types
$ticket_types = array();
if (!empty($event_data['ticket_types'])) {
    if (is_string($event_data['ticket_types'])) {
        $decoded = json_decode($event_data['ticket_types'], true);
        $ticket_types = is_array($decoded) ? $decoded : array();
    } elseif (is_array($event_data['ticket_types'])) {
        $ticket_types = $event_data['ticket_types'];
    }
}

$lowest_price = null;
if (!empty($ticket_types)) {
    foreach ($ticket_types as $ticket_type) {
        if (is_array($ticket_type) && isset($ticket_type['price'])) {
            $ticket_price = floatval($ticket_type['price']);
            if ($lowest_price === null || $ticket_price < $lowest_price) {
                $lowest_price = $ticket_price;
            }
        }
    }
}

// Use lowest ticket price if available, otherwise use event price
$display_price = ($lowest_price !== null) ? $lowest_price : floatval($event_data['price'] ?? 0);
?>

<?php
// Enqueue inline styles for dynamic colors - MOVED TO ticketpaygo-lite.php
?><?php


if ($display_price > 0) {
    $formatted_price = number_format($display_price, 2);
    if ($currency_position === 'before') {
        $price_display = $currency_symbol . ' ' . $formatted_price;
    } else {
        $price_display = $formatted_price . ' ' . $currency_symbol;
    }
} else {
    $price_display = __('Free', 'ticketpaygo-lite');
}

// Get remaining tickets (returns -1 for unlimited)
$remaining_tickets = $event->get_remaining_tickets();
if ($remaining_tickets < 0) {
    $remaining_tickets = null; // Unlimited tickets
}
?>

<!-- Event Card - Matches Live Preview -->
<div class="ticketpaygo-card" data-event-id="<?php echo esc_attr($event_id); ?>" onclick="openEventModal(<?php echo esc_attr($event_id); ?>)">
    <!-- Image Area -->
    <div class="ticketpaygo-card-image">
        <?php if (!empty($event_data['featured_image'])): ?>
            <img src="<?php echo esc_url($event_data['featured_image']); ?>" alt="<?php echo esc_attr($event_data['title']); ?>" style="object-fit: <?php echo esc_attr(isset($event_data['image_fit']) ? $event_data['image_fit'] : 'cover'); ?>;" />
        <?php endif; ?>
        <?php if (!isset($event_data['show_title_on_card']) || $event_data['show_title_on_card']): ?>
            <h3 class="ticketpaygo-card-title"><?php echo esc_html($event_data['title']); ?></h3>
        <?php endif; ?>
    </div>
    
    <!-- Content Area -->
    <div class="ticketpaygo-card-content">
        <div class="ticketpaygo-card-meta">
            <span class="meta-item">
                <svg class="event-icon" width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" style="color: <?php echo esc_attr($settings['icon_color'] ?? '#666666'); ?>; flex-shrink: 0;">
                    <path d="M19 4H5C3.89543 4 3 4.89543 3 6V20C3 21.1046 3.89543 22 5 22H19C20.1046 22 21 21.1046 21 20V6C21 4.89543 20.1046 4 19 4Z" stroke="currentColor" stroke-width="2" fill="none"/>
                    <path d="M16 2V6M8 2V6M3 10H21" stroke="currentColor" stroke-width="2" fill="none"/>
                </svg>
                <?php echo esc_html(date_i18n('M j, Y', strtotime($event_data['start_date']))); ?>
            </span>
            <?php if (!empty($event_data['location'])): ?>
                <span class="meta-item">
                    <svg class="event-icon" width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" style="color: <?php echo esc_attr($settings['icon_color'] ?? '#666666'); ?>; flex-shrink: 0;">
                        <path d="M21 10C21 17 12 23 12 23C12 23 3 17 3 10C3 7.61305 3.94821 5.32387 5.63604 3.63604C7.32387 1.94821 9.61305 1 12 1C14.3869 1 16.6761 1.94821 18.364 3.63604C20.0518 5.32387 21 7.61305 21 10Z" stroke="currentColor" stroke-width="2" fill="none"/>
                        <path d="M12 13C13.6569 13 15 11.6569 15 10C15 8.34315 13.6569 7 12 7C10.3431 7 9 8.34315 9 10C9 11.6569 10.3431 13 12 13Z" stroke="currentColor" stroke-width="2" fill="none"/>
                    </svg>
                    <?php echo esc_html($event_data['location']); ?>
                </span>
            <?php endif; ?>
            <span class="meta-item price">
                <svg class="event-icon" width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" style="color: <?php echo esc_attr($settings['icon_color'] ?? '#666666'); ?>; flex-shrink: 0;">
                    <path d="M12 1V23M17 5H9.5C8.57174 5 7.6815 5.36875 7.02513 6.02513C6.36875 6.6815 6 7.57174 6 8.5C6 9.42826 6.36875 10.3185 7.02513 10.9749C7.6815 11.6313 8.57174 12 9.5 12H14.5C15.4283 12 16.3185 12.3687 16.9749 13.0251C17.6313 13.6815 18 14.5717 18 15.5C18 16.4283 17.6313 17.3185 16.9749 17.9749C16.3185 18.6313 15.4283 19 14.5 19H6" stroke="currentColor" stroke-width="2" fill="none"/>
                </svg>
                <?php echo esc_html($price_display); ?>
            </span>
            <?php if ($remaining_tickets !== null && $remaining_tickets >= 0): ?>
                <span class="meta-item">
                    <svg class="event-icon" width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" style="color: <?php echo esc_attr($settings['icon_color'] ?? '#666666'); ?>; flex-shrink: 0;">
                        <path d="M22 10V6C22 4.9 21.1 4 20 4H4C2.9 4 2 4.9 2 6V10C3.1 10 4 10.9 4 12C4 13.1 3.1 14 2 14V18C2 19.1 2.9 20 4 20H20C21.1 20 22 19.1 22 18V14C20.9 14 20 13.1 20 12C20 10.9 20.9 10 22 10Z" stroke="currentColor" stroke-width="2" fill="none"/>
                    </svg>
                    <?php echo intval($remaining_tickets); ?> <?php esc_html_e('tickets', 'ticketpaygo-lite'); ?>
                </span>
            <?php endif; ?>
        </div>
        
        <button class="ticketpaygo-card-button" type="button">
            <?php echo esc_html($settings['button_text']); ?>
        </button>
    </div>
</div>

<!-- Event Modal (hidden by default) -->
<div class="ticketpaygo-modal" id="modal-<?php echo esc_attr($event_id); ?>" style="display: none;">
    <div class="ticketpaygo-modal-overlay" onclick="closeEventModal(<?php echo esc_attr($event_id); ?>)"></div>
    <div class="ticketpaygo-modal-content">
        <button class="ticketpaygo-modal-close" onclick="closeEventModal(<?php echo esc_attr($event_id); ?>)" aria-label="<?php esc_html_e('Close', 'ticketpaygo-lite'); ?>">
            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M18 6L6 18M6 6L18 18" stroke="currentColor" stroke-width="2" stroke-linecap="round"/>
            </svg>
        </button>
        
        <!-- Modal Header with Image -->
        <div class="ticketpaygo-modal-header">
            <?php if (!empty($event_data['featured_image'])): ?>
                <img src="<?php echo esc_url($event_data['featured_image']); ?>" alt="<?php echo esc_attr($event_data['title']); ?>" style="object-fit: <?php echo esc_attr($event_data['image_fit'] ?? 'cover'); ?>;" />
            <?php endif; ?>
            <h2><?php echo wp_kses_post($event_data['title']); ?></h2>
        </div>
        
        <!-- Modal Body -->
        <div class="ticketpaygo-modal-body">
            <!-- Event Info Grid -->
            <div class="modal-event-info-grid">
                <div class="info-cell">
                    <svg class="info-icon" width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M19 4H5C3.89543 4 3 4.89543 3 6V20C3 21.1046 3.89543 22 5 22H19C20.1046 22 21 21.1046 21 20V6C21 4.89543 20.1046 4 19 4Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                        <path d="M16 2V6M8 2V6M3 10H21" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                    </svg>
                    <span class="info-label"><?php esc_html_e('Date', 'ticketpaygo-lite'); ?></span>
                    <span class="info-value"><?php echo esc_html(date_i18n('F j, Y - g:i A', strtotime($event_data['start_date']))); ?></span>
                </div>
                <?php if (!empty($event_data['location'])): ?>
                    <div class="info-cell">
                        <svg class="info-icon" width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path d="M21 10C21 17 12 23 12 23C12 23 3 17 3 10C3 7.61305 3.94821 5.32387 5.63604 3.63604C7.32387 1.94821 9.61305 1 12 1C14.3869 1 16.6761 1.94821 18.364 3.63604C20.0518 5.32387 21 7.61305 21 10Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                            <path d="M12 13C13.6569 13 15 11.6569 15 10C15 8.34315 13.6569 7 12 7C10.3431 7 9 8.34315 9 10C9 11.6569 10.3431 13 12 13Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                        </svg>
                        <span class="info-label"><?php esc_html_e('Location', 'ticketpaygo-lite'); ?></span>
                        <span class="info-value"><?php echo esc_html($event_data['location']); ?></span>
                    </div>
                <?php endif; ?>
                <div class="info-cell">
                    <svg class="info-icon" width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M12 1V23M17 5H9.5C8.57174 5 7.6815 5.36875 7.02513 6.02513C6.36875 6.6815 6 7.57174 6 8.5C6 9.42826 6.36875 10.3185 7.02513 10.9749C7.6815 11.6313 8.57174 12 9.5 12H14.5C15.4283 12 16.3185 12.3687 16.9749 13.0251C17.6313 13.6815 18 14.5717 18 15.5C18 16.4283 17.6313 17.3185 16.9749 17.9749C16.3185 18.6313 15.4283 19 14.5 19H6" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                    </svg>
                    <span class="info-label"><?php esc_html_e('Price', 'ticketpaygo-lite'); ?></span>
                    <span class="info-value"><?php echo esc_html($price_display); ?></span>
                </div>
                <?php if ($remaining_tickets !== null && $remaining_tickets >= 0): ?>
                    <div class="info-cell">
                        <svg class="info-icon" width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path d="M22 10V6C22 4.9 21.1 4 20 4H4C2.9 4 2 4.9 2 6V10C3.1 10 4 10.9 4 12C4 13.1 3.1 14 2 14V18C2 19.1 2.9 20 4 20H20C21.1 20 22 19.1 22 18V14C20.9 14 20 13.1 20 12C20 10.9 20.9 10 22 10Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                        </svg>
                        <span class="info-label"><?php esc_html_e('Available', 'ticketpaygo-lite'); ?></span>
                        <span class="info-value"><?php echo intval($remaining_tickets); ?> <?php esc_html_e('remaining', 'ticketpaygo-lite'); ?></span>
                    </div>
                <?php endif; ?>
            </div>
            
            <?php if (!empty($event_data['description'])): ?>
                <div class="modal-description">
                    <h4><?php esc_html_e('About This Event', 'ticketpaygo-lite'); ?></h4>
                    <?php echo wp_kses_post($event_data['description']); ?>
                </div>
            <?php endif; ?>
            
            <?php if (!empty($event_data['address'])): ?>
                <div class="modal-venue">
                    <h4><?php esc_html_e('Venue', 'ticketpaygo-lite'); ?></h4>
                    <p><?php echo esc_html($event_data['address']); ?></p>
                </div>
            <?php endif; ?>
            
            <?php if (!empty($event_data['latitude']) && !empty($event_data['longitude']) && $event_data['latitude'] != 0 && $event_data['longitude'] != 0): 
                $lat = floatval($event_data['latitude']);
                $lng = floatval($event_data['longitude']);
                // Calculate bounding box for the map (approximately 500m around the point)
                $bbox_margin = 0.005; // ~500m
                $bbox = ($lng - $bbox_margin) . '%2C' . ($lat - $bbox_margin) . '%2C' . ($lng + $bbox_margin) . '%2C' . ($lat + $bbox_margin);
            ?>
                <div class="modal-map">
                    <h4><?php esc_html_e('Location Map', 'ticketpaygo-lite'); ?></h4>
                    <iframe 
                        width="100%" 
                        height="200" 
                        frameborder="0" 
                        scrolling="no" 
                        marginheight="0" 
                        marginwidth="0" 
                        src="https://www.openstreetmap.org/export/embed.html?bbox=<?php echo esc_attr($bbox); ?>&amp;layer=mapnik&amp;marker=<?php echo esc_attr($lat); ?>%2C<?php echo esc_attr($lng); ?>"
                        style="border-radius: 8px; border: 1px solid #e2e8f0;"
                        loading="lazy">
                    </iframe>
                    <a href="https://www.google.com/maps?q=<?php echo esc_attr($lat); ?>,<?php echo esc_attr($lng); ?>" 
                       target="_blank" 
                       class="get-directions-link"
                       style="display: inline-flex; align-items: center; gap: 5px; color: #0073aa; text-decoration: none; font-size: 14px; margin-top: 10px;">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path d="M21 10C21 17 12 23 12 23C12 23 3 17 3 10C3 7.61305 3.94821 5.32387 5.63604 3.63604C7.32387 1.94821 9.61305 1 12 1C14.3869 1 16.6761 1.94821 18.364 3.63604C20.0518 5.32387 21 7.61305 21 10Z" stroke="currentColor" stroke-width="2"/>
                        </svg>
                        <?php esc_html_e('Get Directions', 'ticketpaygo-lite'); ?>
                    </a>
                </div>
            <?php endif; ?>
            
            <!-- Ticket Selection -->
            <div class="modal-ticket-types">
                <h4><?php esc_html_e('Select Tickets', 'ticketpaygo-lite'); ?></h4>
                
                <?php if (!empty($event_data['ticket_types']) && is_array($event_data['ticket_types'])): ?>
                    <!-- Multiple Ticket Types -->
                    <?php foreach ($event_data['ticket_types'] as $index => $ticket_type): ?>
                        <?php if (!empty($ticket_type['name'])): ?>
                            <div class="ticket-type-item" data-ticket-index="<?php echo esc_attr($index); ?>">
                                <div class="ticket-type-info">
                                    <div class="ticket-type-name"><?php echo esc_html($ticket_type['name']); ?></div>
                                    <?php if (!empty($ticket_type['description'])): ?>
                                        <div class="ticket-type-description"><?php echo esc_html($ticket_type['description']); ?></div>
                                    <?php endif; ?>
                                    <div class="ticket-type-price">
                                        <?php 
                                        $ticket_price = floatval(isset($ticket_type['price']) ? $ticket_type['price'] : 0);
                                        if ($ticket_price > 0) {
                                            $formatted_ticket_price = number_format($ticket_price, 2);
                                            if ($currency_position === 'before') {
                                                echo esc_html($currency_symbol . ' ' . $formatted_ticket_price);
                                            } else {
                                                echo esc_html($formatted_ticket_price . ' ' . $currency_symbol);
                                            }
                                        } else {
                                            esc_html_e('Free', 'ticketpaygo-lite');
                                        }
                                        ?>
                                    </div>
                                </div>
                                <div class="ticket-type-quantity">
                                    <button type="button" class="qty-btn qty-minus" data-ticket-index="<?php echo esc_attr($index); ?>">−</button>
                                    <input type="number" class="qty-input ticket-type-qty-input" min="0" max="<?php echo esc_attr($ticket_type['quantity'] ?? 999); ?>" value="0" data-ticket-index="<?php echo esc_attr($index); ?>" data-price="<?php echo esc_attr($ticket_price); ?>" data-type-name="<?php echo esc_attr($ticket_type['name'] ?? ''); ?>" />
                                    <button type="button" class="qty-btn qty-plus" data-ticket-index="<?php echo esc_attr($index); ?>">+</button>
                                </div>
                            </div>
                        <?php endif; ?>
                    <?php endforeach; ?>
                <?php else: ?>
                    <!-- Single Ticket Type -->
                    <div class="ticket-type-item single-ticket">
                        <div class="ticket-type-info">
                            <div class="ticket-type-name"><?php echo esc_html(!empty($event_data['ticket_name']) ? $event_data['ticket_name'] : __('General Admission', 'ticketpaygo-lite')); ?></div>
                            <div class="ticket-type-price"><?php echo esc_html($price_display); ?></div>
                        </div>
                        <div class="ticket-type-quantity">
                            <button type="button" class="qty-btn qty-minus">−</button>
                            <input type="number" class="qty-input" min="0" max="<?php echo $remaining_tickets !== null ? esc_attr($remaining_tickets) : 999; ?>" value="0" data-price="<?php echo esc_attr($display_price); ?>" />
                            <button type="button" class="qty-btn qty-plus">+</button>
                        </div>
                    </div>
                <?php endif; ?>
                
                <div class="ticket-total">
                    <span class="total-label"><?php esc_html_e('Total:', 'ticketpaygo-lite'); ?></span>
                    <span class="total-amount" data-currency-symbol="<?php echo esc_attr($currency_symbol); ?>" data-currency-position="<?php echo esc_attr($currency_position); ?>">
                        <?php 
                        if ($currency_position === 'before') {
                            echo esc_html($currency_symbol . ' 0.00');
                        } else {
                            echo esc_html('0.00 ' . $currency_symbol);
                        }
                        ?>
                    </span>
                </div>
            </div>
        </div>
        
        <!-- Modal Footer with Checkout Button -->
        <div class="ticketpaygo-modal-footer">
            <button type="button" 
                    class="ticketpaygo-book-now open-checkout-modal" 
                    data-event-id="<?php echo esc_attr($event_id); ?>"
                    data-event-title="<?php echo esc_attr($event_data['title']); ?>"
                    data-event-price="<?php echo esc_attr($display_price); ?>">
                <?php echo esc_html($settings['button_text']); ?>
            </button>
        </div>
    </div>
</div>

<?php
// Include checkout modal (only once per page)
static $checkout_modal_loaded = false;
if (!$checkout_modal_loaded) {
    include TICKETPAYGO_PLUGIN_PATH . 'templates/checkout-modal.php';
    $checkout_modal_loaded = true;
}
?>

