<?php
/**
 * Booking Modal Template
 * Modern popup-style booking interface
 * 
 * @var int $event_id Event ID
 * 
 * phpcs:disable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
 */

if (!defined('ABSPATH')) {
    exit;
}

$event = new TicketPayGo_Event($event_id);
$event_data = $event->get_data();

if (!$event_data) {
    return;
}

$currency_symbol = TicketPayGo_Database::get_setting('currency_symbol', '€');
$max_tickets = intval($event_data['max_tickets']);
$remaining_tickets = ($max_tickets > 0) ? ($max_tickets - $event_data['tickets_sold']) : null;

// Get Terms and Privacy URLs from settings
$terms_url = TicketPayGo_Database::get_setting('terms_url', '');
$privacy_url = TicketPayGo_Database::get_setting('privacy_url', '');

// Get free event registration setting
$free_event_registration = TicketPayGo_Database::get_setting('free_event_registration', 'required');
$is_free_event = floatval($event_data['price']) <= 0;
?>

<div id="ticketpaygo-modal-<?php echo esc_attr($event_id); ?>" class="ticketpaygo-modal" style="display: none;">
    <div class="ticketpaygo-modal-overlay"></div>
    <div class="ticketpaygo-modal-container">
        <div class="ticketpaygo-modal-header">
            <button class="ticketpaygo-modal-close" aria-label="<?php esc_html_e('Close', 'ticketpaygo-lite'); ?>">
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M18 6L6 18M6 6L18 18" stroke="currentColor" stroke-width="2" stroke-linecap="round"/>
                </svg>
            </button>
        </div>
        
        <div class="ticketpaygo-modal-content">
            <!-- Event Info Section -->
            <div class="modal-event-info">
                <?php if (!empty($event_data['featured_image'])): ?>
                    <div class="modal-event-image">
                        <img src="<?php echo esc_url($event_data['featured_image']); ?>" alt="<?php echo esc_attr($event_data['title']); ?>">
                    </div>
                <?php endif; ?>
                
                <div class="modal-event-details">
                    <h2 class="modal-event-title"><?php echo esc_html($event_data['title']); ?></h2>
                    
                    <div class="modal-event-meta">
                        <div class="meta-item">
                            <span class="dashicons dashicons-calendar-alt"></span>
                            <span><?php echo esc_html(date_i18n(get_option('date_format'), strtotime($event_data['start_date']))); ?></span>
                        </div>
                        
                        <div class="meta-item">
                            <span class="dashicons dashicons-clock"></span>
                            <span><?php echo esc_html(date_i18n(get_option('time_format'), strtotime($event_data['start_date']))); ?></span>
                        </div>
                        
                        <?php if (!empty($event_data['location'])): ?>
                        <div class="meta-item">
                            <span class="dashicons dashicons-location"></span>
                            <span><?php echo esc_html($event_data['location']); ?></span>
                        </div>
                        <?php endif; ?>
                        
                        <?php if (!empty($event_data['latitude']) && !empty($event_data['longitude'])): ?>
                        <!-- Event Location Map -->
                        <div class="event-location-map-container" style="margin-top: 15px;">
                            <div id="event-location-map-<?php echo esc_attr($event_id); ?>" 
                                 class="event-location-map" 
                                 data-lat="<?php echo esc_attr($event_data['latitude']); ?>"
                                 data-lng="<?php echo esc_attr($event_data['longitude']); ?>"
                                 style="width: 100%; height: 200px; border-radius: 8px; overflow: hidden; border: 1px solid #ddd;"></div>
                            <a href="https://www.google.com/maps?q=<?php echo esc_attr($event_data['latitude']); ?>,<?php echo esc_attr($event_data['longitude']); ?>" 
                               target="_blank" 
                               class="get-directions-btn" 
                               style="display: inline-block; margin-top: 10px; padding: 8px 16px; background: #0073aa; color: white; text-decoration: none; border-radius: 4px; font-size: 14px;">
                                <span class="dashicons dashicons-location-alt" style="vertical-align: middle; margin-right: 5px;"></span>
                                <?php esc_html_e('Get Directions', 'ticketpaygo-lite'); ?>
                            </a>
                        </div>
                        <?php endif; ?>
                        
                        <div class="meta-item price-display">
                            <span class="dashicons dashicons-tickets-alt"></span>
                            <span class="price"><?php echo esc_html($currency_symbol . number_format($event_data['price'], 2)); ?></span>
                        </div>
                    </div>
                    
                    <?php if ($remaining_tickets !== null && $remaining_tickets <= 10 && $remaining_tickets > 0): ?>
                        <div class="tickets-warning">
                            <span class="dashicons dashicons-warning"></span>
                            <?php /* translators: %d: number of remaining tickets */ ?>
                            <?php printf(esc_html__('Only %d tickets remaining!', 'ticketpaygo-lite'), intval($remaining_tickets)); ?>
                        </div>
                    <?php endif; ?>
                    
                    <!-- Calendar Buttons (Grayed Out - Paid Feature) -->
                    <div class="calendar-buttons-lite" style="margin-top: 20px; padding-top: 20px; border-top: 1px solid #e5e5e5;">
                        <h4 style="margin: 0 0 12px 0; font-size: 14px; color: #64748b;"><?php esc_html_e('Add to Calendar', 'ticketpaygo-lite'); ?></h4>
                        <div style="display: grid; grid-template-columns: repeat(2, 1fr); gap: 8px;">
                            <button type="button" disabled style="display: flex; align-items: center; gap: 8px; padding: 10px 12px; border: 1px solid #e5e5e5; border-radius: 6px; background: #f8fafc; color: #94a3b8; cursor: not-allowed; opacity: 0.5; font-size: 13px;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M19 3h-1V1h-2v2H8V1H6v2H5c-1.1 0-2 .9-2 2v14c0 1.1.9 2 2 2h14c1.1 0 2-.9 2-2V5c0-1.1-.9-2-2-2zm0 16H5V8h14v11z"/></svg>
                                <?php esc_html_e('Google Calendar', 'ticketpaygo-lite'); ?>
                            </button>
                            <button type="button" disabled style="display: flex; align-items: center; gap: 8px; padding: 10px 12px; border: 1px solid #e5e5e5; border-radius: 6px; background: #f8fafc; color: #94a3b8; cursor: not-allowed; opacity: 0.5; font-size: 13px;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M17.9 17.39c-.26-.8-1.01-1.39-1.9-1.39h-1v-3c0-.55-.45-1-1-1H8v-2h2c.55 0 1-.45 1-1V7h2c1.1 0 2-.9 2-2v-.41c2.93 1.19 5 4.06 5 7.41 0 2.08-.8 3.97-2.1 5.39z"/><path d="M11 19.93c-3.95-.49-7-3.85-7-7.93 0-.62.08-1.21.21-1.79L9 15v1c0 1.1.9 2 2 2v1.93z"/></svg>
                                <?php esc_html_e('Apple Calendar', 'ticketpaygo-lite'); ?>
                            </button>
                            <button type="button" disabled style="display: flex; align-items: center; gap: 8px; padding: 10px 12px; border: 1px solid #e5e5e5; border-radius: 6px; background: #f8fafc; color: #94a3b8; cursor: not-allowed; opacity: 0.5; font-size: 13px;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M20 4H4c-1.1 0-1.99.9-1.99 2L2 18c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V6c0-1.1-.9-2-2-2zm0 4l-8 5-8-5V6l8 5 8-5v2z"/></svg>
                                <?php esc_html_e('Outlook', 'ticketpaygo-lite'); ?>
                            </button>
                            <button type="button" disabled style="display: flex; align-items: center; gap: 8px; padding: 10px 12px; border: 1px solid #e5e5e5; border-radius: 6px; background: #f8fafc; color: #94a3b8; cursor: not-allowed; opacity: 0.5; font-size: 13px;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z"/></svg>
                                <?php esc_html_e('Yahoo Calendar', 'ticketpaygo-lite'); ?>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Booking Form Section -->
            <div class="modal-booking-form">
                <?php if ($remaining_tickets === null || $remaining_tickets > 0): ?>
                    <form id="ticketpaygo-modal-form-<?php echo esc_attr($event_id); ?>" class="modal-form">
                        <input type="hidden" name="event_id" value="<?php echo esc_attr($event_id); ?>">
                        <input type="hidden" name="action" value="ticketpaygo_purchase_ticket">
                        <input type="hidden" name="nonce" value="<?php echo esc_attr(wp_create_nonce('ticketpaygo_frontend_nonce')); ?>">
                        
                        <!-- Ticket Quantity -->
                        <div class="form-step active" data-step="1">
                            <h3><?php esc_html_e('Select Tickets', 'ticketpaygo-lite'); ?></h3>
                            
                            <?php 
                            $ticket_types = isset($event_data['ticket_types']) ? (is_string($event_data['ticket_types']) ? json_decode($event_data['ticket_types'], true) : $event_data['ticket_types']) : array();
                            $has_multiple_types = !empty($ticket_types) && count($ticket_types) > 0;
                            ?>
                            
                            <?php if ($has_multiple_types): ?>
                                <!-- Multiple Ticket Types Selection -->
                                <div class="ticket-types-selection">
                                    <?php foreach ($ticket_types as $type_index => $ticket_type): 
                                        $type_available = true;
                                        $type_remaining = $ticket_type['quantity'] > 0 ? ($ticket_type['quantity'] - (isset($ticket_type['sold']) ? $ticket_type['sold'] : 0)) : 999;
                                        
                                        // Check sale dates
                                        if (!empty($ticket_type['sale_start']) && strtotime($ticket_type['sale_start']) > time()) {
                                            $type_available = false;
                                        }
                                        if (!empty($ticket_type['sale_end']) && strtotime($ticket_type['sale_end']) < time()) {
                                            $type_available = false;
                                        }
                                        if ($type_remaining <= 0) {
                                            $type_available = false;
                                        }
                                    ?>
                                        <div class="ticket-type-option <?php echo !$type_available ? 'unavailable' : ''; ?>">
                                            <div class="ticket-type-info">
                                                <div class="ticket-type-name"><?php echo esc_html($ticket_type['name']); ?></div>
                                                <?php if (!empty($ticket_type['description'])): ?>
                                                    <div class="ticket-type-desc"><?php echo esc_html($ticket_type['description']); ?></div>
                                                <?php endif; ?>
                                                <div class="ticket-type-price"><?php echo esc_html($currency_symbol . number_format($ticket_type['price'], 2)); ?></div>
                                                <?php if (!$type_available): ?>
                                                    <div class="ticket-type-status"><?php esc_html_e('Sold Out', 'ticketpaygo-lite'); ?></div>
                                                <?php elseif ($type_remaining <= 10): ?>
                                                    <?php /* translators: %d: number of tickets left for this type */ ?>
                                                    <div class="ticket-type-status warning"><?php printf(esc_html__('Only %d left!', 'ticketpaygo-lite'), intval($type_remaining)); ?></div>
                                                <?php endif; ?>
                                            </div>
                                            <?php if ($type_available): ?>
                                                <div class="ticket-type-qty">
                                                    <button type="button" class="qty-btn qty-minus" data-target="ticket_type_qty_<?php echo esc_attr($event_id); ?>_<?php echo esc_attr($type_index); ?>">-</button>
                                                    <input type="number" 
                                                           id="ticket_type_qty_<?php echo esc_attr($event_id); ?>_<?php echo esc_attr($type_index); ?>" 
                                                           name="ticket_types_qty[<?php echo esc_attr($type_index); ?>]" 
                                                           value="0" 
                                                           min="0" 
                                                           max="<?php echo esc_attr(min(isset($ticket_type['max_per_order']) ? $ticket_type['max_per_order'] : 10, $type_remaining)); ?>" 
                                                           class="qty-input ticket-type-qty-input"
                                                           data-price="<?php echo esc_attr($ticket_type['price']); ?>"
                                                           data-type-name="<?php echo esc_attr($ticket_type['name']); ?>">
                                                    <button type="button" class="qty-btn qty-plus" data-target="ticket_type_qty_<?php echo esc_attr($event_id); ?>_<?php echo esc_attr($type_index); ?>">+</button>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                                <input type="hidden" name="quantity" id="total_quantity_<?php echo esc_attr($event_id); ?>" value="0">
                            <?php else: ?>
                                <!-- Simple Single Ticket Type -->
                                <div class="quantity-selector-modern">
                                    <label for="ticket_quantity_<?php echo esc_attr($event_id); ?>"><?php esc_html_e('Number of Tickets', 'ticketpaygo-lite'); ?></label>
                                    <div class="quantity-controls">
                                        <button type="button" class="qty-btn qty-minus" data-target="ticket_quantity_<?php echo esc_attr($event_id); ?>">-</button>
                                        <input type="number" 
                                               id="ticket_quantity_<?php echo esc_attr($event_id); ?>" 
                                               name="quantity" 
                                               value="1" 
                                               min="1" 
                                               max="<?php echo esc_attr($remaining_tickets !== null ? min(10, $remaining_tickets) : 10); ?>" 
                                               class="qty-input"
                                               data-price="<?php echo esc_attr($event_data['price']); ?>">
                                        <button type="button" class="qty-btn qty-plus" data-target="ticket_quantity_<?php echo esc_attr($event_id); ?>">+</button>
                                    </div>
                                </div>
                            <?php endif; ?>
                            
                            <div class="price-summary-modern">
                                <div class="price-row">
                                    <span class="price-label"><?php esc_html_e('Subtotal', 'ticketpaygo-lite'); ?></span>
                                    <span class="price-value subtotal"><?php echo esc_html($currency_symbol); ?>0.00</span>
                                </div>
                                <div class="price-row">
                                    <span class="price-label"><?php esc_html_e('Processing Fee', 'ticketpaygo-lite'); ?></span>
                                    <span class="price-value processing-fee"><?php echo esc_html($currency_symbol); ?>0.00</span>
                                </div>
                                <div class="price-row total">
                                    <span class="price-label"><?php esc_html_e('Total', 'ticketpaygo-lite'); ?></span>
                                    <span class="price-value total-amount"><?php echo esc_html($currency_symbol); ?>0.00</span>
                                </div>
                            </div>
                            
                            <button type="button" class="btn-next-step"><?php esc_html_e('Continue', 'ticketpaygo-lite'); ?></button>
                        </div>
                        
                        <!-- Customer Information -->
                        <div class="form-step" data-step="2">
                            <h3><?php esc_html_e('Your Information', 'ticketpaygo-lite'); ?></h3>
                            
                            <div class="form-grid">
                                <div class="form-group">
                                    <label for="customer_name_<?php echo esc_attr($event_id); ?>"><?php esc_html_e('Full Name', 'ticketpaygo-lite'); ?> *</label>
                                    <input type="text" 
                                           id="customer_name_<?php echo esc_attr($event_id); ?>" 
                                           name="customer_name" 
                                           required>
                                </div>
                                
                                <div class="form-group">
                                    <label for="customer_email_<?php echo esc_attr($event_id); ?>"><?php esc_html_e('Email Address', 'ticketpaygo-lite'); ?> *</label>
                                    <input type="email" 
                                           id="customer_email_<?php echo esc_attr($event_id); ?>" 
                                           name="customer_email" 
                                           required>
                                </div>
                                
                                <div class="form-group">
                                    <label for="customer_phone_<?php echo esc_attr($event_id); ?>"><?php esc_html_e('Phone Number', 'ticketpaygo-lite'); ?></label>
                                    <input type="tel" 
                                           id="customer_phone_<?php echo esc_attr($event_id); ?>" 
                                           name="customer_phone">
                                </div>
                                
                                <div class="form-group full-width">
                                    <label for="special_requests_<?php echo esc_attr($event_id); ?>"><?php esc_html_e('Special Requests', 'ticketpaygo-lite'); ?></label>
                                    <textarea id="special_requests_<?php echo esc_attr($event_id); ?>" 
                                              name="special_requests" 
                                              rows="3" 
                                              placeholder="<?php esc_html_e('Any special requirements or notes...', 'ticketpaygo-lite'); ?>"></textarea>
                                </div>
                            </div>
                            
                            <div class="form-navigation">
                                <button type="button" class="btn-prev-step"><?php esc_html_e('Back', 'ticketpaygo-lite'); ?></button>
                                <button type="button" class="btn-next-step"><?php esc_html_e('Continue', 'ticketpaygo-lite'); ?></button>
                            </div>
                        </div>
                        
                        <!-- Payment Method -->
                        <div class="form-step" data-step="3">
                            <h3><?php esc_html_e('Payment Method', 'ticketpaygo-lite'); ?></h3>
                            
                            <div class="payment-methods-modern">
                                <?php
                                $payment_methods = array();
                                if (TicketPayGo_Database::get_setting('mollie_enabled', '0') === '1') {
                                    $payment_methods['mollie'] = array(
                                        'name' => 'Mollie',
                                        'icon' => TICKETPAYGO_PLUGIN_URL . 'assets/images/mollie-logo.png',
                                        'description' => __('Credit Card, PayPal, Bank Transfer', 'ticketpaygo-lite')
                                    );
                                }
                                if (TicketPayGo_Database::get_setting('paypal_enabled', '0') === '1') {
                                    $payment_methods['paypal'] = array(
                                        'name' => 'PayPal',
                                        'icon' => TICKETPAYGO_PLUGIN_URL . 'assets/images/paypal-logo.png',
                                        'description' => __('PayPal Account or Credit Card', 'ticketpaygo-lite')
                                    );
                                }
                                if (TicketPayGo_Database::get_setting('stripe_enabled', '0') === '1') {
                                    $payment_methods['stripe'] = array(
                                        'name' => 'Stripe',
                                        'icon' => TICKETPAYGO_PLUGIN_URL . 'assets/images/stripe-logo.png',
                                        'description' => __('Credit Card (Visa, MasterCard, Amex)', 'ticketpaygo-lite')
                                    );
                                }
                                
                                foreach ($payment_methods as $method_id => $method): ?>
                                    <div class="payment-method-modern">
                                        <input type="radio" 
                                               id="payment_<?php echo esc_attr($method_id); ?>_<?php echo esc_attr($event_id); ?>" 
                                               name="payment_method" 
                                               value="<?php echo esc_attr($method_id); ?>" 
                                               <?php echo empty($selected_method) ? 'checked' : ''; ?>>
                                        <label for="payment_<?php echo esc_attr($method_id); ?>_<?php echo esc_attr($event_id); ?>">
                                            <div class="payment-method-content">
                                                <div class="payment-method-icon">
                                                    <img src="<?php echo esc_url($method['icon']); ?>" alt="<?php echo esc_attr($method['name']); ?>">
                                                </div>
                                                <div class="payment-method-info">
                                                    <span class="payment-method-name"><?php echo esc_html($method['name']); ?></span>
                                                    <span class="payment-method-desc"><?php echo esc_html($method['description']); ?></span>
                                                </div>
                                            </div>
                                        </label>
                                    </div>
                                <?php 
                                $selected_method = true;
                                endforeach; ?>
                            </div>
                            
                            <?php if (!empty($terms_url) || !empty($privacy_url)): ?>
                            <div class="terms-agreement">
                                <label class="checkbox-container">
                                    <input type="checkbox" name="agree_terms" required>
                                    <span class="checkmark"></span>
                                    <?php 
                                    if (!empty($terms_url) && !empty($privacy_url)) {
                                        /* translators: %1$s: Terms & Conditions link, %2$s: Privacy Policy link */
                                        printf(esc_html__('I agree to the %1$s and %2$s', 'ticketpaygo-lite'), 
                                            '<a href="' . esc_url($terms_url) . '" target="_blank">' . esc_html__('Terms & Conditions', 'ticketpaygo-lite') . '</a>',
                                            '<a href="' . esc_url($privacy_url) . '" target="_blank">' . esc_html__('Privacy Policy', 'ticketpaygo-lite') . '</a>'
                                        );
                                    } elseif (!empty($terms_url)) {
                                        /* translators: %s: Terms & Conditions link */
                                        printf(esc_html__('I agree to the %s', 'ticketpaygo-lite'), 
                                            '<a href="' . esc_url($terms_url) . '" target="_blank">' . esc_html__('Terms & Conditions', 'ticketpaygo-lite') . '</a>'
                                        );
                                    } elseif (!empty($privacy_url)) {
                                        /* translators: %s: Privacy Policy link */
                                        printf(esc_html__('I agree to the %s', 'ticketpaygo-lite'), 
                                            '<a href="' . esc_url($privacy_url) . '" target="_blank">' . esc_html__('Privacy Policy', 'ticketpaygo-lite') . '</a>'
                                        );
                                    }
                                    ?>
                                </label>
                            </div>
                            <?php endif; ?>
                            
                            <div class="form-navigation">
                                <button type="button" class="btn-prev-step"><?php esc_html_e('Back', 'ticketpaygo-lite'); ?></button>
                                <button type="submit" class="btn-complete-booking">
                                    <span class="btn-text"><?php esc_html_e('Complete Booking', 'ticketpaygo-lite'); ?></span>
                                    <span class="btn-loading" style="display: none;">
                                        <span class="spinner"></span>
                                        <?php esc_html_e('Processing...', 'ticketpaygo-lite'); ?>
                                    </span>
                                </button>
                            </div>
                        </div>
                    </form>
                <?php else: ?>
                    <div class="sold-out-message">
                        <span class="dashicons dashicons-warning"></span>
                        <h3><?php esc_html_e('Event Sold Out', 'ticketpaygo-lite'); ?></h3>
                        <p><?php esc_html_e('Unfortunately, this event is completely sold out.', 'ticketpaygo-lite'); ?></p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Progress Indicator -->
        <div class="modal-progress">
            <div class="progress-steps">
                <div class="progress-step active" data-step="1">
                    <span class="step-number">1</span>
                    <span class="step-label"><?php esc_html_e('Tickets', 'ticketpaygo-lite'); ?></span>
                </div>
                <div class="progress-step" data-step="2">
                    <span class="step-number">2</span>
                    <span class="step-label"><?php esc_html_e('Details', 'ticketpaygo-lite'); ?></span>
                </div>
                <div class="progress-step" data-step="3">
                    <span class="step-number">3</span>
                    <span class="step-label"><?php esc_html_e('Payment', 'ticketpaygo-lite'); ?></span>
                </div>
            </div>
            <div class="progress-bar">
                <div class="progress-fill" style="width: 33.33%;"></div>
            </div>
        </div>
    </div>
</div>
