<?php
/**
 * Base Widget Class for TicketPayGo Lite Elementor Widgets
 * 
 * Provides shared functionality for all TicketPayGo Lite Elementor widgets.
 * Handles template rendering, event data access, and common controls.
 */

if (!defined('ABSPATH')) {
    exit;
}

abstract class TicketPayGo_Lite_Elementor_Widget_Base extends \Elementor\Widget_Base {
    
    /**
     * Get widget categories
     */
    public function get_categories() {
        return ['ticketpaygo-lite'];
    }
    
    /**
     * Get widget icon
     */
    public function get_icon() {
        return 'eicon-calendar';
    }
    
    /**
     * Get script dependencies
     */
    public function get_script_depends() {
        return ['tpgl-frontend'];
    }
    
    /**
     * Get style dependencies
     */
    public function get_style_depends() {
        return ['tpgl-frontend'];
    }
    
    /**
     * Get all events for dropdown selection
     */
    protected function get_events_options() {
        $options = ['' => __('Select an Event', 'ticketpaygo-lite')];
        
        if (!class_exists('TicketPayGo_Event')) {
            return $options;
        }
        
        $events = TicketPayGo_Event::get_all_events([
            'status' => 'active',
            'limit' => 100,
            'orderby' => 'start_date',
            'order' => 'ASC'
        ]);
        
        if (!empty($events)) {
            foreach ($events as $event) {
                $date = !empty($event['start_date']) ? date_i18n(get_option('date_format'), strtotime($event['start_date'])) : '';
                $options[$event['id']] = $event['title'] . ($date ? ' (' . $date . ')' : '');
            }
        }
        
        return $options;
    }
    
    /**
     * Get event object by ID
     */
    protected function get_event($event_id) {
        if (!class_exists('TicketPayGo_Event') || empty($event_id)) {
            return null;
        }
        
        $event = new TicketPayGo_Event($event_id);
        
        if (!$event->exists()) {
            return null;
        }
        
        return $event;
    }
    
    /**
     * Render template with data
     * Reuses existing TicketPayGo Lite templates
     */
    protected function render_template($template_name, $data = []) {
        $template_path = TPGL_PLUGIN_PATH . 'templates/' . $template_name;
        
        if (!file_exists($template_path)) {
            if (\Elementor\Plugin::$instance->editor->is_edit_mode()) {
                echo '<div class="ticketpaygo-elementor-notice">';
                echo esc_html__('Template not found: ', 'ticketpaygo-lite') . esc_html($template_name);
                echo '</div>';
            }
            return;
        }
        
        // Extract data to variables for template
        extract($data);
        
        // Include template
        include $template_path;
    }
    
    /**
     * Render empty state for editor
     */
    protected function render_empty_state($message = '') {
        if (empty($message)) {
            $message = __('Please select an event to display.', 'ticketpaygo-lite');
        }
        
        echo '<div class="ticketpaygo-elementor-empty-state" style="padding: 40px; text-align: center; background: #f8f9fa; border-radius: 8px; border: 2px dashed #dee2e6;">';
        echo '<div class="ticketpaygo-elementor-empty-icon" style="margin-bottom: 15px; color: #6c757d;">';
        echo '<svg width="48" height="48" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">';
        echo '<path d="M19 3H18V1H16V3H8V1H6V3H5C3.89 3 3.01 3.9 3.01 5L3 19C3 20.1 3.89 21 5 21H19C20.1 21 21 20.1 21 19V5C21 3.9 20.1 3 19 3ZM19 19H5V8H19V19Z" fill="currentColor" opacity="0.5"/>';
        echo '</svg>';
        echo '</div>';
        echo '<p style="margin: 0; color: #6c757d; font-size: 14px;">' . esc_html($message) . '</p>';
        echo '</div>';
    }
    
    /**
     * Render error state for editor
     */
    protected function render_error_state($message) {
        if (!\Elementor\Plugin::$instance->editor->is_edit_mode()) {
            return;
        }
        
        echo '<div class="ticketpaygo-elementor-error-state" style="padding: 20px; text-align: center; background: #fff3cd; border-radius: 8px; border: 1px solid #ffc107;">';
        echo '<p style="margin: 0; color: #856404;">' . esc_html($message) . '</p>';
        echo '</div>';
    }
    
    /**
     * Add common style controls for cards
     * Note: Lite version has limited styling options
     */
    protected function add_card_style_controls() {
        $this->start_controls_section(
            'section_card_style',
            [
                'label' => __('Card Style', 'ticketpaygo-lite'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );
        
        $this->add_control(
            'card_border_radius',
            [
                'label' => __('Border Radius', 'ticketpaygo-lite'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                    ],
                ],
                'default' => [
                    'size' => 12,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .event-card' => 'border-radius: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'card_box_shadow',
                'selector' => '{{WRAPPER}} .event-card',
            ]
        );
        
        $this->end_controls_section();
    }
    
    /**
     * Add button style controls
     * Note: Lite version has limited styling options
     */
    protected function add_button_style_controls() {
        $this->start_controls_section(
            'section_button_style',
            [
                'label' => __('Button Style', 'ticketpaygo-lite'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );
        
        $this->add_control(
            'button_background',
            [
                'label' => __('Button Background', 'ticketpaygo-lite'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .event-card .buy-ticket-btn' => 'background-color: {{VALUE}};',
                ],
            ]
        );
        
        $this->add_control(
            'button_text_color',
            [
                'label' => __('Button Text Color', 'ticketpaygo-lite'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .event-card .buy-ticket-btn' => 'color: {{VALUE}};',
                ],
            ]
        );
        
        $this->add_control(
            'button_border_radius',
            [
                'label' => __('Button Border Radius', 'ticketpaygo-lite'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 30,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .event-card .buy-ticket-btn' => 'border-radius: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->end_controls_section();
    }
}
