<?php
/**
 * Events Grid Elementor Widget for TicketPayGo Lite
 * 
 * Displays a grid/list of events using the existing TicketPayGo Lite template.
 * Reuses templates/events-list.php for rendering.
 */

if (!defined('ABSPATH')) {
    exit;
}

class TicketPayGo_Lite_Elementor_Events_Grid extends TicketPayGo_Lite_Elementor_Widget_Base {
    
    /**
     * Get widget name
     */
    public function get_name() {
        return 'ticketpaygo-lite-events-grid';
    }
    
    /**
     * Get widget title
     */
    public function get_title() {
        return __('Events Grid', 'ticketpaygo-lite');
    }
    
    /**
     * Get widget icon
     */
    public function get_icon() {
        return 'eicon-gallery-grid';
    }
    
    /**
     * Get widget keywords
     */
    public function get_keywords() {
        return ['events', 'grid', 'list', 'ticketpaygo', 'tickets', 'booking'];
    }
    
    /**
     * Register widget controls
     */
    protected function register_controls() {
        // Content Section
        $this->start_controls_section(
            'section_content',
            [
                'label' => __('Events Query', 'ticketpaygo-lite'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );
        
        $this->add_control(
            'events_filter',
            [
                'label' => __('Show Events', 'ticketpaygo-lite'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'upcoming' => __('Upcoming Events', 'ticketpaygo-lite'),
                    'all' => __('All Active Events', 'ticketpaygo-lite'),
                ],
                'default' => 'upcoming',
                'description' => __('Filter which events to display', 'ticketpaygo-lite'),
            ]
        );
        
        $this->add_control(
            'events_limit',
            [
                'label' => __('Number of Events', 'ticketpaygo-lite'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 1,
                'max' => 50,
                'default' => 10,
                'description' => __('Maximum number of events to display', 'ticketpaygo-lite'),
            ]
        );
        
        $this->end_controls_section();
        
        // Layout Section
        $this->start_controls_section(
            'section_layout',
            [
                'label' => __('Layout', 'ticketpaygo-lite'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );
        
        $this->add_control(
            'layout',
            [
                'label' => __('Layout Style', 'ticketpaygo-lite'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'grid' => __('Grid', 'ticketpaygo-lite'),
                    'list' => __('List', 'ticketpaygo-lite'),
                ],
                'default' => 'grid',
            ]
        );
        
        $this->add_responsive_control(
            'columns',
            [
                'label' => __('Columns', 'ticketpaygo-lite'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    '1' => '1',
                    '2' => '2',
                    '3' => '3',
                    '4' => '4',
                ],
                'default' => '3',
                'tablet_default' => '2',
                'mobile_default' => '1',
                'selectors' => [
                    '{{WRAPPER}} .events-grid' => 'grid-template-columns: repeat({{VALUE}}, 1fr);',
                ],
                'condition' => [
                    'layout' => 'grid',
                ],
            ]
        );
        
        $this->add_responsive_control(
            'gap',
            [
                'label' => __('Gap', 'ticketpaygo-lite'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 60,
                    ],
                ],
                'default' => [
                    'size' => 20,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .events-grid' => 'gap: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->end_controls_section();
        
        // Empty State Section
        $this->start_controls_section(
            'section_empty_state',
            [
                'label' => __('Empty State', 'ticketpaygo-lite'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );
        
        $this->add_control(
            'empty_title',
            [
                'label' => __('Empty Title', 'ticketpaygo-lite'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('No Events Found', 'ticketpaygo-lite'),
                'placeholder' => __('No Events Found', 'ticketpaygo-lite'),
            ]
        );
        
        $this->add_control(
            'empty_message',
            [
                'label' => __('Empty Message', 'ticketpaygo-lite'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => __('There are currently no events available. Check back soon for upcoming events!', 'ticketpaygo-lite'),
                'placeholder' => __('There are currently no events available.', 'ticketpaygo-lite'),
            ]
        );
        
        $this->end_controls_section();
        
        // Add style controls from base class
        $this->add_card_style_controls();
        $this->add_button_style_controls();
        
        // Grid Style Section
        $this->start_controls_section(
            'section_grid_style',
            [
                'label' => __('Grid Style', 'ticketpaygo-lite'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );
        
        $this->add_responsive_control(
            'grid_padding',
            [
                'label' => __('Padding', 'ticketpaygo-lite'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .ticketpaygo-events-list' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->end_controls_section();
    }
    
    /**
     * Render widget output
     */
    protected function render() {
        $settings = $this->get_settings_for_display();
        
        // Get events based on filter
        $events = $this->get_events($settings);
        
        // Set layout variable for template
        $layout = $settings['layout'];
        
        // Render wrapper with custom classes
        echo '<div class="ticketpaygo-elementor-events-grid-wrapper">';
        
        // Include the existing template
        include TPGL_PLUGIN_PATH . 'templates/events-list.php';
        
        // Render booking modals for all events
        $this->render_booking_modals($events);
        
        echo '</div>';
    }
    
    /**
     * Get events based on settings
     */
    private function get_events($settings) {
        if (!class_exists('TicketPayGo_Event')) {
            return [];
        }
        
        $limit = !empty($settings['events_limit']) ? intval($settings['events_limit']) : 10;
        
        if ($settings['events_filter'] === 'upcoming') {
            return TicketPayGo_Event::get_upcoming_events($limit);
        }
        
        return TicketPayGo_Event::get_all_events([
            'status' => 'active',
            'limit' => $limit,
        ]);
    }
    
    /**
     * Render booking modals for all events
     */
    private function render_booking_modals($events) {
        if (empty($events)) {
            return;
        }
        
        $modal_path = TPGL_PLUGIN_PATH . 'templates/booking-modal.php';
        
        if (!file_exists($modal_path)) {
            return;
        }
        
        foreach ($events as $event_data) {
            $event_id = is_array($event_data) ? $event_data['id'] : $event_data->id;
            include $modal_path;
        }
    }
    
    /**
     * Render widget output in the editor (live preview)
     */
    protected function content_template() {
        ?>
        <div class="ticketpaygo-elementor-events-grid-wrapper">
            <div class="ticketpaygo-elementor-preview-placeholder" style="padding: 40px; text-align: center; background: #e8f4fd; border-radius: 8px; border: 1px solid #bee5eb;">
                <div class="ticketpaygo-elementor-empty-icon" style="margin-bottom: 15px; color: #0c5460;">
                    <svg width="48" height="48" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M4 8h4V4H4v4zm6 12h4v-4h-4v4zm-6 0h4v-4H4v4zm0-6h4v-4H4v4zm6 0h4v-4h-4v4zm6-10v4h4V4h-4zm-6 4h4V4h-4v4zm6 6h4v-4h-4v4zm0 6h4v-4h-4v4z" fill="currentColor" opacity="0.5"/>
                    </svg>
                </div>
                <p style="margin: 0 0 5px 0; font-weight: 600; color: #0c5460;"><?php echo esc_html__('Events Grid Preview', 'ticketpaygo-lite'); ?></p>
                <small style="color: #0c5460;"><?php echo esc_html__('The actual events grid will be displayed on the frontend.', 'ticketpaygo-lite'); ?></small>
            </div>
        </div>
        <?php
    }
}
