<?php
/**
 * Event Card Elementor Widget for TicketPayGo Lite
 * 
 * Displays a single event card using the existing TicketPayGo Lite template.
 * Reuses templates/event-default.php for rendering.
 */

if (!defined('ABSPATH')) {
    exit;
}

class TicketPayGo_Lite_Elementor_Event_Card extends TicketPayGo_Lite_Elementor_Widget_Base {
    
    /**
     * Get widget name
     */
    public function get_name() {
        return 'ticketpaygo-lite-event-card';
    }
    
    /**
     * Get widget title
     */
    public function get_title() {
        return __('Event Card', 'ticketpaygo-lite');
    }
    
    /**
     * Get widget icon
     */
    public function get_icon() {
        return 'eicon-single-post';
    }
    
    /**
     * Get widget keywords
     */
    public function get_keywords() {
        return ['event', 'ticket', 'card', 'ticketpaygo', 'booking'];
    }
    
    /**
     * Register widget controls
     */
    protected function register_controls() {
        // Content Section
        $this->start_controls_section(
            'section_content',
            [
                'label' => __('Event Selection', 'ticketpaygo-lite'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );
        
        $this->add_control(
            'event_id',
            [
                'label' => __('Select Event', 'ticketpaygo-lite'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => $this->get_events_options(),
                'default' => '',
                'description' => __('Choose the event to display', 'ticketpaygo-lite'),
            ]
        );
        
        $this->end_controls_section();
        
        // Layout Section
        $this->start_controls_section(
            'section_layout',
            [
                'label' => __('Layout', 'ticketpaygo-lite'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );
        
        $this->add_responsive_control(
            'card_alignment',
            [
                'label' => __('Alignment', 'ticketpaygo-lite'),
                'type' => \Elementor\Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => __('Left', 'ticketpaygo-lite'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => __('Center', 'ticketpaygo-lite'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => __('Right', 'ticketpaygo-lite'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'center',
                'selectors' => [
                    '{{WRAPPER}}' => 'text-align: {{VALUE}}; display: flex; justify-content: {{VALUE}};',
                ],
            ]
        );
        
        $this->end_controls_section();
        
        // Add style controls from base class
        $this->add_card_style_controls();
        $this->add_button_style_controls();
    }
    
    /**
     * Render widget output
     */
    protected function render() {
        $settings = $this->get_settings_for_display();
        $event_id = $settings['event_id'];
        
        // Check if event is selected
        if (empty($event_id)) {
            $this->render_empty_state(__('Please select an event to display.', 'ticketpaygo-lite'));
            return;
        }
        
        // Get event
        $event = $this->get_event($event_id);
        
        if (!$event) {
            $this->render_error_state(__('Event not found or no longer available.', 'ticketpaygo-lite'));
            return;
        }
        
        // Render wrapper
        echo '<div class="ticketpaygo-elementor-event-card-wrapper">';
        
        // Include the existing template
        // The template expects $event to be a TicketPayGo_Event object
        include TPGL_PLUGIN_PATH . 'templates/event-default.php';
        
        // Include booking modal for this event
        $this->render_booking_modal($event_id);
        
        echo '</div>';
    }
    
    /**
     * Render booking modal
     */
    private function render_booking_modal($event_id) {
        $modal_path = TPGL_PLUGIN_PATH . 'templates/booking-modal.php';
        
        if (file_exists($modal_path)) {
            include $modal_path;
        }
    }
    
    /**
     * Render widget output in the editor (live preview)
     */
    protected function content_template() {
        ?>
        <#
        if (!settings.event_id) {
            #>
            <div class="ticketpaygo-elementor-empty-state" style="padding: 40px; text-align: center; background: #f8f9fa; border-radius: 8px; border: 2px dashed #dee2e6;">
                <div class="ticketpaygo-elementor-empty-icon" style="margin-bottom: 15px; color: #6c757d;">
                    <svg width="48" height="48" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M19 3H18V1H16V3H8V1H6V3H5C3.89 3 3.01 3.9 3.01 5L3 19C3 20.1 3.89 21 5 21H19C20.1 21 21 20.1 21 19V5C21 3.9 20.1 3 19 3ZM19 19H5V8H19V19Z" fill="currentColor" opacity="0.5"/>
                    </svg>
                </div>
                <p style="margin: 0; color: #6c757d; font-size: 14px;"><?php echo esc_html__('Please select an event to display.', 'ticketpaygo-lite'); ?></p>
            </div>
            <#
        } else {
            #>
            <div class="ticketpaygo-elementor-event-card-wrapper">
                <div class="ticketpaygo-elementor-preview-placeholder" style="padding: 40px; text-align: center; background: #e8f4fd; border-radius: 8px; border: 1px solid #bee5eb;">
                    <p style="margin: 0 0 5px 0; font-weight: 600; color: #0c5460;"><?php echo esc_html__('Event Card Preview', 'ticketpaygo-lite'); ?></p>
                    <small style="color: #0c5460;"><?php echo esc_html__('The actual event card will be displayed on the frontend.', 'ticketpaygo-lite'); ?></small>
                </div>
            </div>
            <#
        }
        #>
        <?php
    }
}
