<?php
/**
 * Elementor Integration Loader for TicketPayGo Lite
 * 
 * Conditionally loads Elementor integration only when Elementor is active.
 * This class serves as the main entry point for the Elementor module.
 */

if (!defined('ABSPATH')) {
    exit;
}

class TicketPayGo_Lite_Elementor_Loader {
    
    /**
     * Minimum Elementor Version
     */
    const MINIMUM_ELEMENTOR_VERSION = '3.0.0';
    
    /**
     * Single instance
     */
    private static $instance = null;
    
    /**
     * Get instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Initialize the Elementor integration
     * Called only when Elementor is confirmed to be active
     */
    public static function init() {
        // Check if Elementor is loaded
        if (!did_action('elementor/loaded')) {
            return;
        }
        
        // Check Elementor version
        if (!version_compare(ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=')) {
            add_action('admin_notices', [__CLASS__, 'admin_notice_minimum_elementor_version']);
            return;
        }
        
        // Load the integration
        self::get_instance();
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->load_dependencies();
        $this->init_hooks();
    }
    
    /**
     * Load required files
     */
    private function load_dependencies() {
        $elementor_path = TPGL_PLUGIN_PATH . 'includes/elementor/';
        
        // Load base widget class
        require_once $elementor_path . 'widgets/class-widget-base.php';
        
        // Load widgets (Lite version: Event Card and Events Grid only)
        require_once $elementor_path . 'widgets/class-event-card.php';
        require_once $elementor_path . 'widgets/class-events-grid.php';
    }
    
    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Register widget category
        add_action('elementor/elements/categories_registered', [$this, 'register_widget_category']);
        
        // Register widgets
        add_action('elementor/widgets/register', [$this, 'register_widgets']);
    }
    
    /**
     * Register TicketPayGo widget category
     */
    public function register_widget_category($elements_manager) {
        $elements_manager->add_category(
            'ticketpaygo-lite',
            [
                'title' => __('TicketPayGo Lite', 'ticketpaygo-lite'),
                'icon' => 'eicon-calendar',
            ]
        );
    }
    
    /**
     * Register Elementor widgets
     */
    public function register_widgets($widgets_manager) {
        // Register Event Card widget
        $widgets_manager->register(new TicketPayGo_Lite_Elementor_Event_Card());
        
        // Register Events Grid widget
        $widgets_manager->register(new TicketPayGo_Lite_Elementor_Events_Grid());
    }
    
    /**
     * Admin notice for minimum Elementor version
     */
    public static function admin_notice_minimum_elementor_version() {
        if (isset($_GET['activate'])) {
            unset($_GET['activate']);
        }
        
        $message = sprintf(
            /* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
            esc_html__('"%1$s" requires "%2$s" version %3$s or greater.', 'ticketpaygo-lite'),
            '<strong>' . esc_html__('TicketPayGo Lite Elementor Integration', 'ticketpaygo-lite') . '</strong>',
            '<strong>' . esc_html__('Elementor', 'ticketpaygo-lite') . '</strong>',
            self::MINIMUM_ELEMENTOR_VERSION
        );
        
        printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
    }
    
    /**
     * Check if we're in Elementor editor
     */
    public static function is_elementor_editor() {
        return \Elementor\Plugin::$instance->editor->is_edit_mode() || 
               \Elementor\Plugin::$instance->preview->is_preview_mode();
    }
}
