<?php
/**
 * Ticket management class
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Ticket class for TicketPayGo Lite
 * 
 * phpcs:disable PluginCheck.Security.DirectDB.UnescapedDBParameter
 */

class TicketPayGo_Ticket {
    
    private $id;
    private $data;
    
    /**
     * Constructor
     */
    public function __construct($id = 0) {
        if ($id > 0) {
            $this->id = $id;
            $this->load_ticket();
        }
    }
    
    /**
     * Load ticket data
     */
    private function load_ticket() {
        global $wpdb;
        
        $tickets_table = $wpdb->prefix . 'ticketpaygo_lite_tickets';
        
        $this->data = $wpdb->get_row($wpdb->prepare( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            "SELECT t.*, e.title as event_title, e.start_date, e.end_date, e.location, e.address, o.customer_name, o.customer_email
             FROM {$wpdb->prefix}ticketpaygo_lite_tickets t
             LEFT JOIN {$wpdb->prefix}ticketpaygo_lite_events e ON t.event_id = e.id
             LEFT JOIN {$wpdb->prefix}ticketpaygo_lite_orders o ON t.order_id = o.id
             WHERE t.id = %d",
            $this->id
        ), ARRAY_A);
        
        if ($this->data) {
            // Decode JSON fields
            $this->data['qr_data'] = json_decode($this->data['qr_data'], true) ?: array();
            $this->data['custom_design'] = json_decode($this->data['custom_design'], true) ?: array();
        }
    }
    
    /**
     * Check if ticket exists
     */
    public function exists() {
        return !empty($this->data);
    }
    
    /**
     * Get ticket data
     */
    public function get_data() {
        return $this->data;
    }
    
    /**
     * Get specific field
     */
    public function get($field) {
        return isset($this->data[$field]) ? $this->data[$field] : null;
    }
    
    /**
     * Create new ticket
     */
    public function create($order_id, $event_id, $ticket_data = array()) {
        global $wpdb;
        
        $tickets_table = $wpdb->prefix . 'ticketpaygo_lite_tickets';
        
        // Generate unique ticket number
        $ticket_number = $this->generate_ticket_number();
        
        // Prepare ticket data (QR codes not available in Lite version)
        $data = array(
            'ticket_number' => $ticket_number,
            'order_id' => intval($order_id),
            'event_id' => intval($event_id),
            'ticket_type' => sanitize_text_field(isset($ticket_data['ticket_type']) ? $ticket_data['ticket_type'] : 'general'),
            'holder_name' => sanitize_text_field(isset($ticket_data['holder_name']) ? $ticket_data['holder_name'] : ''),
            'holder_email' => sanitize_email(isset($ticket_data['holder_email']) ? $ticket_data['holder_email'] : ''),
            'qr_data' => '',
            'background_image' => sanitize_url(isset($ticket_data['background_image']) ? $ticket_data['background_image'] : ''),
            'custom_design' => json_encode(isset($ticket_data['custom_design']) ? $ticket_data['custom_design'] : array()),
            'status' => 'active'
        );
        
        $result = $wpdb->insert($tickets_table, $data); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        
        if ($result !== false) {
            $this->id = $wpdb->insert_id;
            
            $this->load_ticket();
            return $this->id;
        }
        
        return false;
    }
    
    /**
     * Generate unique ticket number
     */
    private function generate_ticket_number() {
        $prefix = 'TPG';
        $timestamp = time();
        $random = wp_rand(1000, 9999);
        return $prefix . $timestamp . $random;
    }
    
    
    /**
     * Update ticket background
     */
    public function update_background($background_image) {
        global $wpdb;
        
        if (!$this->exists()) {
            return false;
        }
        
        $tickets_table = $wpdb->prefix . 'ticketpaygo_lite_tickets';
        
        $result = $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            $tickets_table,
            array('background_image' => sanitize_url($background_image)),
            array('id' => $this->id),
            array('%s'),
            array('%d')
        );
        
        if ($result !== false) {
            $this->data['background_image'] = $background_image;
            return true;
        }
        
        return false;
    }
    
    /**
     * Update custom design
     */
    public function update_design($design_data) {
        global $wpdb;
        
        if (!$this->exists()) {
            return false;
        }
        
        $tickets_table = $wpdb->prefix . 'ticketpaygo_lite_tickets';
        
        $result = $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            $tickets_table,
            array('custom_design' => json_encode($design_data)),
            array('id' => $this->id),
            array('%s'),
            array('%d')
        );
        
        if ($result !== false) {
            $this->data['custom_design'] = $design_data;
            return true;
        }
        
        return false;
    }
    
    /**
     * Scan ticket (mark as used)
     */
    public function scan($scanned_by = '') {
        global $wpdb;
        
        if (!$this->exists()) {
            return array('success' => false, 'message' => 'Ticket not found');
        }
        
        if ($this->get('status') !== 'active') {
            return array('success' => false, 'message' => 'Ticket is not active');
        }
        
        if ($this->get('scanned_at')) {
            return array('success' => false, 'message' => 'Ticket already scanned');
        }
        
        $tickets_table = $wpdb->prefix . 'ticketpaygo_lite_tickets';
        
        $result = $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            $tickets_table,
            array(
                'scanned_at' => current_time('mysql'),
                'scanned_by' => sanitize_text_field($scanned_by)
            ),
            array('id' => $this->id),
            array('%s', '%s'),
            array('%d')
        );
        
        if ($result !== false) {
            $this->load_ticket();
            return array('success' => true, 'message' => 'Ticket scanned successfully');
        }
        
        return array('success' => false, 'message' => 'Failed to scan ticket');
    }
    
    /**
     * Verify ticket
     */
    public function verify() {
        if (!$this->exists()) {
            return array('valid' => false, 'message' => 'Ticket not found');
        }
        
        if ($this->get('status') !== 'active') {
            return array('valid' => false, 'message' => 'Ticket is not active');
        }
        
        // Check if event has ended
        $event = new TicketPayGo_Event($this->get('event_id'));
        if ($event->has_ended()) {
            return array('valid' => false, 'message' => 'Event has ended');
        }
        
        return array(
            'valid' => true,
            'message' => 'Ticket is valid',
            'ticket_data' => $this->get_data()
        );
    }
    
    /**
     * Generate ticket PDF
     * Note: PDF generation not available in Lite version
     */
    public function generate_pdf() {
        // PDF generation requires the Full version
        return false;
    }
    
    /**
     * Get ticket HTML for display
     */
    public function get_html($template = 'default') {
        if (!$this->exists()) {
            return '';
        }
        
        ob_start();
        
        $template_file = TICKETPAYGO_PLUGIN_PATH . "templates/ticket-{$template}.php";
        if (file_exists($template_file)) {
            include $template_file;
        } else {
            include TICKETPAYGO_PLUGIN_PATH . 'templates/ticket-default.php';
        }
        
        return ob_get_clean();
    }
    
    /**
     * Get tickets by order ID
     */
    public static function get_tickets_by_order($order_id) {
        global $wpdb;
        
        $tickets_table = $wpdb->prefix . 'ticketpaygo_lite_tickets';
        
        $tickets = $wpdb->get_results($wpdb->prepare( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            "SELECT t.*, e.title as event_title, e.start_date, e.end_date, e.location
             FROM {$wpdb->prefix}ticketpaygo_lite_tickets t
             LEFT JOIN {$wpdb->prefix}ticketpaygo_lite_events e ON t.event_id = e.id
             WHERE t.order_id = %d
             ORDER BY t.id ASC",
            $order_id
        ), ARRAY_A);
        
        // Decode JSON fields for each ticket
        foreach ($tickets as &$ticket) {
            $ticket['qr_data'] = json_decode($ticket['qr_data'], true) ?: array();
            $ticket['custom_design'] = json_decode($ticket['custom_design'], true) ?: array();
        }
        
        return $tickets;
    }
    
    /**
     * Get tickets by event ID
     */
    public static function get_tickets_by_event($event_id) {
        global $wpdb;
        
        $tickets_table = $wpdb->prefix . 'ticketpaygo_lite_tickets';
        
        $tickets = $wpdb->get_results($wpdb->prepare( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            "SELECT t.*, o.customer_name, o.customer_email
             FROM {$wpdb->prefix}ticketpaygo_lite_tickets t
             LEFT JOIN {$wpdb->prefix}ticketpaygo_lite_orders o ON t.order_id = o.id
             WHERE t.event_id = %d
             ORDER BY t.created_at DESC",
            $event_id
        ), ARRAY_A);
        
        // Decode JSON fields for each ticket
        foreach ($tickets as &$ticket) {
            $ticket['qr_data'] = json_decode($ticket['qr_data'], true) ?: array();
            $ticket['custom_design'] = json_decode($ticket['custom_design'], true) ?: array();
        }
        
        return $tickets;
    }
    
    /**
     * Search tickets
     */
    public static function search_tickets($search_term) {
        global $wpdb;
        
        $tickets_table = $wpdb->prefix . 'ticketpaygo_lite_tickets';
        
        $search_term = '%' . $wpdb->esc_like($search_term) . '%';
        
        $tickets = $wpdb->get_results($wpdb->prepare( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            "SELECT t.*, e.title as event_title, o.customer_name, o.customer_email
             FROM {$wpdb->prefix}ticketpaygo_lite_tickets t
             LEFT JOIN {$wpdb->prefix}ticketpaygo_lite_events e ON t.event_id = e.id
             LEFT JOIN {$wpdb->prefix}ticketpaygo_lite_orders o ON t.order_id = o.id
             WHERE t.ticket_number LIKE %s 
                OR t.holder_name LIKE %s 
                OR t.holder_email LIKE %s
                OR o.customer_name LIKE %s
                OR o.customer_email LIKE %s
             ORDER BY t.created_at DESC",
            $search_term, $search_term, $search_term, $search_term, $search_term
        ), ARRAY_A);
        
        // Decode JSON fields for each ticket
        foreach ($tickets as &$ticket) {
            $ticket['qr_data'] = json_decode($ticket['qr_data'], true) ?: array();
            $ticket['custom_design'] = json_decode($ticket['custom_design'], true) ?: array();
        }
        
        return $tickets;
    }
    
    /**
     * Get ticket statistics
     */
    public static function get_ticket_stats($event_id = null) {
        global $wpdb;
        
        $tickets_table = $wpdb->prefix . 'ticketpaygo_lite_tickets';
        
        $where_clause = '';
        $where_values = array();
        
        if ($event_id) {
            $where_clause = 'WHERE event_id = %d';
            $where_values[] = $event_id;
        }
        
        $query = "SELECT 
                    COUNT(*) as total_tickets,
                    SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active_tickets,
                    SUM(CASE WHEN scanned_at IS NOT NULL THEN 1 ELSE 0 END) as scanned_tickets,
                    SUM(CASE WHEN status = 'cancelled' THEN 1 ELSE 0 END) as cancelled_tickets
                  FROM $tickets_table $where_clause";
        
        if (!empty($where_values)) {
            $query = $wpdb->prepare($query, $where_values); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query is dynamically built with proper placeholders
        }
        
        return $wpdb->get_row($query, ARRAY_A); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared,WordPress.DB.DirectDatabaseQuery
    }
}
