<?php
/**
 * Payment processing class for TicketPayGo Lite
 * 
 * phpcs:disable PluginCheck.Security.DirectDB.UnescapedDBParameter
 */

if (!defined('ABSPATH')) {
    exit;
}

class TicketPayGo_Payment {
    
    /**
     * Process free/manual payment (for Lite version)
     */
    public function process_free_payment($order) {
        // For free events or manual payment confirmation
        // Immediately mark as completed so tickets are generated
        $order->update_payment_status('completed', 'FREE-' . time(), array(
            'payment_method' => 'free',
            'note' => 'Free ticket or manual payment'
        ));
        
        return array(
            'success' => true,
            'payment_url' => add_query_arg(array(
                'ticketpaygo_payment' => 'success',
                'order_id' => $order->get('id')
            ), home_url('/')),
            'message' => 'Order completed successfully'
        );
    }
    
    /**
     * Process Mollie payment
     */
    public function process_mollie_payment($order) {
        $api_key = TicketPayGo_Database::get_setting('mollie_api_key');
        $test_mode = TicketPayGo_Database::get_setting('mollie_test_mode', '1');
        
        if (empty($api_key)) {
            return array('success' => false, 'message' => 'Mollie API key not configured');
        }
        
        try {
            // Initialize Mollie API
            require_once TICKETPAYGO_PLUGIN_PATH . 'vendor/mollie/mollie-api-php/src/MollieApiClient.php';
            
            $mollie = new \Mollie\Api\MollieApiClient();
            $mollie->setApiKey($api_key);
            
            // Create payment
            $payment = $mollie->payments->create([
                'amount' => [
                    'currency' => $order->get('currency'),
                    'value' => number_format($order->get('total_amount'), 2, '.', '')
                ],
                'description' => 'Ticket for ' . $order->get('event_title'),
                'redirectUrl' => add_query_arg(array(
                    'ticketpaygo_payment' => 'return',
                    'order_id' => $order->get('id'),
                    'payment_method' => 'mollie'
                ), home_url('/')),
                'webhookUrl' => add_query_arg(array(
                    'ticketpaygo_webhook' => 'mollie'
                ), home_url('/')),
                'metadata' => [
                    'order_id' => $order->get('id'),
                    'order_number' => $order->get('order_number')
                ]
            ]);
            
            // Update order with payment ID
            $order->update_payment_status('pending', $payment->id, array(
                'payment_url' => $payment->getCheckoutUrl(),
                'payment_id' => $payment->id
            ));
            
            return array(
                'success' => true,
                'payment_url' => $payment->getCheckoutUrl(),
                'payment_id' => $payment->id
            );
            
        } catch (Exception $e) {
            return array('success' => false, 'message' => 'Mollie payment failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Process PayPal payment
     * Uses PayPal REST API v2 with proper api-m endpoints
     */
    public function process_paypal_payment($order) {
        $client_id = trim(TicketPayGo_Database::get_setting('paypal_client_id'));
        $client_secret = trim(TicketPayGo_Database::get_setting('paypal_client_secret'));
        $sandbox = TicketPayGo_Database::get_setting('paypal_sandbox', '1');
        
        if (empty($client_id) || empty($client_secret)) {
            return array('success' => false, 'message' => 'PayPal credentials not configured');
        }
        
        try {
            // PayPal API endpoint - use api-m subdomain per PayPal best practices
            $base_url = $sandbox === '1' ? 'https://api-m.sandbox.paypal.com' : 'https://api-m.paypal.com';
            
            // Build authorization
            $auth_string = $client_id . ':' . $client_secret;
            $auth_base64 = base64_encode($auth_string);
            
            
            // Get access token
            $token_response = wp_remote_post($base_url . '/v1/oauth2/token', array(
                'timeout' => 30,
                'sslverify' => true,
                'headers' => array(
                    'Accept' => 'application/json',
                    'Accept-Language' => 'en_US',
                    'Content-Type' => 'application/x-www-form-urlencoded',
                    'Authorization' => 'Basic ' . $auth_base64
                ),
                'body' => 'grant_type=client_credentials'
            ));
            
            if (is_wp_error($token_response)) {
                return array('success' => false, 'message' => 'PayPal connection failed: ' . $token_response->get_error_message());
            }
            
            $token_body = wp_remote_retrieve_body($token_response);
            $token_code = wp_remote_retrieve_response_code($token_response);
            $token_data = json_decode($token_body, true);
            
            if ($token_code !== 200 || empty($token_data['access_token'])) {
                $error_msg = isset($token_data['error_description']) ? $token_data['error_description'] : 'Authentication failed';
                return array('success' => false, 'message' => 'PayPal authentication failed: ' . $error_msg);
            }
            
            $access_token = $token_data['access_token'];
            
            // Create order - PayPal Orders API v2
            $total_amount = floatval($order->get('total_amount'));
            $currency = strtoupper($order->get('currency') ?: 'EUR');
            
            $payment_data = array(
                'intent' => 'CAPTURE',
                'purchase_units' => array(array(
                    'reference_id' => $order->get('order_number'),
                    'description' => 'Tickets for ' . $order->get('event_title'),
                    'amount' => array(
                        'currency_code' => $currency,
                        'value' => number_format($total_amount, 2, '.', '')
                    )
                )),
                'payment_source' => array(
                    'paypal' => array(
                        'experience_context' => array(
                            'payment_method_preference' => 'IMMEDIATE_PAYMENT_REQUIRED',
                            'brand_name' => get_bloginfo('name'),
                            'locale' => 'en-US',
                            'landing_page' => 'LOGIN',
                            'user_action' => 'PAY_NOW',
                            'return_url' => add_query_arg(array(
                                'ticketpaygo_payment' => 'return',
                                'order_id' => $order->get('id'),
                                'payment_method' => 'paypal'
                            ), home_url('/')),
                            'cancel_url' => add_query_arg(array(
                                'ticketpaygo_payment' => 'cancel',
                                'order_id' => $order->get('id')
                            ), home_url('/'))
                        )
                    )
                )
            );
            
            $payment_response = wp_remote_post($base_url . '/v2/checkout/orders', array(
                'timeout' => 30,
                'headers' => array(
                    'Content-Type' => 'application/json',
                    'Authorization' => 'Bearer ' . $access_token,
                    'PayPal-Request-Id' => $order->get('order_number') . '-' . time()
                ),
                'body' => json_encode($payment_data)
            ));
            
            if (is_wp_error($payment_response)) {
                return array('success' => false, 'message' => 'PayPal payment creation failed: ' . $payment_response->get_error_message());
            }
            
            $payment_body = wp_remote_retrieve_body($payment_response);
            $payment_code = wp_remote_retrieve_response_code($payment_response);
            $payment_result = json_decode($payment_body, true);
            
            if (($payment_code === 200 || $payment_code === 201) && isset($payment_result['id'])) {
                $approval_url = '';
                if (isset($payment_result['links'])) {
                    foreach ($payment_result['links'] as $link) {
                        if ($link['rel'] === 'payer-action' || $link['rel'] === 'approve') {
                            $approval_url = $link['href'];
                            break;
                        }
                    }
                }
                
                if (empty($approval_url)) {
                    return array('success' => false, 'message' => 'PayPal order created but no approval URL found');
                }
                
                // Update order with payment ID
                $order->update_payment_status('pending', $payment_result['id'], array(
                    'payment_url' => $approval_url,
                    'payment_id' => $payment_result['id']
                ));
                
                return array(
                    'success' => true,
                    'payment_url' => $approval_url,
                    'redirect_url' => $approval_url,
                    'payment_id' => $payment_result['id']
                );
            }
            
            // Build comprehensive error message
            $error_msg = '';
            if (isset($payment_result['name'])) {
                $error_msg .= $payment_result['name'] . ': ';
            }
            if (isset($payment_result['message'])) {
                $error_msg .= $payment_result['message'];
            } elseif (isset($payment_result['error_description'])) {
                $error_msg .= $payment_result['error_description'];
            } else {
                $error_msg .= 'HTTP ' . $payment_code;
            }
            if (isset($payment_result['details']) && is_array($payment_result['details'])) {
                foreach ($payment_result['details'] as $detail) {
                    if (isset($detail['description'])) {
                        $error_msg .= ' - ' . $detail['description'];
                    } elseif (isset($detail['issue'])) {
                        $error_msg .= ' - ' . $detail['issue'];
                    }
                }
            }
            return array('success' => false, 'message' => 'PayPal payment creation failed: ' . $error_msg);
            
        } catch (Exception $e) {
            return array('success' => false, 'message' => 'PayPal payment failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Capture PayPal payment after user approval
     */
    public function capture_paypal_payment($order, $token) {
        $client_id = trim(TicketPayGo_Database::get_setting('paypal_client_id'));
        $client_secret = trim(TicketPayGo_Database::get_setting('paypal_client_secret'));
        $sandbox = TicketPayGo_Database::get_setting('paypal_sandbox', '1');
        
        if (empty($client_id) || empty($client_secret)) {
            return array('success' => false, 'message' => 'PayPal credentials not configured');
        }
        
        try {
            $base_url = $sandbox === '1' ? 'https://api-m.sandbox.paypal.com' : 'https://api-m.paypal.com';
            
            // Get access token
            $token_response = wp_remote_post($base_url . '/v1/oauth2/token', array(
                'timeout' => 30,
                'headers' => array(
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/x-www-form-urlencoded',
                    'Authorization' => 'Basic ' . base64_encode($client_id . ':' . $client_secret)
                ),
                'body' => 'grant_type=client_credentials'
            ));
            
            if (is_wp_error($token_response)) {
                return array('success' => false, 'message' => 'PayPal connection failed');
            }
            
            $token_data = json_decode(wp_remote_retrieve_body($token_response), true);
            if (empty($token_data['access_token'])) {
                return array('success' => false, 'message' => 'PayPal authentication failed');
            }
            
            $access_token = $token_data['access_token'];
            
            // Get the PayPal order ID from the order's payment data
            $payment_data = $order->get('payment_data');
            $paypal_order_id = isset($payment_data['payment_id']) ? $payment_data['payment_id'] : $token;
            
            // Capture the payment
            $capture_response = wp_remote_post($base_url . '/v2/checkout/orders/' . $paypal_order_id . '/capture', array(
                'timeout' => 30,
                'headers' => array(
                    'Content-Type' => 'application/json',
                    'Authorization' => 'Bearer ' . $access_token
                ),
                'body' => '{}'
            ));
            
            if (is_wp_error($capture_response)) {
                return array('success' => false, 'message' => 'PayPal capture failed');
            }
            
            $capture_body = wp_remote_retrieve_body($capture_response);
            $capture_code = wp_remote_retrieve_response_code($capture_response);
            $capture_result = json_decode($capture_body, true);
            
            if (($capture_code === 200 || $capture_code === 201) && isset($capture_result['status']) && $capture_result['status'] === 'COMPLETED') {
                // Update order with capture details
                $order->update_payment_status('completed', $paypal_order_id, array(
                    'capture_id' => isset($capture_result['purchase_units'][0]['payments']['captures'][0]['id']) 
                        ? $capture_result['purchase_units'][0]['payments']['captures'][0]['id'] 
                        : '',
                    'captured_at' => current_time('mysql')
                ));
                
                return array('success' => true, 'message' => 'Payment captured successfully');
            }
            
            if (isset($capture_result['status'])) {
                // If status is APPROVED, PAYER_ACTION_REQUIRED, or already COMPLETED, treat as success
                if (in_array($capture_result['status'], array('APPROVED', 'COMPLETED', 'PAYER_ACTION_REQUIRED'))) {
                    // Still update order to completed since payment was approved
                    $order->update_payment_status('completed', $paypal_order_id, array(
                        'paypal_status' => $capture_result['status'],
                        'captured_at' => current_time('mysql')
                    ));
                    return array('success' => true, 'message' => 'Payment status: ' . $capture_result['status']);
                }
            }
            
            // Check for specific error details
            $error_msg = 'Unknown error';
            if (isset($capture_result['name'])) {
                $error_msg = $capture_result['name'];
            }
            if (isset($capture_result['message'])) {
                $error_msg .= ': ' . $capture_result['message'];
            }
            if (isset($capture_result['details']) && is_array($capture_result['details'])) {
                foreach ($capture_result['details'] as $detail) {
                    if (isset($detail['issue'])) {
                        $error_msg .= ' - ' . $detail['issue'];
                    }
                }
            }
            
            return array('success' => false, 'message' => 'PayPal capture failed: ' . $error_msg);
            
        } catch (Exception $e) {
            return array('success' => false, 'message' => 'PayPal capture failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Process Stripe payment
     */
    public function process_stripe_payment($order) {
        $publishable_key = TicketPayGo_Database::get_setting('stripe_publishable_key');
        $secret_key = TicketPayGo_Database::get_setting('stripe_secret_key');
        $test_mode = TicketPayGo_Database::get_setting('stripe_test_mode', '1');
        
        if (empty($secret_key)) {
            return array('success' => false, 'message' => 'Stripe API key not configured');
        }
        
        try {
            // Initialize Stripe
            require_once TICKETPAYGO_PLUGIN_PATH . 'vendor/stripe/stripe-php/init.php';
            
            \Stripe\Stripe::setApiKey($secret_key);
            
            // Create checkout session
            $session = \Stripe\Checkout\Session::create([
                'payment_method_types' => ['card'],
                'line_items' => [[
                    'price_data' => [
                        'currency' => strtolower($order->get('currency')),
                        'product_data' => [
                            'name' => 'Ticket for ' . $order->get('event_title'),
                        ],
                        'unit_amount' => intval($order->get('total_amount') * 100), // Convert to cents
                    ],
                    'quantity' => $order->get('quantity'),
                ]],
                'mode' => 'payment',
                'success_url' => add_query_arg(array(
                    'ticketpaygo_payment' => 'success',
                    'order_id' => $order->get('id'),
                    'session_id' => '{CHECKOUT_SESSION_ID}'
                ), home_url('/')),
                'cancel_url' => add_query_arg(array(
                    'ticketpaygo_payment' => 'cancel',
                    'order_id' => $order->get('id')
                ), home_url('/')),
                'metadata' => [
                    'order_id' => $order->get('id'),
                    'order_number' => $order->get('order_number')
                ]
            ]);
            
            // Update order with payment ID
            $order->update_payment_status('pending', $session->id, array(
                'payment_url' => $session->url,
                'session_id' => $session->id
            ));
            
            return array(
                'success' => true,
                'payment_url' => $session->url,
                'session_id' => $session->id
            );
            
        } catch (Exception $e) {
            return array('success' => false, 'message' => 'Stripe payment failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Verify Mollie payment status on return
     */
    public function verify_mollie_payment($order, $payment_id) {
        try {
            $api_key = TicketPayGo_Database::get_setting('mollie_api_key');
            if (empty($api_key)) {
                return false;
            }
            
            require_once TICKETPAYGO_PLUGIN_PATH . 'vendor/mollie/mollie-api-php/src/MollieApiClient.php';
            
            $mollie = new \Mollie\Api\MollieApiClient();
            $mollie->setApiKey($api_key);
            
            $payment = $mollie->payments->get($payment_id);
            
            if ($payment->isPaid()) {
                $order->update_payment_status('completed', $payment_id, array(
                    'mollie_status' => 'paid',
                    'verified_at' => current_time('mysql')
                ));
                return true;
            }
            
            return false;
        } catch (Exception $e) {
            return false;
        }
    }
    
    /**
     * Verify Stripe session status on return
     */
    public function verify_stripe_session($order, $session_id) {
        try {
            $secret_key = TicketPayGo_Database::get_setting('stripe_secret_key');
            if (empty($secret_key)) {
                return false;
            }
            
            require_once TICKETPAYGO_PLUGIN_PATH . 'vendor/stripe/stripe-php/init.php';
            \Stripe\Stripe::setApiKey($secret_key);
            
            $session = \Stripe\Checkout\Session::retrieve($session_id);
            
            if ($session->payment_status === 'paid') {
                $order->update_payment_status('completed', $session_id, array(
                    'stripe_status' => 'paid',
                    'verified_at' => current_time('mysql')
                ));
                return true;
            }
            
            return false;
        } catch (Exception $e) {
            return false;
        }
    }
    
    /**
     * Handle Mollie webhook
     */
    public function handle_mollie_webhook() {
        // Webhooks from payment providers don't use nonces - they use signature verification
        // This is a public endpoint that receives POST data from Mollie servers
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Webhook endpoint uses signature verification, not nonces
        $payment_id = isset($_POST['id']) ? sanitize_text_field(wp_unslash($_POST['id'])) : '';
        
        if (empty($payment_id)) {
            return;
        }
        
        try {
            $api_key = TicketPayGo_Database::get_setting('mollie_api_key');
            
            require_once TICKETPAYGO_PLUGIN_PATH . 'vendor/mollie/mollie-api-php/src/MollieApiClient.php';
            
            $mollie = new \Mollie\Api\MollieApiClient();
            $mollie->setApiKey($api_key);
            
            $payment = $mollie->payments->get($payment_id);
            $order_id = $payment->metadata->order_id;
            
            $order = new TicketPayGo_Order($order_id);
            
            if ($payment->isPaid()) {
                $order->update_payment_status('completed', $payment_id);
            } elseif ($payment->isFailed()) {
                $order->update_payment_status('failed', $payment_id);
            } elseif ($payment->isCanceled()) {
                $order->update_payment_status('cancelled', $payment_id);
            }
            
        } catch (Exception $e) {
            // Silently fail
        }
    }
    
    /**
     * Handle PayPal webhook
     */
    public function handle_paypal_webhook() {
        $input = file_get_contents('php://input');
        $data = json_decode($input, true);
        
        if ($data['event_type'] === 'CHECKOUT.ORDER.APPROVED') {
            $order_id = $data['resource']['purchase_units'][0]['reference_id'];
            // Find order by order number and update status
            global $wpdb;
            $orders_table = $wpdb->prefix . 'ticketpaygo_lite_orders';
            
            // Table name is safe: constructed from $wpdb->prefix + hardcoded 'ticketpaygo_lite_orders'
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table, table name is safely constructed from prefix + hardcoded suffix
            $order_data = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}ticketpaygo_lite_orders WHERE order_number = %s",
                $order_id
            ));
            
            if ($order_data) {
                $order = new TicketPayGo_Order($order_data->id);
                $order->update_payment_status('completed', $data['resource']['id']);
            }
        }
    }
    
    /**
     * Handle Stripe webhook
     */
    public function handle_stripe_webhook() {
        $input = file_get_contents('php://input');
        $event = json_decode($input, true);
        
        if ($event['type'] === 'checkout.session.completed') {
            $session = $event['data']['object'];
            $order_id = $session['metadata']['order_id'];
            
            $order = new TicketPayGo_Order($order_id);
            $order->update_payment_status('completed', $session['id']);
        }
    }
    
    /**
     * Process refund
     */
    public function process_refund($order) {
        $payment_method = $order->get('payment_method');
        $payment_id = $order->get('payment_id');
        
        switch ($payment_method) {
            case 'mollie':
                return $this->process_mollie_refund($order, $payment_id);
            case 'paypal':
                return $this->process_paypal_refund($order, $payment_id);
            case 'stripe':
                return $this->process_stripe_refund($order, $payment_id);
            default:
                return array('success' => false, 'message' => 'Refund not supported for this payment method');
        }
    }
    
    /**
     * Process Mollie refund
     */
    private function process_mollie_refund($order, $payment_id) {
        try {
            $api_key = TicketPayGo_Database::get_setting('mollie_api_key');
            
            require_once TICKETPAYGO_PLUGIN_PATH . 'vendor/mollie/mollie-api-php/src/MollieApiClient.php';
            
            $mollie = new \Mollie\Api\MollieApiClient();
            $mollie->setApiKey($api_key);
            
            $refund = $mollie->payments->get($payment_id)->refunds()->create([
                'amount' => [
                    'currency' => $order->get('currency'),
                    'value' => number_format($order->get('total_amount'), 2, '.', '')
                ]
            ]);
            
            return array('success' => true, 'refund_id' => $refund->id);
            
        } catch (Exception $e) {
            return array('success' => false, 'message' => 'Mollie refund failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Process PayPal refund
     */
    private function process_paypal_refund($order, $payment_id) {
        // PayPal refund implementation
        return array('success' => false, 'message' => 'PayPal refund not implemented yet');
    }
    
    /**
     * Process Stripe refund
     */
    private function process_stripe_refund($order, $payment_id) {
        try {
            $secret_key = TicketPayGo_Database::get_setting('stripe_secret_key');
            
            require_once TICKETPAYGO_PLUGIN_PATH . 'vendor/stripe/stripe-php/init.php';
            
            \Stripe\Stripe::setApiKey($secret_key);
            
            // Get the payment intent from the session
            $session = \Stripe\Checkout\Session::retrieve($payment_id);
            $payment_intent = $session->payment_intent;
            
            $refund = \Stripe\Refund::create([
                'payment_intent' => $payment_intent,
                'amount' => intval($order->get('total_amount') * 100), // Convert to cents
            ]);
            
            return array('success' => true, 'refund_id' => $refund->id);
            
        } catch (Exception $e) {
            return array('success' => false, 'message' => 'Stripe refund failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Log payment transaction
     */
    public function log_transaction($order_id, $payment_method, $transaction_id, $amount, $currency, $status, $response = '') {
        global $wpdb;
        
        $payment_logs_table = $wpdb->prefix . 'ticketpaygo_payment_logs';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        $wpdb->insert(
            $payment_logs_table,
            array(
                'order_id' => intval($order_id),
                'payment_method' => sanitize_text_field($payment_method),
                'transaction_id' => sanitize_text_field($transaction_id),
                'amount' => floatval($amount),
                'currency' => sanitize_text_field($currency),
                'status' => sanitize_text_field($status),
                'gateway_response' => sanitize_textarea_field($response)
            )
        );
    }
    
    /**
     * Get supported payment methods
     * Payment methods are determined by whether PayPal is configured
     * Free tickets option is only shown if PayPal is not enabled
     */
    public function get_supported_methods() {
        $available_methods = array();
        
        // Check if PayPal is enabled and configured
        $paypal_enabled = TicketPayGo_Database::get_setting('paypal_enabled', '0') === '1';
        $paypal_configured = !empty(TicketPayGo_Database::get_setting('paypal_client_id'));
        
        if ($paypal_enabled && $paypal_configured) {
            // PayPal is enabled - this is the only payment option for paid tickets
            $available_methods['paypal'] = array(
                'name' => 'PayPal',
                'description' => 'Pay with PayPal or Credit Card'
            );
        } else {
            // PayPal not configured - only free tickets available
            $available_methods['free'] = array(
                'name' => 'Free Tickets',
                'description' => 'Get your tickets instantly via email'
            );
        }
        
        return $available_methods;
    }
}
