<?php
/**
 * Order management class
 * 
 * phpcs:disable PluginCheck.Security.DirectDB.UnescapedDBParameter
 */

if (!defined('ABSPATH')) {
    exit;
}

class TicketPayGo_Order {
    
    private $id;
    private $data;
    
    /**
     * Constructor
     */
    public function __construct($id = 0) {
        if ($id > 0) {
            $this->id = $id;
            $this->load_order();
        }
    }
    
    /**
     * Load order data
     */
    private function load_order() {
        global $wpdb;
        
        // Check cache first
        $cache_key = 'tpg_order_' . $this->id;
        $cached = wp_cache_get($cache_key, 'ticketpaygo');
        if (false !== $cached) {
            $this->data = $cached;
            return;
        }
        
        $orders_table = $wpdb->prefix . 'ticketpaygo_lite_orders';
        
        // Table names are safe: constructed from $wpdb->prefix + hardcoded suffixes
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom tables, table names are safely constructed from prefix + hardcoded suffix
        $this->data = $wpdb->get_row($wpdb->prepare(
            "SELECT o.*, e.title as event_title, e.start_date, e.end_date, e.location
             FROM {$orders_table} o
             LEFT JOIN {$wpdb->prefix}ticketpaygo_lite_events e ON o.event_id = e.id
             WHERE o.id = %d",
            $this->id
        ), ARRAY_A);
        
        if ($this->data) {
            // Decode JSON fields
            $this->data['payment_data'] = json_decode($this->data['payment_data'], true) ?: array();
            
            // Cache order data for 5 minutes
            wp_cache_set($cache_key, $this->data, 'ticketpaygo', 300);
        }
    }
    
    /**
     * Check if order exists
     */
    public function exists() {
        return !empty($this->data);
    }
    
    /**
     * Get order data
     */
    public function get_data() {
        return $this->data;
    }
    
    /**
     * Get specific field
     */
    public function get($field) {
        return isset($this->data[$field]) ? $this->data[$field] : null;
    }
    
    /**
     * Create new order
     */
    public function create_order($event_id, $quantity, $customer_data, $payment_method) {
        global $wpdb;
        
        // Validate event
        $event = new TicketPayGo_Event($event_id);
        if (!$event->exists() || !$event->is_active()) {
            return array('success' => false, 'message' => 'Event not available');
        }
        
        // Check ticket availability
        $remaining_tickets = $event->get_remaining_tickets();
        if ($remaining_tickets !== -1 && $remaining_tickets < $quantity) {
            return array('success' => false, 'message' => 'Not enough tickets available');
        }
        
        $orders_table = $wpdb->prefix . 'ticketpaygo_lite_orders';
        
        // Generate unique order number
        $order_number = $this->generate_order_number();
        
        // Use total amount from frontend if provided, otherwise calculate from event price
        if (is_array($customer_data) && isset($customer_data['total_amount']) && $customer_data['total_amount'] > 0) {
            $total_amount = floatval($customer_data['total_amount']);
        } else {
            $price = floatval($event->get('price'));
            $total_amount = $price * $quantity;
        }
        
        // Parse customer data (accept both array and JSON string)
        if (is_array($customer_data)) {
            $customer_info = $customer_data;
        } else {
            $customer_info = json_decode($customer_data, true);
        }
        
        if (!$customer_info || empty($customer_info['name']) || empty($customer_info['email'])) {
            return array('success' => false, 'message' => 'Invalid customer data');
        }
        
        // Build initial payment_data including ticket_type_name
        $initial_payment_data = array();
        if (!empty($customer_info['ticket_type_name'])) {
            $initial_payment_data['ticket_type_name'] = sanitize_text_field($customer_info['ticket_type_name']);
        }

        // Prepare order data
        $order_data = array(
            'order_number' => $order_number,
            'event_id' => intval($event_id),
            'customer_name' => sanitize_text_field($customer_info['name']),
            'customer_email' => sanitize_email($customer_info['email']),
            'customer_phone' => sanitize_text_field($customer_info['phone'] ?? ''),
            'customer_address' => sanitize_textarea_field($customer_info['address'] ?? ''),
            'customer_city' => sanitize_text_field($customer_info['city'] ?? ''),
            'customer_country' => sanitize_text_field($customer_info['country'] ?? ''),
            'quantity' => intval($quantity),
            'total_amount' => $total_amount,
            'currency' => $event->get('currency'),
            'payment_method' => sanitize_text_field($payment_method),
            'payment_status' => 'pending',
            'status' => 'pending',
            'payment_data' => json_encode($initial_payment_data)
        );
        
        $result = $wpdb->insert($orders_table, $order_data); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        
        if ($result !== false) {
            $this->id = $wpdb->insert_id;
            // Invalidate event tickets sold cache
            wp_cache_delete('tpg_tickets_sold_' . $event_id, 'ticketpaygo');
            $this->load_order();
            
            // Process payment
            $payment_result = $this->process_payment($payment_method);
            
            if ($payment_result['success']) {
                return array(
                    'success' => true,
                    'order_id' => $this->id,
                    'order_number' => $order_number,
                    'payment_url' => $payment_result['payment_url'] ?? '',
                    'message' => 'Order created successfully'
                );
            } else {
                // Delete order if payment failed
                $this->delete();
                return $payment_result;
            }
        }
        
        return array('success' => false, 'message' => 'Failed to create order');
    }
    
    /**
     * Generate unique order number
     */
    private function generate_order_number() {
        $prefix = 'TPG';
        $date = gmdate('Ymd');
        $random = wp_rand(1000, 9999);
        return $prefix . $date . $random;
    }
    
    /**
     * Process payment
     */
    private function process_payment($payment_method) {
        $payment_processor = new TicketPayGo_Payment();
        
        switch ($payment_method) {
            case 'free':
            case 'manual':
                // For Lite version: free tickets or manual payment confirmation
                return $payment_processor->process_free_payment($this);
            case 'mollie':
                return $payment_processor->process_mollie_payment($this);
            case 'paypal':
                return $payment_processor->process_paypal_payment($this);
            case 'stripe':
                return $payment_processor->process_stripe_payment($this);
            default:
                // Default to free payment for Lite version if no valid payment method
                return $payment_processor->process_free_payment($this);
        }
    }
    
    /**
     * Update payment status
     */
    public function update_payment_status($status, $payment_id = '', $payment_data = array()) {
        global $wpdb;
        
        if (!$this->exists()) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Logging payment errors for debugging
            error_log('TicketPayGo: update_payment_status failed - order does not exist');
            return false;
        }
        
        $orders_table = $wpdb->prefix . 'ticketpaygo_lite_orders';
        
        // Merge with existing payment data
        $existing_payment_data = $this->get('payment_data') ?: array();
        if (is_array($existing_payment_data) && is_array($payment_data)) {
            $payment_data = array_merge($existing_payment_data, $payment_data);
        }
        
        $update_data = array(
            'payment_status' => sanitize_text_field($status),
            'payment_id' => sanitize_text_field($payment_id),
            'payment_data' => json_encode($payment_data)
        );
        
        $formats = array('%s', '%s', '%s');
        
        // Update order status based on payment status
        if ($status === 'completed') {
            $update_data['status'] = 'completed';
            $formats[] = '%s';
        } elseif ($status === 'failed') {
            $update_data['status'] = 'failed';
            $formats[] = '%s';
        }
        
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Logging payment status updates for debugging
        error_log('TicketPayGo: Updating order ' . $this->id . ' payment status to: ' . $status);
        
        $result = $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            $orders_table,
            $update_data,
            array('id' => $this->id),
            $formats,
            array('%d')
        );
        
        if ($result === false) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Logging database errors for debugging
            error_log('TicketPayGo: update_payment_status DB error - ' . $wpdb->last_error);
            return false;
        }
        
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Logging order updates for debugging
        error_log('TicketPayGo: Order ' . $this->id . ' updated successfully, rows affected: ' . $result);
        
        // Invalidate caches
        wp_cache_delete('tpg_order_' . $this->id, 'ticketpaygo');
        $event_id = $this->get('event_id');
        if ($event_id) {
            wp_cache_delete('tpg_tickets_sold_' . $event_id, 'ticketpaygo');
        }
        
        $this->load_order();
        
        // If payment completed, generate tickets and send email
        if ($status === 'completed') {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Logging ticket generation for debugging
            error_log('TicketPayGo: Generating tickets and sending email for order ' . $this->id);
            $this->generate_tickets();
            $this->send_confirmation_email();
            $this->record_analytics();
        }
        
        return true;
    }
    
    /**
     * Generate tickets for completed order
     */
    private function generate_tickets() {
        $quantity = intval($this->get('quantity'));
        $event_id = intval($this->get('event_id'));

        // Get ticket type name from payment_data, fall back to event ticket_name
        $payment_data = $this->get('payment_data') ?: array();
        $ticket_type  = !empty($payment_data['ticket_type_name']) ? $payment_data['ticket_type_name'] : '';
        if (empty($ticket_type)) {
            $event       = new TicketPayGo_Event($event_id);
            $ticket_name = $event->get('ticket_name');
            if (!empty($ticket_name)) {
                $ticket_type = $ticket_name;
            } else {
                $ticket_types_raw = $event->get('ticket_types');
                $ticket_types_arr = is_string($ticket_types_raw) ? json_decode($ticket_types_raw, true) : $ticket_types_raw;
                if (!empty($ticket_types_arr) && is_array($ticket_types_arr)) {
                    $ticket_type = $ticket_types_arr[0]['name'] ?? __('General Admission', 'ticketpaygo-lite');
                } else {
                    $ticket_type = __('General Admission', 'ticketpaygo-lite');
                }
            }
        }

        for ($i = 0; $i < $quantity; $i++) {
            $ticket = new TicketPayGo_Ticket();
            $ticket_data = array(
                'ticket_type' => $ticket_type,
                'holder_name' => $this->get('customer_name'),
                'holder_email' => $this->get('customer_email')
            );
            $ticket->create($this->id, $event_id, $ticket_data);
        }
    }
    
    /**
     * Send confirmation email
     */
    private function send_confirmation_email() {
        $email_handler = new TicketPayGo_Email();
        $email_handler->send_order_confirmation($this);
    }
    
    /**
     * Record analytics
     */
    private function record_analytics() {
        $event = new TicketPayGo_Event($this->get('event_id'));
        
        // Record sale
        $event->record_analytics('ticket_sale', $this->get('quantity'), json_encode(array(
            'order_id' => $this->id,
            'amount' => $this->get('total_amount'),
            'payment_method' => $this->get('payment_method')
        )));
        
        // Record customer city
        if ($this->get('customer_city')) {
            $event->record_analytics('customer_city', 1, $this->get('customer_city'));
        }
    }
    
    /**
     * Cancel order
     */
    public function cancel($reason = '') {
        global $wpdb;
        
        if (!$this->exists()) {
            return false;
        }
        
        $orders_table = $wpdb->prefix . 'ticketpaygo_lite_orders';
        
        $result = $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            $orders_table,
            array(
                'status' => 'cancelled',
                'notes' => sanitize_textarea_field($reason)
            ),
            array('id' => $this->id),
            array('%s', '%s'),
            array('%d')
        );
        
        if ($result !== false) {
            // Cancel associated tickets
            $this->cancel_tickets();
            
            // Process refund if payment was completed
            if ($this->get('payment_status') === 'completed') {
                $this->process_refund();
            }
            
            // Invalidate caches
            wp_cache_delete('tpg_order_' . $this->id, 'ticketpaygo');
            $event_id = $this->get('event_id');
            if ($event_id) {
                wp_cache_delete('tpg_tickets_sold_' . $event_id, 'ticketpaygo');
            }
            
            $this->load_order();
            return true;
        }
        
        return false;
    }
    
    /**
     * Cancel associated tickets
     */
    private function cancel_tickets() {
        global $wpdb;
        
        $tickets_table = $wpdb->prefix . 'ticketpaygo_lite_tickets';
        
        $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            $tickets_table,
            array('status' => 'cancelled'),
            array('order_id' => $this->id),
            array('%s'),
            array('%d')
        );
    }
    
    /**
     * Process refund
     */
    private function process_refund() {
        $payment_processor = new TicketPayGo_Payment();
        return $payment_processor->process_refund($this);
    }
    
    /**
     * Delete order
     */
    public function delete() {
        global $wpdb;
        
        if (!$this->exists()) {
            return false;
        }
        
        // Delete associated tickets first
        $tickets_table = $wpdb->prefix . 'ticketpaygo_lite_tickets';
        $wpdb->delete($tickets_table, array('order_id' => $this->id), array('%d')); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        
        // Delete order
        $orders_table = $wpdb->prefix . 'ticketpaygo_lite_orders';
        $result = $wpdb->delete($orders_table, array('id' => $this->id), array('%d')); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        
        if ($result !== false) {
            // Invalidate caches
            wp_cache_delete('tpg_order_' . $this->id, 'ticketpaygo');
            $event_id = $this->get('event_id');
            if ($event_id) {
                wp_cache_delete('tpg_tickets_sold_' . $event_id, 'ticketpaygo');
            }
            
            $this->data = null;
            $this->id = 0;
            return true;
        }
        
        return false;
    }
    
    /**
     * Get order tickets
     */
    public function get_tickets() {
        return TicketPayGo_Ticket::get_tickets_by_order($this->id);
    }
    
    /**
     * Get formatted total
     */
    public function get_formatted_total() {
        $total = floatval($this->get('total_amount'));
        $currency = $this->get('currency') ?: 'EUR';
        $symbol = TicketPayGo_Database::get_setting('currency_symbol', '€');
        $position = TicketPayGo_Database::get_setting('currency_position', 'before');
        
        $formatted_total = number_format($total, 2);
        
        if ($position === 'before') {
            return $symbol . $formatted_total;
        } else {
            return $formatted_total . $symbol;
        }
    }
    
    /**
     * Get all orders
     */
    public static function get_all_orders($args = array()) {
        global $wpdb;
        
        $orders_table = $wpdb->prefix . 'ticketpaygo_lite_orders';
        
        $defaults = array(
            'status' => '',
            'payment_status' => '',
            'limit' => 20,
            'offset' => 0,
            'orderby' => 'created_at',
            'order' => 'DESC'
        );
        
        $args = wp_parse_args($args, $defaults);
        
        $where = array();
        $where_values = array();
        
        if (!empty($args['status'])) {
            $where[] = 'o.status = %s';
            $where_values[] = $args['status'];
        }
        
        if (!empty($args['payment_status'])) {
            $where[] = 'o.payment_status = %s';
            $where_values[] = $args['payment_status'];
        }
        
        if (!empty($args['event_id'])) {
            $where[] = 'o.event_id = %d';
            $where_values[] = intval($args['event_id']);
        }
        
        if (!empty($args['search'])) {
            $where[] = '(o.order_number LIKE %s OR o.customer_name LIKE %s OR o.customer_email LIKE %s)';
            $search_term = '%' . $wpdb->esc_like($args['search']) . '%';
            $where_values[] = $search_term;
            $where_values[] = $search_term;
            $where_values[] = $search_term;
        }
        
        $where_clause = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';
        
        $orderby = sanitize_sql_orderby('o.' . $args['orderby'] . ' ' . $args['order']);
        $limit = intval($args['limit']);
        $offset = intval($args['offset']);
        
        $query = "SELECT o.*, e.title as event_title, e.start_date
                  FROM $orders_table o
                  LEFT JOIN {$wpdb->prefix}ticketpaygo_lite_events e ON o.event_id = e.id
                  $where_clause 
                  ORDER BY $orderby 
                  LIMIT $limit OFFSET $offset";
        
        if (!empty($where_values)) {
            $query = $wpdb->prepare($query, $where_values); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query is dynamically built with proper placeholders
        }
        
        $orders = $wpdb->get_results($query, ARRAY_A); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared,WordPress.DB.DirectDatabaseQuery
        
        // Decode JSON fields for each order
        foreach ($orders as &$order) {
            $order['payment_data'] = json_decode($order['payment_data'], true) ?: array();
        }
        
        return $orders;
    }
    
    /**
     * Get orders count
     */
    public static function get_orders_count($status = '') {
        global $wpdb;
        
        // Check cache first
        $cache_key = 'tpg_orders_count_' . ($status ? $status : 'all');
        $cached = wp_cache_get($cache_key, 'ticketpaygo');
        if (false !== $cached) {
            return intval($cached);
        }
        
        $orders_table = $wpdb->prefix . 'ticketpaygo_lite_orders';
        
        if ($status) {
            // Table name is safe: constructed from $wpdb->prefix + hardcoded 'ticketpaygo_lite_orders'
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table, table name is safely constructed from prefix + hardcoded suffix
            $count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$orders_table} WHERE status = %s",
                $status
            ));
        } else {
            // Table name is safe: constructed from $wpdb->prefix + hardcoded 'ticketpaygo_lite_orders'
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table, table name is safely constructed from prefix + hardcoded suffix
            $count = $wpdb->get_var("SELECT COUNT(*) FROM {$orders_table}");
        }
        
        $count = intval($count);
        // Cache for 2 minutes
        wp_cache_set($cache_key, $count, 'ticketpaygo', 120);
        
        return $count;
    }
    
    /**
     * Get revenue statistics
     */
    public static function get_revenue_stats($event_id = null, $period = 'all') {
        global $wpdb;
        
        $orders_table = $wpdb->prefix . 'ticketpaygo_lite_orders';
        
        $where = array("status = 'completed'");
        $where_values = array();
        
        if ($event_id) {
            $where[] = 'event_id = %d';
            $where_values[] = $event_id;
        }
        
        // Add date filter based on period
        switch ($period) {
            case 'today':
                $where[] = 'DATE(created_at) = CURDATE()';
                break;
            case 'week':
                $where[] = 'created_at >= DATE_SUB(NOW(), INTERVAL 1 WEEK)';
                break;
            case 'month':
                $where[] = 'created_at >= DATE_SUB(NOW(), INTERVAL 1 MONTH)';
                break;
            case 'year':
                $where[] = 'created_at >= DATE_SUB(NOW(), INTERVAL 1 YEAR)';
                break;
        }
        
        $where_clause = 'WHERE ' . implode(' AND ', $where);
        
        $query = "SELECT 
                    COUNT(*) as total_orders,
                    SUM(total_amount) as total_revenue,
                    SUM(quantity) as total_tickets,
                    AVG(total_amount) as average_order_value
                  FROM $orders_table $where_clause";
        
        if (!empty($where_values)) {
            $query = $wpdb->prepare($query, $where_values); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query is dynamically built with proper placeholders
        }
        
        return $wpdb->get_row($query, ARRAY_A); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared,WordPress.DB.DirectDatabaseQuery
    }
}
