<?php
/**
 * Frontend functionality for TicketPayGo Lite
 * 
 * phpcs:disable PluginCheck.Security.DirectDB.UnescapedDBParameter
 */

if (!defined('ABSPATH')) {
    exit;
}

class TicketPayGo_Frontend {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('init', array($this, 'init'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('template_redirect', array($this, 'handle_payment_returns'));
        add_action('wp_footer', array($this, 'display_success_message'));
        add_action('wp_ajax_ticketpaygo_purchase_ticket', array($this, 'handle_ticket_purchase'));
        add_action('wp_ajax_nopriv_ticketpaygo_purchase_ticket', array($this, 'handle_ticket_purchase'));
        add_action('wp_ajax_ticketpaygo_verify_ticket', array($this, 'handle_ticket_verification'));
        add_action('wp_ajax_nopriv_ticketpaygo_verify_ticket', array($this, 'handle_ticket_verification'));
        
        // Dynamic nonce refresh endpoint - prevents cache issues
        add_action('wp_ajax_ticketpaygo_refresh_nonce', array($this, 'handle_refresh_nonce'));
        add_action('wp_ajax_nopriv_ticketpaygo_refresh_nonce', array($this, 'handle_refresh_nonce'));
        
        // Prevent caching of checkout/event pages
        add_action('template_redirect', array($this, 'add_cache_headers'), 1);
    }
    
    /**
     * Initialize frontend
     */
    public function init() {
        // Add rewrite rules for ticket pages
        add_rewrite_rule('^event/([0-9]+)/?', 'index.php?ticketpaygo_event=$matches[1]', 'top');
        add_rewrite_rule('^ticket/([0-9]+)/?', 'index.php?ticketpaygo_ticket=$matches[1]', 'top');
        add_rewrite_rule('^verify-ticket/?', 'index.php?ticketpaygo_verify=1', 'top');
        
        // Add query vars
        add_filter('query_vars', array($this, 'add_query_vars'));
        
        // Handle custom pages
        add_action('template_include', array($this, 'template_include'));
    }
    
    /**
     * Add query variables
     */
    public function add_query_vars($vars) {
        $vars[] = 'ticketpaygo_event';
        $vars[] = 'ticketpaygo_ticket';
        $vars[] = 'ticketpaygo_verify';
        return $vars;
    }
    
    /**
     * Include custom templates
     */
    public function template_include($template) {
        if (get_query_var('ticketpaygo_event')) {
            return $this->load_template('single-event.php');
        }
        
        if (get_query_var('ticketpaygo_ticket')) {
            return $this->load_template('single-ticket.php');
        }
        
        if (get_query_var('ticketpaygo_verify')) {
            return $this->load_template('verify-ticket.php');
        }
        
        return $template;
    }
    
    /**
     * Load plugin template
     */
    private function load_template($template_name) {
        $template_path = TICKETPAYGO_PLUGIN_PATH . 'templates/' . $template_name;
        
        if (file_exists($template_path)) {
            return $template_path;
        }
        
        // Fallback to default template
        return get_index_template();
    }
    
    /**
     * Enqueue frontend scripts and styles
     */
    public function enqueue_scripts() {
        wp_enqueue_script('jquery');
        
        // Generate a unique version number for strong cache busting
        $version = TICKETPAYGO_VERSION . '.' . time() . '.' . wp_rand(1000, 9999);
        
        // Enqueue Leaflet for maps (bundled locally for WordPress.org compliance)
        wp_enqueue_style('leaflet-css', TICKETPAYGO_PLUGIN_URL . 'assets/css/leaflet.css', array(), '1.9.4');
        wp_enqueue_script('leaflet-js', TICKETPAYGO_PLUGIN_URL . 'assets/js/leaflet.js', array(), '1.9.4', false);
        
        // Enqueue frontend CSS with high priority
        wp_enqueue_style(
            'ticketpaygo-frontend',
            TICKETPAYGO_PLUGIN_URL . 'assets/css/frontend.css',
            array('leaflet-css'),
            $version
        );
        
        // Enqueue checkout modal CSS
        wp_enqueue_style(
            'ticketpaygo-checkout-modal',
            TICKETPAYGO_PLUGIN_URL . 'assets/css/checkout-modal.css',
            array(),
            $version
        );
        
        // Enqueue events list CSS
        wp_enqueue_style(
            'ticketpaygo-events-list',
            TICKETPAYGO_PLUGIN_URL . 'assets/css/events-list.css',
            array(),
            $version
        );
        
        // Custom CSS removed in Lite version - available in Full
        
        // Add direct CSS fixes for card display
        $emergency_css = "
            .ticketpaygo-event-card, .ticketpaygo-event-card-list {
                display: block !important;
                visibility: visible !important;
            }
            .event-overlay {
                display: block !important;
                visibility: visible !important;
                opacity: 1 !important;
            }
            .event-title, .event-quick-info, .info-item, .price-value {
                display: block !important;
                visibility: visible !important;
                opacity: 1 !important;
                color: #ffffff !important;
            }
            .event-card-compact {
                cursor: pointer !important;
            }
        ";
        wp_add_inline_style('ticketpaygo-frontend', $emergency_css);
        
        wp_enqueue_script(
            'ticketpaygo-frontend',
            TICKETPAYGO_PLUGIN_URL . 'assets/js/frontend.js',
            array('jquery'),
            $version,
            false  // Load in header instead of footer for immediate availability
        );
        
        // Enqueue events list JS
        wp_enqueue_script(
            'ticketpaygo-events-list',
            TICKETPAYGO_PLUGIN_URL . 'assets/js/events-list.js',
            array('jquery'),
            $version,
            true
        );
        
        // Localize script for AJAX and translations
        wp_localize_script('ticketpaygo-frontend', 'ticketpaygo_frontend', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('ticketpaygo_frontend_nonce'),
            'currency_symbol' => $this->get_setting('currency_symbol', '€'),
            'currency_position' => $this->get_setting('currency_position', 'before'),
            'ticket_url' => home_url('/ticket/'),
            'strings' => array(
                /* translators: %d: number of tickets */
                'book_tickets' => __('Book %d Tickets', 'ticketpaygo-lite'),
                'select_tickets' => __('Select Tickets', 'ticketpaygo-lite'),
                'invalid_email' => __('Please enter a valid email address', 'ticketpaygo-lite'),
                'field_required' => __('This field is required', 'ticketpaygo-lite'),
                'select_quantity' => __('Please select number of tickets', 'ticketpaygo-lite'),
                'select_payment' => __('Please select a payment method', 'ticketpaygo-lite'),
                'booking_failed' => __('Booking failed. Please try again.', 'ticketpaygo-lite'),
                'verifying' => __('Verifying...', 'ticketpaygo-lite'),
                'verify' => __('Verify', 'ticketpaygo-lite'),
                'ticket_valid' => __('Ticket is valid', 'ticketpaygo-lite'),
                'ticket_invalid' => __('Ticket is invalid', 'ticketpaygo-lite'),
                'ticket_already_used' => __('Ticket has already been used', 'ticketpaygo-lite'),
                'ticket_expired' => __('Ticket has expired', 'ticketpaygo-lite'),
                'verification_failed' => __('Verification failed', 'ticketpaygo-lite'),
                'event' => __('Event', 'ticketpaygo-lite'),
                'date' => __('Date', 'ticketpaygo-lite'),
                'holder' => __('Holder', 'ticketpaygo-lite'),
                'ticket_number' => __('Ticket Number', 'ticketpaygo-lite'),
                'scan_qr' => __('Scan QR Code', 'ticketpaygo-lite'),
                'stop_scan' => __('Stop Scanning', 'ticketpaygo-lite'),
                'scanner_failed' => __('QR Scanner failed to start', 'ticketpaygo-lite'),
                'filter_failed' => __('Failed to filter events', 'ticketpaygo-lite'),
                'loading' => __('Loading...', 'ticketpaygo-lite'),
                'load_more' => __('Load More Events', 'ticketpaygo-lite'),
                'load_failed' => __('Failed to load events', 'ticketpaygo-lite'),
                'payment_success' => __('Payment successful! Redirecting to your tickets...', 'ticketpaygo-lite'),
                'payment_cancelled' => __('Payment was cancelled', 'ticketpaygo-lite'),
                'payment_failed' => __('Payment failed. Please try again.', 'ticketpaygo-lite'),
                'url_copied' => __('Event URL copied to clipboard', 'ticketpaygo-lite'),
                'confirm_purchase' => __('Are you sure you want to purchase these tickets?', 'ticketpaygo-lite'),
                'sold_out' => __('Sorry, this event is sold out.', 'ticketpaygo-lite'),
                'invalid_quantity' => __('Please select a valid quantity.', 'ticketpaygo-lite'),
                'fill_required_fields' => __('Please fill in all required fields.', 'ticketpaygo-lite')
            )
        ));
    }
    
    /**
     * Get setting value
     */
    private function get_setting($key, $default = '') {
        $cache_key = 'tpgl_setting_' . $key;
        $cached = wp_cache_get($cache_key, 'ticketpaygo_lite');
        
        if (false !== $cached) {
            return $cached;
        }
        
        global $wpdb;
        $settings_table = $wpdb->prefix . 'ticketpaygo_lite_settings';
        
        // Check if settings table exists
        $table_exists = wp_cache_get('tpgl_settings_table_exists', 'ticketpaygo_lite');
        if (false === $table_exists) {
            $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $settings_table)) == $settings_table ? 'yes' : 'no'; // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            wp_cache_set('tpgl_settings_table_exists', $table_exists, 'ticketpaygo_lite', 3600);
        }
        
        if ($table_exists !== 'yes') {
            return $default;
        }
        
        // Table name is safe: constructed from $wpdb->prefix + hardcoded 'ticketpaygo_lite_settings'
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table, table name is safely constructed from prefix + hardcoded suffix
        $value = $wpdb->get_var($wpdb->prepare(
            "SELECT setting_value FROM {$settings_table} WHERE setting_key = %s",
            $key
        ));
        
        $result = $value !== null ? $value : $default;
        wp_cache_set($cache_key, $result, 'ticketpaygo_lite', 3600);
        
        return $result;
    }
    
    /**
     * Handle payment returns
     */
    public function handle_payment_returns() {
        // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Payment gateway callback, verified by payment processor
        if (!isset($_GET['ticketpaygo_payment'])) {
            return;
        }
        
        $payment_status = sanitize_text_field(wp_unslash($_GET['ticketpaygo_payment']));
        $order_id = isset($_GET['order_id']) ? intval($_GET['order_id']) : 0;
        
        if (!$order_id) {
            return;
        }
        
        $order = new TicketPayGo_Order($order_id);
        if (!$order->exists()) {
            return;
        }
        
        switch ($payment_status) {
            case 'success':
            case 'return':
                $this->handle_payment_success($order);
                break;
            case 'cancel':
                $this->handle_payment_cancel($order);
                break;
            case 'error':
                $this->handle_payment_error($order);
                break;
        }
    }
    
    /**
     * Handle successful payment
     */
    private function handle_payment_success($order) {
        $payment_method = isset($_GET['payment_method']) ? sanitize_text_field(wp_unslash($_GET['payment_method'])) : '';
        $payment_processor = new TicketPayGo_Payment();
        
        // For PayPal, capture the payment if token is present
        if ($payment_method === 'paypal' && isset($_GET['token'])) {
            $capture_result = $payment_processor->capture_paypal_payment($order, sanitize_text_field(wp_unslash($_GET['token'])));
            
            if (!$capture_result['success']) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Logging payment errors for debugging
                error_log('TicketPayGo: PayPal capture failed - ' . ($capture_result['message'] ?? 'Unknown error'));
            }
            
            // Reload order to get updated status
            $order = new TicketPayGo_Order($order->get('id'));
        }
        
        // For Mollie, verify payment status
        if ($payment_method === 'mollie') {
            $payment_data = $order->get('payment_data') ?: array();
            $mollie_payment_id = isset($payment_data['payment_id']) ? $payment_data['payment_id'] : '';
            
            if ($mollie_payment_id && method_exists($payment_processor, 'verify_mollie_payment')) {
                $verify_result = $payment_processor->verify_mollie_payment($order, $mollie_payment_id);
                if ($verify_result) {
                    $order = new TicketPayGo_Order($order->get('id'));
                }
            }
        }
        
        // For Stripe, verify session status
        if (isset($_GET['session_id']) && method_exists($payment_processor, 'verify_stripe_session')) {
            $session_id = sanitize_text_field(wp_unslash($_GET['session_id']));
            $verify_result = $payment_processor->verify_stripe_session($order, $session_id);
            if ($verify_result) {
                $order = new TicketPayGo_Order($order->get('id'));
            }
        }
        
        // If order is still pending (capture/verify didn't update it), manually complete it
        // This handles cases where payment was already captured or for free tickets
        if ($order->get('status') === 'pending' || $order->get('payment_status') !== 'completed') {
            $payment_data = $order->get('payment_data') ?: array();
            $payment_id = isset($payment_data['payment_id']) ? $payment_data['payment_id'] : '';
            
            // Update payment status to completed - this also sends email and generates tickets
            $order->update_payment_status('completed', $payment_id, $payment_data);
            
            // Reload order to get updated data
            $order = new TicketPayGo_Order($order->get('id'));
        }
        
        // Redirect back to the referring page with success parameters
        $redirect_url = add_query_arg(array(
            'ticketpaygo_success' => '1',
            'order_id' => $order->get('id'),
            'order_number' => $order->get('order_number')
        ), home_url('/'));
        
        wp_safe_redirect($redirect_url);
        exit;
    }
    
    /**
     * Handle cancelled payment
     */
    private function handle_payment_cancel($order) {
        // Redirect back to event page with cancel message
        $event_id = $order->get('event_id');
        $redirect_url = add_query_arg(array(
            'ticketpaygo_cancelled' => '1'
        ), home_url('/event/' . $event_id . '/'));
        
        wp_safe_redirect($redirect_url);
        exit;
    }
    
    /**
     * Handle payment error
     */
    private function handle_payment_error($order) {
        // Redirect back to event page with error message
        $event_id = $order->get('event_id');
        $redirect_url = add_query_arg(array(
            'ticketpaygo_error' => '1'
        ), home_url('/event/' . $event_id . '/'));
        
        wp_safe_redirect($redirect_url);
        exit;
    }
    
    /**
     * Handle ticket purchase AJAX
     */
    public function handle_ticket_purchase() {
        check_ajax_referer('ticketpaygo_frontend_nonce', 'nonce');
        
        $event_id = isset($_POST['event_id']) ? intval($_POST['event_id']) : 0;
        $quantity = isset($_POST['quantity']) ? intval($_POST['quantity']) : 0;
        $total_amount = floatval($_POST['total_amount'] ?? 0);
        $payment_method = isset($_POST['payment_method']) ? sanitize_text_field(wp_unslash($_POST['payment_method'])) : '';
        
        // Collect customer data from individual fields
        $customer_data = array(
            'name' => isset($_POST['customer_name']) ? sanitize_text_field(wp_unslash($_POST['customer_name'])) : '',
            'email' => isset($_POST['customer_email']) ? sanitize_email(wp_unslash($_POST['customer_email'])) : '',
            'phone' => isset($_POST['customer_phone']) ? sanitize_text_field(wp_unslash($_POST['customer_phone'])) : '',
            'address' => isset($_POST['customer_address']) ? sanitize_text_field(wp_unslash($_POST['customer_address'])) : '',
            'city' => isset($_POST['customer_city']) ? sanitize_text_field(wp_unslash($_POST['customer_city'])) : '',
            'country' => isset($_POST['customer_country']) ? sanitize_text_field(wp_unslash($_POST['customer_country'])) : '',
            'total_amount' => $total_amount
        );
        
        // Validate input
        if (!$event_id || !$quantity || empty($customer_data['name']) || empty($customer_data['email']) || !$payment_method) {
            wp_send_json_error(__('Missing required data.', 'ticketpaygo-lite'));
        }
        
        // Validate event
        $event = new TicketPayGo_Event($event_id);
        if (!$event->exists() || !$event->is_active()) {
            wp_send_json_error(__('Event not available.', 'ticketpaygo-lite'));
        }
        
        // Check if event has ended
        if ($event->has_ended()) {
            wp_send_json_error(__('Ticket sales have ended for this event.', 'ticketpaygo-lite'));
        }
        
        // Check ticket availability
        $remaining_tickets = $event->get_remaining_tickets();
        if ($remaining_tickets !== -1 && $remaining_tickets < $quantity) {
            wp_send_json_error(__('Not enough tickets available.', 'ticketpaygo-lite'));
        }
        
        // Create order
        $order = new TicketPayGo_Order();
        $result = $order->create_order($event_id, $quantity, $customer_data, $payment_method);
        
        // Normalize the response - frontend expects redirect_url
        if ($result['success'] && isset($result['payment_url'])) {
            $result['redirect_url'] = $result['payment_url'];
        }
        
        wp_send_json($result);
    }
    
    /**
     * Handle ticket verification AJAX
     */
    public function handle_ticket_verification() {
        // Nonce is checked in the calling AJAX handler
        // phpcs:disable WordPress.Security.NonceVerification.Missing
        $ticket_id = isset($_POST['ticket_id']) ? intval($_POST['ticket_id']) : 0;
        $ticket_number = isset($_POST['ticket_number']) ? sanitize_text_field(wp_unslash($_POST['ticket_number'])) : '';
        $hash = isset($_POST['hash']) ? sanitize_text_field(wp_unslash($_POST['hash'])) : '';
        // phpcs:enable WordPress.Security.NonceVerification.Missing
        
        if (!$ticket_id || !$ticket_number || !$hash) {
            wp_send_json_error(__('Invalid verification data.', 'ticketpaygo-lite'));
        }
        
        // Verify hash
        $qr_generator = new TicketPayGo_QR_Generator();
        if (!$qr_generator->verify_hash($ticket_id, $ticket_number, $hash)) {
            wp_send_json_error(__('Invalid verification hash.', 'ticketpaygo-lite'));
        }
        
        // Verify ticket
        $ticket = new TicketPayGo_Ticket($ticket_id);
        $verification_result = $ticket->verify();
        
        if ($verification_result['valid']) {
            wp_send_json_success($verification_result);
        } else {
            wp_send_json_error($verification_result['message']);
        }
    }
    
    /**
     * Get event booking form HTML
     */
    public function get_booking_form($event_id, $layout = 'default') {
        $event = new TicketPayGo_Event($event_id);
        if (!$event->exists()) {
            return '<p>' . __('Event not found.', 'ticketpaygo-lite') . '</p>';
        }
        
        // Check if event is available for booking
        if (!$event->is_active()) {
            return '<p>' . __('This event is not available for booking.', 'ticketpaygo-lite') . '</p>';
        }
        
        if ($event->has_ended()) {
            return '<p>' . __('Ticket sales have ended for this event.', 'ticketpaygo-lite') . '</p>';
        }
        
        if ($event->is_sold_out()) {
            return '<p>' . __('Sorry, this event is sold out.', 'ticketpaygo-lite') . '</p>';
        }
        
        // Get available payment methods
        $payment_processor = new TicketPayGo_Payment();
        $payment_methods = $payment_processor->get_supported_methods();
        
        if (empty($payment_methods)) {
            return '<p>' . __('No payment methods available. Please contact the organizer.', 'ticketpaygo-lite') . '</p>';
        }
        
        ob_start();
        include TICKETPAYGO_PLUGIN_PATH . "templates/booking-form-{$layout}.php";
        return ob_get_clean();
    }
    
    /**
     * Display success message as modal
     */
    public function display_success_message() {
        // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Success message display, no sensitive action
        if (!isset($_GET['ticketpaygo_success'])) {
            return;
        }
        
        $order_id = isset($_GET['order_id']) ? intval($_GET['order_id']) : 0;
        $order_number = isset($_GET['order_number']) ? sanitize_text_field(wp_unslash($_GET['order_number'])) : '';
        // phpcs:enable WordPress.Security.NonceVerification.Recommended
        
        if (!$order_id || !$order_number) {
            return;
        }
        
        $order = new TicketPayGo_Order($order_id);
        if (!$order->exists() || $order->get('order_number') !== $order_number) {
            return;
        }
        
        // Get order details
        $customer_name = $order->get('customer_name');
        $customer_email = $order->get('customer_email');
        $event_title = $order->get('event_title');
        $quantity = $order->get('quantity');
        $total_amount = $order->get('total_amount');
        $currency = $order->get('currency') ?: 'EUR';
        $event_date = $order->get('start_date');
        $location = $order->get('location');
        
        // Format date
        $formatted_date = '';
        if ($event_date) {
            $formatted_date = date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($event_date));
        }
        
        // Currency symbol
        $currency_symbols = array('EUR' => '€', 'USD' => '$', 'GBP' => '£');
        $currency_symbol = isset($currency_symbols[$currency]) ? $currency_symbols[$currency] : $currency . ' ';
        
        ?>
        <div id="ticketpaygo-confirmation-modal" class="tpg-modal-overlay">
            <div class="tpg-modal-content">
                <button type="button" class="tpg-modal-close" onclick="document.getElementById('ticketpaygo-confirmation-modal').style.display='none'; history.replaceState(null, '', window.location.pathname);">&times;</button>
                
                <div class="tpg-modal-header">
                    <div class="tpg-checkmark">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3">
                            <polyline points="20 6 9 17 4 12"></polyline>
                        </svg>
                    </div>
                    <h2><?php esc_html_e('Order Confirmed!', 'ticketpaygo-lite'); ?></h2>
                    <p><?php esc_html_e('Thank you for your purchase', 'ticketpaygo-lite'); ?>, <?php echo esc_html($customer_name); ?>!</p>
                </div>
                
                <div class="tpg-modal-body">
                    <div class="tpg-order-number">
                        <span class="tpg-label"><?php esc_html_e('Order Number', 'ticketpaygo-lite'); ?></span>
                        <strong><?php echo esc_html($order_number); ?></strong>
                    </div>
                    
                    <div class="tpg-order-details">
                        <div class="tpg-event-title"><?php echo esc_html($event_title); ?></div>
                        
                        <?php if ($formatted_date): ?>
                        <div class="tpg-detail-row">
                            <span><?php esc_html_e('Date & Time', 'ticketpaygo-lite'); ?></span>
                            <span><?php echo esc_html($formatted_date); ?></span>
                        </div>
                        <?php endif; ?>
                        
                        <?php if ($location): ?>
                        <div class="tpg-detail-row">
                            <span><?php esc_html_e('Location', 'ticketpaygo-lite'); ?></span>
                            <span><?php echo esc_html($location); ?></span>
                        </div>
                        <?php endif; ?>
                        
                        <div class="tpg-detail-row">
                            <span><?php esc_html_e('Tickets', 'ticketpaygo-lite'); ?></span>
                            <span><?php echo esc_html($quantity); ?> &times; <?php echo esc_html($currency_symbol . number_format($total_amount / max($quantity, 1), 2)); ?></span>
                        </div>
                        
                        <div class="tpg-total-row">
                            <span><?php esc_html_e('Total Paid', 'ticketpaygo-lite'); ?></span>
                            <span><?php echo esc_html($currency_symbol . number_format($total_amount, 2)); ?></span>
                        </div>
                    </div>
                    
                    <div class="tpg-email-notice">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="20" height="20">
                            <path d="M4 4h16c1.1 0 2 .9 2 2v12c0 1.1-.9 2-2 2H4c-1.1 0-2-.9-2-2V6c0-1.1.9-2 2-2z"></path>
                            <polyline points="22,6 12,13 2,6"></polyline>
                        </svg>
                        <?php /* translators: %s: customer email address */ ?>
                                        <p><?php echo sprintf(esc_html__('A confirmation email with your tickets has been sent to %s', 'ticketpaygo-lite'), '<strong>' . esc_html($customer_email) . '</strong>'); ?></p>
                    </div>
                </div>
                
                <div class="tpg-modal-footer">
                    <button type="button" class="tpg-btn-close" onclick="document.getElementById('ticketpaygo-confirmation-modal').style.display='none'; history.replaceState(null, '', window.location.pathname);">
                        <?php esc_html_e('Close', 'ticketpaygo-lite'); ?>
                    </button>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Display error messages
     */
    public function display_error_messages() {
        // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Error message display, no sensitive action
        if (isset($_GET['ticketpaygo_cancelled'])) {
            echo '<div class="ticketpaygo-error-message">';
            echo '<p>' . esc_html__('Payment was cancelled. You can try again if you wish.', 'ticketpaygo-lite') . '</p>';
            echo '</div>';
        }
        
        if (isset($_GET['ticketpaygo_error'])) {
            echo '<div class="ticketpaygo-error-message">';
            echo '<p>' . esc_html__('There was an error processing your payment. Please try again.', 'ticketpaygo-lite') . '</p>';
            echo '</div>';
        }
        // phpcs:enable WordPress.Security.NonceVerification.Recommended
    }
    
    /**
     * Add custom body classes
{{ ... }}
    public function add_body_classes($classes) {
        if (get_query_var('ticketpaygo_event')) {
            $classes[] = 'ticketpaygo-event-page';
        }
        
        if (get_query_var('ticketpaygo_ticket')) {
            $classes[] = 'ticketpaygo-ticket-page';
        }
        
        if (get_query_var('ticketpaygo_verify')) {
            $classes[] = 'ticketpaygo-verify-page';
        }
        
        return $classes;
    }
    
    /**
     * Handle webhook requests from payment processors
     * 
     * Note: Nonce verification is intentionally not used here because webhooks are
     * server-to-server callbacks initiated by external payment processors (PayPal, Stripe, Mollie).
     * These requests do not originate from user browsers and cannot include WordPress nonces.
     * Each payment processor handler implements its own request verification mechanism.
     */
    public function handle_webhooks() {
        // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Server-to-server webhook callback from external payment processor (PayPal/Stripe/Mollie). Cannot use nonces as requests originate from payment processor servers, not user browsers.
        if (!isset($_GET['ticketpaygo_webhook'])) {
            return;
        }
        
        $webhook_type = sanitize_text_field(wp_unslash($_GET['ticketpaygo_webhook']));
        // phpcs:enable WordPress.Security.NonceVerification.Recommended
        $payment_processor = new TicketPayGo_Payment();
        
        switch ($webhook_type) {
            case 'mollie':
                $payment_processor->handle_mollie_webhook();
                break;
            case 'paypal':
                $payment_processor->handle_paypal_webhook();
                break;
            case 'stripe':
                $payment_processor->handle_stripe_webhook();
                break;
        }
        
        exit;
    }
    
    /**
     * Handle nonce refresh AJAX request
     * Returns a fresh nonce to prevent cache-related nonce failures
     */
    public function handle_refresh_nonce() {
        wp_send_json_success(array(
            'nonce' => wp_create_nonce('ticketpaygo_frontend_nonce'),
            'timestamp' => time()
        ));
    }
    
    /**
     * Add cache-control headers to prevent caching of checkout pages
     * This ensures the nonce is always fresh
     */
    public function add_cache_headers() {
        // Only apply to TicketPayGo event pages and checkout-related pages
        if (get_query_var('ticketpaygo_event') || get_query_var('ticketpaygo_ticket') || isset($_GET['ticketpaygo_success'])) {
            nocache_headers();
            header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
            header('Cache-Control: post-check=0, pre-check=0', false);
            header('Pragma: no-cache');
            header('Expires: Wed, 11 Jan 1984 05:00:00 GMT');
        }
    }
}
