<?php
/**
 * Event management class
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Event class for TicketPayGo Lite
 * 
 * phpcs:disable PluginCheck.Security.DirectDB.UnescapedDBParameter
 */

class TicketPayGo_Event {
    
    private $id;
    private $data;
    
    /**
     * Constructor
     */
    public function __construct($id = 0) {
        if ($id > 0) {
            $this->id = $id;
            $this->load_event();
        }
    }
    
    /**
     * Load event data
     */
    private function load_event() {
        global $wpdb;
        
        // Check cache first
        $cache_key = 'tpg_event_' . $this->id;
        $cached = wp_cache_get($cache_key, 'ticketpaygo');
        if (false !== $cached) {
            $this->data = $cached;
            return;
        }
        
        $events_table = $wpdb->prefix . 'ticketpaygo_lite_events';
        
        // Check if events table exists
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table check
        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $events_table)) != $events_table) {
            $this->data = null;
            return;
        }
        
        // Table name is safe: constructed from $wpdb->prefix + hardcoded 'ticketpaygo_lite_events'
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table, table name is safely constructed from prefix + hardcoded suffix
        $this->data = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$events_table} WHERE id = %d",
            $this->id
        ), ARRAY_A);
        
        if ($this->data) {
            // Decode JSON fields
            $this->data['ticket_types'] = json_decode($this->data['ticket_types'], true) ?: array();
            $this->data['recurrence'] = json_decode($this->data['recurrence'], true) ?: array();
            $this->data['custom_fields'] = json_decode($this->data['custom_fields'], true) ?: array();
            $this->data['gallery'] = json_decode($this->data['gallery'], true) ?: array();
            
            // Cast boolean/tinyint fields to integers
            $this->data['show_title_on_card'] = isset($this->data['show_title_on_card']) ? intval($this->data['show_title_on_card']) : 1;
            $this->data['max_tickets'] = isset($this->data['max_tickets']) ? intval($this->data['max_tickets']) : 0;
            
            // Cache event data for 1 hour
            wp_cache_set($cache_key, $this->data, 'ticketpaygo', 3600);
        }
    }
    
    /**
     * Check if event exists
     */
    public function exists() {
        return !empty($this->data);
    }
    
    /**
     * Get event data
     */
    public function get_data() {
        return $this->data;
    }
    
    /**
     * Get specific field
     */
    public function get($field) {
        return isset($this->data[$field]) ? $this->data[$field] : null;
    }
    
    /**
     * Create new event
     */
    public function create($data) {
        global $wpdb;
        
        $events_table = $wpdb->prefix . 'ticketpaygo_lite_events';
        
        // Prepare data
        $event_data = array(
            'title' => sanitize_text_field($data['title']),
            'description' => wp_kses_post($data['description']),
            'start_date' => sanitize_text_field($data['start_date']),
            'end_date' => sanitize_text_field($data['end_date']),
            'location' => sanitize_text_field($data['location']),
            'address' => sanitize_textarea_field($data['address']),
            'latitude' => floatval($data['latitude']),
            'longitude' => floatval($data['longitude']),
            'max_tickets' => intval($data['max_tickets']),
            'price' => floatval($data['price']),
            'currency' => sanitize_text_field($data['currency']),
            'status' => sanitize_text_field($data['status']),
            'featured_image' => sanitize_url($data['featured_image']),
            'image_fit' => sanitize_text_field(isset($data['image_fit']) ? $data['image_fit'] : ''),
            'show_title_on_card' => !empty($data['show_title_on_card']) ? 1 : 0,
            'ticket_name' => sanitize_text_field(isset($data['ticket_name']) ? $data['ticket_name'] : ''),
            'gallery' => json_encode($data['gallery']),
            'organizer_name' => sanitize_text_field($data['organizer_name']),
            'organizer_email' => sanitize_email($data['organizer_email']),
            'organizer_phone' => sanitize_text_field($data['organizer_phone']),
            'ticket_types' => json_encode($data['ticket_types']),
            'recurrence' => json_encode(isset($data['recurrence']) ? $data['recurrence'] : array()),
            'custom_fields' => json_encode($data['custom_fields']),
            'seo_title' => sanitize_text_field($data['seo_title']),
            'seo_description' => sanitize_textarea_field($data['seo_description'])
        );
        
        $result = $wpdb->insert($events_table, $event_data); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        
        if ($result !== false) {
            $this->id = $wpdb->insert_id;
            $this->load_event();
            
            // Record analytics (with error handling)
            try {
                $this->record_analytics('event_created', 1);
            } catch (Exception $e) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Logging analytics errors for debugging
                error_log('TicketPayGo Analytics Error: ' . $e->getMessage());
            }
            
            return $this->id;
        }
        
        // Log the error for debugging
        if ($wpdb->last_error) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Logging database errors for debugging
            error_log('TicketPayGo Event Creation Error: ' . $wpdb->last_error);
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log, WordPress.PHP.DevelopmentFunctions.error_log_print_r -- Logging event data for debugging
            error_log('Event Data: ' . print_r($event_data, true));
        }
        
        return false;
    }
    
    /**
     * Update event
     */
    public function update($data) {
        global $wpdb;
        
        if (!$this->exists()) {
            return false;
        }
        
        $events_table = $wpdb->prefix . 'ticketpaygo_lite_events';
        
        // Prepare data
        $event_data = array(
            'title' => sanitize_text_field($data['title']),
            'description' => wp_kses_post($data['description']),
            'start_date' => sanitize_text_field($data['start_date']),
            'end_date' => sanitize_text_field($data['end_date']),
            'location' => sanitize_text_field($data['location']),
            'address' => sanitize_textarea_field($data['address']),
            'latitude' => floatval($data['latitude']),
            'longitude' => floatval($data['longitude']),
            'max_tickets' => intval($data['max_tickets']),
            'price' => floatval($data['price']),
            'currency' => sanitize_text_field($data['currency']),
            'status' => sanitize_text_field($data['status']),
            'featured_image' => sanitize_url($data['featured_image']),
            'image_fit' => sanitize_text_field(isset($data['image_fit']) ? $data['image_fit'] : ''),
            'show_title_on_card' => !empty($data['show_title_on_card']) ? 1 : 0,
            'ticket_name' => sanitize_text_field(isset($data['ticket_name']) ? $data['ticket_name'] : ''),
            'gallery' => json_encode($data['gallery']),
            'organizer_name' => sanitize_text_field($data['organizer_name']),
            'organizer_email' => sanitize_email($data['organizer_email']),
            'organizer_phone' => sanitize_text_field($data['organizer_phone']),
            'ticket_types' => json_encode($data['ticket_types']),
            'recurrence' => json_encode(isset($data['recurrence']) ? $data['recurrence'] : array()),
            'custom_fields' => json_encode($data['custom_fields']),
            'seo_title' => sanitize_text_field($data['seo_title']),
            'seo_description' => sanitize_textarea_field($data['seo_description'])
        );
        
        $result = $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            $events_table,
            $event_data,
            array('id' => $this->id),
            null,
            array('%d')
        );
        
        if ($result !== false) {
            // Invalidate event cache
            wp_cache_delete('tpg_event_' . $this->id, 'ticketpaygo');
            $this->load_event();
            return true;
        }
        
        return false;
    }
    
    /**
     * Delete event
     */
    public function delete() {
        global $wpdb;
        
        if (!$this->exists()) {
            return false;
        }
        
        $events_table = $wpdb->prefix . 'ticketpaygo_lite_events';
        
        // Delete related orders and tickets first
        $this->delete_related_data();
        
        $result = $wpdb->delete( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            $events_table,
            array('id' => $this->id),
            array('%d')
        );
        
        if ($result !== false) {
            // Invalidate event cache
            wp_cache_delete('tpg_event_' . $this->id, 'ticketpaygo');
            $this->data = null;
            $this->id = 0;
            return true;
        }
        
        return false;
    }
    
    /**
     * Delete related data (orders, tickets, analytics)
     */
    private function delete_related_data() {
        global $wpdb;
        
        $orders_table = $wpdb->prefix . 'ticketpaygo_lite_orders';
        $tickets_table = $wpdb->prefix . 'ticketpaygo_lite_tickets';
        $analytics_table = $wpdb->prefix . 'ticketpaygo_analytics';
        
        // Get order IDs for this event
        // Table name is safe: constructed from $wpdb->prefix + hardcoded 'ticketpaygo_lite_orders'
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table, table name is safely constructed from prefix + hardcoded suffix
        $order_ids = $wpdb->get_col($wpdb->prepare(
            "SELECT id FROM {$orders_table} WHERE event_id = %d",
            $this->id
        ));
        
        if (!empty($order_ids)) {
            $placeholders = implode(',', array_fill(0, count($order_ids), '%d'));
            $sql = "DELETE FROM {$tickets_table} WHERE order_id IN ($placeholders)";
            $prepared = $wpdb->prepare($sql, $order_ids); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
            
            // Delete tickets
            $wpdb->query($prepared); // phpcs:ignore WordPress.DB.DirectDatabaseQuery,WordPress.DB.PreparedSQL.NotPrepared
        }
        
        // Delete orders
        $wpdb->delete($orders_table, array('event_id' => $this->id), array('%d')); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        
        // Delete analytics
        $wpdb->delete($analytics_table, array('event_id' => $this->id), array('%d')); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    }
    
    /**
     * Get tickets sold count
     */
    public function get_tickets_sold() {
        global $wpdb;
        
        // Check cache first (60 second TTL for dynamic sales data)
        $cache_key = 'tpg_tickets_sold_' . $this->id;
        $cached = wp_cache_get($cache_key, 'ticketpaygo');
        if (false !== $cached) {
            return intval($cached);
        }
        
        $orders_table = $wpdb->prefix . 'ticketpaygo_lite_orders';
        
        // Table name is safe: constructed from $wpdb->prefix + hardcoded 'ticketpaygo_lite_orders'
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table, table name is safely constructed from prefix + hardcoded suffix
        $count = $wpdb->get_var($wpdb->prepare(
            "SELECT SUM(quantity) FROM {$orders_table} WHERE event_id = %d AND status = 'completed'",
            $this->id
        ));
        
        $count = intval($count);
        // Cache for 60 seconds (dynamic sales data)
        wp_cache_set($cache_key, $count, 'ticketpaygo', 60);
        
        return $count;
    }
    
    /**
     * Get remaining tickets
     */
    public function get_remaining_tickets() {
        $max_tickets = intval($this->get('max_tickets'));
        if ($max_tickets <= 0) {
            return -1; // Unlimited
        }
        
        $sold = $this->get_tickets_sold();
        return max(0, $max_tickets - $sold);
    }
    
    /**
     * Check if event is sold out
     */
    public function is_sold_out() {
        $remaining = $this->get_remaining_tickets();
        return $remaining === 0;
    }
    
    /**
     * Check if event is active
     */
    public function is_active() {
        return $this->get('status') === 'active';
    }
    
    /**
     * Check if event has started
     */
    public function has_started() {
        $start_date = $this->get('start_date');
        return strtotime($start_date) <= current_time('timestamp');
    }
    
    /**
     * Check if event has ended
     */
    public function has_ended() {
        $end_date = $this->get('end_date');
        return strtotime($end_date) <= current_time('timestamp');
    }
    
    /**
     * Get event status based on dates (upcoming, ongoing, ended)
     */
    public function get_event_status() {
        $now = current_time('timestamp');
        $start_time = strtotime($this->get('start_date'));
        $end_time = strtotime($this->get('end_date'));
        
        if ($now < $start_time) {
            return 'upcoming';
        } elseif ($now >= $start_time && $now <= $end_time) {
            return 'ongoing';
        } else {
            return 'ended';
        }
    }
    
    /**
     * Get display status (combines manual status with date-based status)
     */
    public function get_display_status() {
        $manual_status = $this->get('status');
        
        // If manually set to inactive, respect that
        if ($manual_status === 'inactive') {
            return 'inactive';
        }
        
        // Otherwise, return date-based status
        return $this->get_event_status();
    }
    
    /**
     * Get formatted price
     */
    public function get_formatted_price() {
        $symbol = TicketPayGo_Database::get_setting('currency_symbol', '€');
        $position = TicketPayGo_Database::get_setting('currency_position', 'before');
        
        // Check if event has multiple ticket types
        $ticket_types = $this->get('ticket_types');
        if (is_string($ticket_types)) {
            $ticket_types = json_decode($ticket_types, true);
        }
        
        if (!empty($ticket_types) && is_array($ticket_types) && count($ticket_types) > 0) {
            // Get min and max prices from ticket types
            $prices = array_column($ticket_types, 'price');
            $min_price = min($prices);
            $max_price = max($prices);
            
            if ($min_price === $max_price) {
                // All tickets same price
                $formatted_price = number_format($min_price, 2);
                return $position === 'before' ? $symbol . $formatted_price : $formatted_price . $symbol;
            } else {
                // Show price range
                $formatted_min = number_format($min_price, 2);
                $formatted_max = number_format($max_price, 2);
                if ($position === 'before') {
                    return $symbol . $formatted_min . ' - ' . $symbol . $formatted_max;
                } else {
                    return $formatted_min . $symbol . ' - ' . $formatted_max . $symbol;
                }
            }
        }
        
        // Single price (no ticket types)
        $price = floatval($this->get('price'));
        $formatted_price = number_format($price, 2);
        
        if ($position === 'before') {
            return $symbol . $formatted_price;
        } else {
            return $formatted_price . $symbol;
        }
    }
    
    /**
     * Get event URL
     */
    public function get_url() {
        return add_query_arg('event_id', $this->id, home_url('/event/'));
    }
    
    /**
     * Record analytics
     */
    public function record_analytics($metric_type, $metric_value, $metric_data = '') {
        global $wpdb;
        
        $analytics_table = $wpdb->prefix . 'ticketpaygo_analytics';
        
        // Check if analytics table exists before trying to insert
        $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $analytics_table)); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery
        if (!$table_exists) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Logging analytics table check for debugging
            error_log('TicketPayGo: Analytics table does not exist, skipping analytics recording');
            return false;
        }
        
        $wpdb->insert( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            $analytics_table,
            array(
                'event_id' => $this->id,
                'metric_type' => $metric_type,
                'metric_value' => $metric_value,
                'metric_data' => $metric_data,
                'date_recorded' => current_time('Y-m-d')
            )
        );
    }
    
    /**
     * Get all events
     */
    public static function get_all_events($args = array()) {
        global $wpdb;
        
        $events_table = $wpdb->prefix . 'ticketpaygo_lite_events';
        
        // Check if events table exists
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table check
        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $events_table)) != $events_table) {
            return array();
        }
        
        $defaults = array(
            'status' => 'active',
            'limit' => 10,
            'offset' => 0,
            'orderby' => 'start_date',
            'order' => 'ASC'
        );
        
        $args = wp_parse_args($args, $defaults);
        
        $where = array();
        $where_values = array();
        
        if (!empty($args['status'])) {
            $where[] = 'status = %s';
            $where_values[] = $args['status'];
        }
        
        if (!empty($args['search'])) {
            $where[] = '(title LIKE %s OR description LIKE %s OR location LIKE %s)';
            $search_term = '%' . $wpdb->esc_like($args['search']) . '%';
            $where_values[] = $search_term;
            $where_values[] = $search_term;
            $where_values[] = $search_term;
        }
        
        $where_clause = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';
        
        $orderby = sanitize_sql_orderby($args['orderby'] . ' ' . $args['order']);
        $limit = intval($args['limit']);
        $offset = intval($args['offset']);
        
        $query = "SELECT * FROM $events_table $where_clause ORDER BY $orderby LIMIT $limit OFFSET $offset";
        
        if (!empty($where_values)) {
            $query = $wpdb->prepare($query, $where_values); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query is dynamically built with proper placeholders
        }
        
        $events = $wpdb->get_results($query, ARRAY_A); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared,WordPress.DB.DirectDatabaseQuery
        
        // Decode JSON fields for each event
        foreach ($events as &$event) {
            $event['ticket_types'] = json_decode($event['ticket_types'], true) ?: array();
            $event['custom_fields'] = json_decode($event['custom_fields'], true) ?: array();
            $event['gallery'] = json_decode($event['gallery'], true) ?: array();
        }
        
        return $events;
    }
    
    /**
     * Get upcoming events
     */
    public static function get_upcoming_events($limit = 10) {
        global $wpdb;
        
        $events_table = $wpdb->prefix . 'ticketpaygo_lite_events';
        
        // Check if events table exists
        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $events_table)) != $events_table) { // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            return array();
        }
        
        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
        $events = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$events_table}
             WHERE status = 'active' AND end_date > %s 
             ORDER BY start_date ASC 
             LIMIT %d",
            current_time('Y-m-d H:i:s'),
            $limit
        ), ARRAY_A);
        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
        
        // Decode JSON fields for each event
        foreach ($events as &$event) {
            $event['ticket_types'] = json_decode($event['ticket_types'], true) ?: array();
            $event['custom_fields'] = json_decode($event['custom_fields'], true) ?: array();
            $event['gallery'] = json_decode($event['gallery'], true) ?: array();
        }
        
        return $events;
    }
    
    /**
     * Get events count
     */
    public static function get_events_count($status = 'active') {
        global $wpdb;
        
        // Check cache first
        $cache_key = 'tpg_events_count_' . $status;
        $cached = wp_cache_get($cache_key, 'ticketpaygo');
        if (false !== $cached) {
            return intval($cached);
        }
        
        $events_table = $wpdb->prefix . 'ticketpaygo_lite_events';
        
        // Table name is safe: constructed from $wpdb->prefix + hardcoded 'ticketpaygo_lite_events'
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table, table name is safely constructed from prefix + hardcoded suffix
        $count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$events_table} WHERE status = %s",
            $status
        ));
        
        $count = intval($count);
        // Cache for 5 minutes
        wp_cache_set($cache_key, $count, 'ticketpaygo', 300);
        
        return $count;
    }
}
