<?php
/**
 * Email functionality class
 * 
 * phpcs:disable PluginCheck.Security.DirectDB.UnescapedDBParameter
 */

if (!defined('ABSPATH')) {
    exit;
}

class TicketPayGo_Email {
    
    private $from_name;
    private $from_email;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->from_name = TicketPayGo_Database::get_setting('email_from_name', get_bloginfo('name'));
        $this->from_email = TicketPayGo_Database::get_setting('email_from_email', get_option('admin_email'));
        
        // Set email headers
        add_filter('wp_mail_from', array($this, 'set_from_email'));
        add_filter('wp_mail_from_name', array($this, 'set_from_name'));
        add_filter('wp_mail_content_type', array($this, 'set_content_type'));
    }
    
    /**
     * Set from email
     */
    public function set_from_email($email) {
        return $this->from_email;
    }
    
    /**
     * Set from name
     */
    public function set_from_name($name) {
        return $this->from_name;
    }
    
    /**
     * Set content type to HTML
     */
    public function set_content_type($content_type) {
        return 'text/html';
    }
    
    /**
     * Send order confirmation email
     */
    public function send_order_confirmation($order) {
        if (!TicketPayGo_Database::get_setting('email_notifications', '1')) {
            return false;
        }
        
        // Get customizable email settings
        $subject = TicketPayGo_Database::get_setting('email_subject', 'Order Confirmation - {event_title}');
        $heading = TicketPayGo_Database::get_setting('email_heading', 'Order Confirmation');
        $greeting = TicketPayGo_Database::get_setting('email_greeting', 'Dear {customer_name},\n\nThank you for your ticket purchase! Your order has been confirmed.');
        $footer_text = TicketPayGo_Database::get_setting('email_footer_text', 'Best regards,\nThe TicketPayGo Team');
        
        // Get template with custom text
        $body = $this->get_customizable_confirmation_template($heading, $greeting, $footer_text);
        
        // Replace placeholders
        $placeholders = $this->get_order_placeholders($order);
        $subject = $this->replace_placeholders($subject, $placeholders);
        $body = $this->replace_placeholders($body, $placeholders);
        
        // Get tickets for attachment
        $tickets = $order->get_tickets();
        $attachments = array();
        
        foreach ($tickets as $ticket_data) {
            $ticket = new TicketPayGo_Ticket($ticket_data['id']);
            $pdf_path = $ticket->generate_pdf();
            if ($pdf_path) {
                $attachments[] = $pdf_path;
            }
        }
        
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . $this->from_name . ' <' . $this->from_email . '>'
        );
        
        return wp_mail(
            $order->get('customer_email'),
            $subject,
            $body,
            $headers,
            $attachments
        );
    }
    
    /**
     * Send event reminder email
     */
    public function send_event_reminder($event_id, $days_before = 1) {
        $event = new TicketPayGo_Event($event_id);
        if (!$event->exists()) {
            return false;
        }
        
        // Get all orders for this event
        $orders = TicketPayGo_Order::get_all_orders(array(
            'event_id' => $event_id,
            'status' => 'completed',
            'limit' => -1
        ));
        
        $templates = json_decode(TicketPayGo_Database::get_setting('email_templates', '{}'), true);
        $template = $templates['ticket_reminder'] ?? array();
        
        $subject = $template['subject'] ?? 'Event Reminder - {event_title}';
        $body = $template['body'] ?? $this->get_default_reminder_template();
        
        $sent_count = 0;
        
        foreach ($orders as $order_data) {
            $order = new TicketPayGo_Order($order_data['id']);
            
            // Replace placeholders
            $placeholders = $this->get_order_placeholders($order);
            $email_subject = $this->replace_placeholders($subject, $placeholders);
            $email_body = $this->replace_placeholders($body, $placeholders);
            
            $headers = array(
                'Content-Type: text/html; charset=UTF-8',
                'From: ' . $this->from_name . ' <' . $this->from_email . '>'
            );
            
            if (wp_mail($order->get('customer_email'), $email_subject, $email_body, $headers)) {
                $sent_count++;
            }
        }
        
        return $sent_count;
    }
    
    /**
     * Send ticket delivery email
     */
    public function send_ticket_delivery($ticket_id) {
        $ticket = new TicketPayGo_Ticket($ticket_id);
        if (!$ticket->exists()) {
            return false;
        }
        
        $order = new TicketPayGo_Order($ticket->get('order_id'));
        $event = new TicketPayGo_Event($ticket->get('event_id'));
        
        $subject = 'Your Ticket for ' . $event->get('title');
        $body = $this->get_ticket_delivery_template();
        
        // Replace placeholders
        $placeholders = array_merge(
            $this->get_order_placeholders($order),
            $this->get_ticket_placeholders($ticket),
            $this->get_event_placeholders($event)
        );
        
        $body = $this->replace_placeholders($body, $placeholders);
        
        // Attach ticket PDF
        $pdf_path = $ticket->generate_pdf();
        $attachments = $pdf_path ? array($pdf_path) : array();
        
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . $this->from_name . ' <' . $this->from_email . '>'
        );
        
        return wp_mail(
            $ticket->get('holder_email') ?: $order->get('customer_email'),
            $subject,
            $body,
            $headers,
            $attachments
        );
    }
    
    /**
     * Send admin notification
     */
    public function send_admin_notification($type, $data) {
        $admin_email = get_option('admin_email');
        
        switch ($type) {
            case 'new_order':
                $subject = 'New Ticket Order - ' . $data['event_title'];
                $body = $this->get_admin_order_template($data);
                break;
            case 'payment_failed':
                $subject = 'Payment Failed - Order #' . $data['order_number'];
                $body = $this->get_admin_payment_failed_template($data);
                break;
            case 'refund_processed':
                $subject = 'Refund Processed - Order #' . $data['order_number'];
                $body = $this->get_admin_refund_template($data);
                break;
            default:
                return false;
        }
        
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . $this->from_name . ' <' . $this->from_email . '>'
        );
        
        return wp_mail($admin_email, $subject, $body, $headers);
    }
    
    /**
     * Get order placeholders
     */
    private function get_order_placeholders($order) {
        $placeholders = array(
            '{order_number}' => $order->get('order_number'),
            '{customer_name}' => $order->get('customer_name'),
            '{customer_email}' => $order->get('customer_email'),
            '{quantity}' => $order->get('quantity'),
            '{total_amount}' => $order->get_formatted_total(),
            '{order_date}' => gmdate('F j, Y', strtotime($order->get('created_at'))),
            '{event_title}' => $order->get('event_title'),
            '{event_date}' => gmdate('F j, Y', strtotime($order->get('start_date'))),
            '{event_time}' => gmdate('g:i A', strtotime($order->get('start_date'))),
            '{event_location}' => $order->get('location')
        );
        
        // Add calendar links if the class exists
        if (class_exists('TicketPayGo_Calendar_Links')) {
            $event_id = $order->get('event_id');
            if ($event_id) {
                $calendar_links = TicketPayGo_Calendar_Links::generate($event_id);
                $placeholders['{calendar_google}'] = $calendar_links['google'] ?? '#';
                $placeholders['{calendar_apple}'] = $calendar_links['apple'] ?? '#';
                $placeholders['{calendar_outlook}'] = $calendar_links['outlook'] ?? '#';
                $placeholders['{calendar_yahoo}'] = $calendar_links['yahoo'] ?? '#';
            }
        }
        
        return $placeholders;
    }
    
    /**
     * Get ticket placeholders
     */
    private function get_ticket_placeholders($ticket) {
        return array(
            '{ticket_number}' => $ticket->get('ticket_number'),
            '{ticket_holder}' => $ticket->get('holder_name')
        );
    }
    
    /**
     * Get event placeholders
     */
    private function get_event_placeholders($event) {
        return array(
            '{event_title}' => $event->get('title'),
            '{event_description}' => $event->get('description'),
            '{event_date}' => gmdate('F j, Y', strtotime($event->get('start_date'))),
            '{event_time}' => gmdate('g:i A', strtotime($event->get('start_date'))),
            '{event_location}' => $event->get('location'),
            '{event_address}' => $event->get('address'),
            '{organizer_name}' => $event->get('organizer_name'),
            '{organizer_email}' => $event->get('organizer_email')
        );
    }
    
    /**
     * Replace placeholders in text
     */
    private function replace_placeholders($text, $placeholders) {
        return str_replace(array_keys($placeholders), array_values($placeholders), $text);
    }
    
    /**
     * Get customizable confirmation email template
     */
    private function get_customizable_confirmation_template($heading, $greeting, $footer_text) {
        // Convert newlines to <br> tags for HTML
        $greeting_html = nl2br(esc_html($greeting));
        $footer_html = nl2br(esc_html($footer_text));
        
        return '
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
            <h2 style="color: #333;">' . esc_html($heading) . '</h2>
            <p>' . $greeting_html . '</p>
            
            <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin: 20px 0;">
                <h3 style="margin-top: 0;">Order Details</h3>
                <p><strong>Order Number:</strong> {order_number}</p>
                <p><strong>Event:</strong> {event_title}</p>
                <p><strong>Date:</strong> {event_date} at {event_time}</p>
                <p><strong>Location:</strong> {event_location}</p>
                <p><strong>Quantity:</strong> {quantity} ticket(s)</p>
                <p><strong>Total:</strong> {total_amount}</p>
            </div>
            
            <p>Please show this email or your order number <strong>{order_number}</strong> at the event entrance.</p>
            
            <p>If you have any questions, please don\'t hesitate to contact us.</p>
            
            <p>' . $footer_html . '</p>
        </div>';
    }
    
    /**
     * Get default confirmation email template
     */
    private function get_default_confirmation_template() {
        return '
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
            <h2 style="color: #333;">Order Confirmation</h2>
            <p>Dear {customer_name},</p>
            <p>Thank you for your ticket purchase! Your order has been confirmed.</p>
            
            <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin: 20px 0;">
                <h3 style="margin-top: 0;">Order Details</h3>
                <p><strong>Order Number:</strong> {order_number}</p>
                <p><strong>Event:</strong> {event_title}</p>
                <p><strong>Date:</strong> {event_date} at {event_time}</p>
                <p><strong>Location:</strong> {event_location}</p>
                <p><strong>Quantity:</strong> {quantity} ticket(s)</p>
                <p><strong>Total:</strong> {total_amount}</p>
            </div>
            
            <p>Please show this email or your order number <strong>{order_number}</strong> at the event entrance.</p>
            
            <p>If you have any questions, please don\'t hesitate to contact us.</p>
            
            <p>Best regards,<br>The TicketPayGo Team</p>
        </div>';
    }
    
    /**
     * Get default reminder email template
     */
    private function get_default_reminder_template() {
        return '
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
            <h2 style="color: #333;">Event Reminder</h2>
            <p>Dear {customer_name},</p>
            <p>This is a friendly reminder about your upcoming event!</p>
            
            <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin: 20px 0;">
                <h3 style="margin-top: 0;">{event_title}</h3>
                <p><strong>Date:</strong> {event_date} at {event_time}</p>
                <p><strong>Location:</strong> {event_location}</p>
                <p><strong>Your Tickets:</strong> {quantity}</p>
            </div>
            
            <p>Don\'t forget to bring your tickets! You can find them attached to your original confirmation email.</p>
            
            <p>We look forward to seeing you at the event!</p>
            
            <p>Best regards,<br>The Event Team</p>
        </div>';
    }
    
    /**
     * Get ticket delivery template
     */
    private function get_ticket_delivery_template() {
        return '
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
            <h2 style="color: #333;">Your Event Ticket</h2>
            <p>Dear {ticket_holder},</p>
            <p>Your ticket for {event_title} is attached to this email.</p>
            
            <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin: 20px 0;">
                <h3 style="margin-top: 0;">Event Details</h3>
                <p><strong>Event:</strong> {event_title}</p>
                <p><strong>Date:</strong> {event_date} at {event_time}</p>
                <p><strong>Location:</strong> {event_location}</p>
                <p><strong>Ticket Number:</strong> {ticket_number}</p>
            </div>
            
            <p>Please bring this ticket (printed or on your mobile device) to the event entrance.</p>
            
            <p>Enjoy the event!</p>
        </div>';
    }
    
    /**
     * Get admin order notification template
     */
    private function get_admin_order_template($data) {
        return '
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
            <h2 style="color: #333;">New Ticket Order</h2>
            <p>A new ticket order has been placed on your website.</p>
            
            <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin: 20px 0;">
                <h3 style="margin-top: 0;">Order Details</h3>
                <p><strong>Order Number:</strong> ' . esc_html($data['order_number']) . '</p>
                <p><strong>Event:</strong> ' . esc_html($data['event_title']) . '</p>
                <p><strong>Customer:</strong> ' . esc_html($data['customer_name']) . ' (' . esc_html($data['customer_email']) . ')</p>
                <p><strong>Quantity:</strong> ' . intval($data['quantity']) . ' ticket(s)</p>
                <p><strong>Total:</strong> ' . esc_html($data['total_amount']) . '</p>
                <p><strong>Payment Method:</strong> ' . esc_html($data['payment_method']) . '</p>
            </div>
            
            <p>You can view and manage this order in your WordPress admin panel.</p>
        </div>';
    }
    
    /**
     * Get admin payment failed template
     */
    private function get_admin_payment_failed_template($data) {
        return '
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
            <h2 style="color: #dc3545;">Payment Failed</h2>
            <p>A payment has failed for order #' . esc_html($data['order_number']) . '.</p>
            
            <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin: 20px 0;">
                <p><strong>Order Number:</strong> ' . esc_html($data['order_number']) . '</p>
                <p><strong>Customer:</strong> ' . esc_html($data['customer_name']) . '</p>
                <p><strong>Amount:</strong> ' . esc_html($data['total_amount']) . '</p>
                <p><strong>Payment Method:</strong> ' . esc_html($data['payment_method']) . '</p>
            </div>
            
            <p>Please check the order details in your admin panel.</p>
        </div>';
    }
    
    /**
     * Get admin refund template
     */
    private function get_admin_refund_template($data) {
        return '
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
            <h2 style="color: #28a745;">Refund Processed</h2>
            <p>A refund has been processed for order #' . esc_html($data['order_number']) . '.</p>
            
            <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin: 20px 0;">
                <p><strong>Order Number:</strong> ' . esc_html($data['order_number']) . '</p>
                <p><strong>Customer:</strong> ' . esc_html($data['customer_name']) . '</p>
                <p><strong>Refund Amount:</strong> ' . esc_html($data['refund_amount']) . '</p>
                <p><strong>Refund ID:</strong> ' . esc_html($data['refund_id']) . '</p>
            </div>
        </div>';
    }
    
    /**
     * Schedule reminder emails
     */
    public function schedule_reminders() {
        // Get events starting in 24 hours
        global $wpdb;
        $events_table = $wpdb->prefix . 'ticketpaygo_lite_events';
        
        $tomorrow = gmdate('Y-m-d H:i:s', strtotime('+24 hours'));
        $day_after = gmdate('Y-m-d H:i:s', strtotime('+25 hours'));
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        $events = $wpdb->get_results($wpdb->prepare(
            "SELECT id FROM {$wpdb->prefix}ticketpaygo_lite_events 
             WHERE status = 'active' 
             AND start_date BETWEEN %s AND %s",
            $tomorrow,
            $day_after
        ));
        
        foreach ($events as $event) {
            $this->send_event_reminder($event->id);
        }
    }
    
    /**
     * Send test email
     */
    public static function send_test_email($to_email) {
        $settings = get_option('ticketpaygo_email_settings', array());
        
        $from_name = $settings['from_name'] ?? get_bloginfo('name');
        $from_email = $settings['from_email'] ?? get_bloginfo('admin_email');
        $subject = 'Test Email - ' . get_bloginfo('name');
        $template = $settings['email_template'] ?? self::get_default_template();
        
        // QR codes and wallet buttons not available in Lite version
        
        $placeholders = array(
            '{customer_name}' => 'John Doe',
            '{customer_email}' => 'john@example.com',
            '{event_title}' => 'Sample Event',
            '{event_date}' => gmdate('F j, Y'),
            '{event_time}' => '19:00',
            '{event_location}' => 'Sample Venue',
            '{order_number}' => 'TEST-12345',
            '{order_id}' => 'TEST-12345',
            '{tickets}' => '2 x General Admission',
            '{total_amount}' => '€50.00'
        );
        
        $body = str_replace(array_keys($placeholders), array_values($placeholders), $template);
        
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . $from_name . ' <' . $from_email . '>'
        );
        
        // Note: Test emails don't include PDF attachments
        // Real order emails will have PDF tickets attached
        
        return wp_mail($to_email, $subject, $body, $headers);
    }
    
    /**
     * Get default email template
     */
    public static function get_default_template() {
        return '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>' . esc_html__('Order Confirmation', 'ticketpaygo-lite') . '</title>
</head>
<body style="margin: 0; padding: 0; font-family: Arial, sans-serif; background-color: #f4f4f4;">
    <table width="100%" cellpadding="0" cellspacing="0" style="background-color: #f4f4f4; padding: 20px 0;">
        <tr>
            <td align="center">
                <table width="600" cellpadding="0" cellspacing="0" style="background-color: #ffffff; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
                    <!-- Header -->
                    <tr>
                        <td style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 40px 30px; text-align: center;">
                            <h1 style="margin: 0; color: #ffffff; font-size: 28px; font-weight: bold;">' . esc_html__('Order Confirmation', 'ticketpaygo-lite') . '</h1>
                            <p style="margin: 10px 0 0 0; color: #ffffff; font-size: 16px; opacity: 0.9;">' . esc_html__('Thank you for your purchase!', 'ticketpaygo-lite') . '</p>
                        </td>
                    </tr>
                    
                    <!-- Body -->
                    <tr>
                        <td style="padding: 40px 30px;">
                            <p style="margin: 0 0 20px 0; color: #333333; font-size: 16px; line-height: 1.6;">
                                ' . esc_html__('Dear', 'ticketpaygo-lite') . ' <strong>{customer_name}</strong>,
                            </p>
                            
                            <p style="margin: 0 0 30px 0; color: #666666; font-size: 15px; line-height: 1.6;">
                                ' . esc_html__('Your ticket order has been confirmed! We\'re excited to see you at the event.', 'ticketpaygo-lite') . '
                            </p>
                            
                            <!-- Event Details Box -->
                            <table width="100%" cellpadding="0" cellspacing="0" style="background-color: #f8fafc; border-radius: 8px; border: 2px solid #e2e8f0; margin-bottom: 30px;">
                                <tr>
                                    <td style="padding: 25px;">
                                        <h2 style="margin: 0 0 20px 0; color: #1e293b; font-size: 20px; font-weight: bold;">' . esc_html__('Event Details', 'ticketpaygo-lite') . '</h2>
                                        
                                        <table width="100%" cellpadding="8" cellspacing="0">
                                            <tr>
                                                <td style="color: #64748b; font-size: 14px; font-weight: 600; width: 120px;">' . esc_html__('Event:', 'ticketpaygo-lite') . '</td>
                                                <td style="color: #1e293b; font-size: 14px; font-weight: bold;">{event_title}</td>
                                            </tr>
                                            <tr>
                                                <td style="color: #64748b; font-size: 14px; font-weight: 600;">' . esc_html__('Date:', 'ticketpaygo-lite') . '</td>
                                                <td style="color: #1e293b; font-size: 14px;">{event_date}</td>
                                            </tr>
                                            <tr>
                                                <td style="color: #64748b; font-size: 14px; font-weight: 600;">' . esc_html__('Location:', 'ticketpaygo-lite') . '</td>
                                                <td style="color: #1e293b; font-size: 14px;">{event_location}</td>
                                            </tr>
                                            <tr>
                                                <td style="color: #64748b; font-size: 14px; font-weight: 600;">' . esc_html__('Order ID:', 'ticketpaygo-lite') . '</td>
                                                <td style="color: #1e293b; font-size: 14px;">{order_id}</td>
                                            </tr>
                                            <tr>
                                                <td style="color: #64748b; font-size: 14px; font-weight: 600;">' . esc_html__('Tickets:', 'ticketpaygo-lite') . '</td>
                                                <td style="color: #1e293b; font-size: 14px;">{tickets}</td>
                                            </tr>
                                            <tr>
                                                <td style="color: #64748b; font-size: 14px; font-weight: 600;">' . esc_html__('Total:', 'ticketpaygo-lite') . '</td>
                                                <td style="color: #22c55e; font-size: 18px; font-weight: bold;">{total_amount}</td>
                                            </tr>
                                        </table>
                                    </td>
                                </tr>
                            </table>
                            
                            <!-- Ticket Number -->
                            <table width="100%" cellpadding="0" cellspacing="0" style="margin-bottom: 30px;">
                                <tr>
                                    <td align="center" style="padding: 20px; background-color: #f8fafc; border-radius: 8px;">
                                        <p style="margin: 0 0 15px 0; color: #1e293b; font-size: 16px; font-weight: 600;">' . esc_html__('Your Order Number', 'ticketpaygo-lite') . '</p>
                                        <p style="margin: 0; font-size: 28px; font-weight: bold; font-family: monospace; color: #1e293b;">{order_number}</p>
                                        <p style="margin: 15px 0 0 0; color: #64748b; font-size: 13px;">' . esc_html__('Present this number at the event entrance', 'ticketpaygo-lite') . '</p>
                                    </td>
                                </tr>
                            </table>
                            
                            <!-- Mobile Wallet Buttons -->
                            {wallet_buttons}
                            
                            <p style="margin: 0 0 20px 0; color: #666666; font-size: 15px; line-height: 1.6;">
                                ' . esc_html__('Please save this email and present your order number at the event entrance.', 'ticketpaygo-lite') . '
                            </p>
                            
                            <p style="margin: 0; color: #666666; font-size: 15px; line-height: 1.6;">
                                ' . esc_html__('If you have any questions, please don\'t hesitate to contact us.', 'ticketpaygo-lite') . '
                            </p>
                        </td>
                    </tr>
                    
                    <!-- Footer -->
                    <tr>
                        <td style="background-color: #f8fafc; padding: 30px; text-align: center; border-top: 1px solid #e2e8f0;">
                            <p style="margin: 0 0 10px 0; color: #64748b; font-size: 14px;">
                                ' . esc_html__('Best regards,', 'ticketpaygo-lite') . '<br>
                                <strong style="color: #1e293b;">' . esc_html__('The TicketPayGo Team', 'ticketpaygo-lite') . '</strong>
                            </p>
                            <p style="margin: 10px 0 0 0; color: #94a3b8; font-size: 12px;">
                                ' . esc_html__('This is an automated email. Please do not reply to this message.', 'ticketpaygo-lite') . '
                            </p>
                        </td>
                    </tr>
                </table>
            </td>
        </tr>
    </table>
</body>
</html>';
    }
}
