<?php
/**
 * Database management class for TicketPayGo Lite
 * 
 * phpcs:disable PluginCheck.Security.DirectDB.UnescapedDBParameter
 */

if (!defined('ABSPATH')) {
    exit;
}

class TicketPayGo_Database {
    
    /**
     * Check if tables exist
     */
    public static function tables_exist() {
        global $wpdb;
        
        $tables = array(
            $wpdb->prefix . 'ticketpaygo_lite_events',
            $wpdb->prefix . 'ticketpaygo_lite_orders',
            $wpdb->prefix . 'ticketpaygo_lite_tickets',
            $wpdb->prefix . 'ticketpaygo_lite_analytics',
            $wpdb->prefix . 'ticketpaygo_lite_payment_logs',
            $wpdb->prefix . 'ticketpaygo_lite_settings'
        );
        
        $missing_tables = array();
        
        foreach ($tables as $table) {
            $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table)); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            if ($table_exists != $table) {
                $missing_tables[] = $table;
            }
        }
        
        if (!empty($missing_tables)) {
            return false;
        }
        return true;
    }
    
    /**
     * Create database tables
     */
    public static function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Events table
        $events_table = $wpdb->prefix . 'ticketpaygo_lite_events';
        $events_sql = "CREATE TABLE $events_table (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            title varchar(255) NOT NULL,
            description longtext,
            start_date datetime NOT NULL,
            end_date datetime NOT NULL,
            location varchar(255),
            address text,
            latitude decimal(10, 8),
            longitude decimal(11, 8),
            max_tickets int(11) DEFAULT 0,
            price decimal(10, 2) NOT NULL DEFAULT 0.00,
            currency varchar(3) DEFAULT 'EUR',
            status varchar(20) DEFAULT 'active',
            featured_image varchar(255),
            image_fit varchar(20) DEFAULT 'cover',
            show_title_on_card tinyint(1) DEFAULT 1,
            ticket_name varchar(255),
            gallery text,
            organizer_name varchar(255),
            organizer_email varchar(255),
            organizer_phone varchar(50),
            ticket_types text,
            recurrence text,
            custom_fields text,
            seo_title varchar(255),
            seo_description text,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY status (status),
            KEY start_date (start_date)
        ) $charset_collate;";
        
        // Orders table
        $orders_table = $wpdb->prefix . 'ticketpaygo_lite_orders';
        $orders_sql = "CREATE TABLE $orders_table (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            order_number varchar(50) NOT NULL UNIQUE,
            event_id mediumint(9) NOT NULL,
            customer_name varchar(255) NOT NULL,
            customer_email varchar(255) NOT NULL,
            customer_phone varchar(50),
            customer_address text,
            customer_city varchar(100),
            customer_country varchar(100),
            quantity int(11) NOT NULL DEFAULT 1,
            total_amount decimal(10, 2) NOT NULL,
            currency varchar(3) DEFAULT 'EUR',
            payment_method varchar(50),
            payment_status varchar(20) DEFAULT 'pending',
            payment_id varchar(255),
            payment_data text,
            status varchar(20) DEFAULT 'pending',
            notes text,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY event_id (event_id),
            KEY customer_email (customer_email),
            KEY payment_status (payment_status),
            KEY status (status)
        ) $charset_collate;";
        
        // Tickets table
        $tickets_table = $wpdb->prefix . 'ticketpaygo_lite_tickets';
        $tickets_sql = "CREATE TABLE $tickets_table (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            ticket_number varchar(50) NOT NULL UNIQUE,
            order_id mediumint(9) NOT NULL,
            event_id mediumint(9) NOT NULL,
            ticket_type varchar(100) DEFAULT 'general',
            holder_name varchar(255),
            holder_email varchar(255),
            qr_code varchar(255),
            qr_data text,
            background_image varchar(255),
            custom_design text,
            status varchar(20) DEFAULT 'active',
            scanned_at datetime NULL,
            scanned_by varchar(255),
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY order_id (order_id),
            KEY event_id (event_id),
            KEY status (status)
        ) $charset_collate;";
        
        // Analytics table
        $analytics_table = $wpdb->prefix . 'ticketpaygo_lite_analytics';
        $analytics_sql = "CREATE TABLE $analytics_table (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            event_id mediumint(9) NOT NULL,
            metric_type varchar(50) NOT NULL,
            metric_value varchar(255) NOT NULL,
            metric_data text,
            date_recorded date NOT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY event_id (event_id),
            KEY metric_type (metric_type),
            KEY date_recorded (date_recorded)
        ) $charset_collate;";
        
        // Payment logs table
        $payment_logs_table = $wpdb->prefix . 'ticketpaygo_lite_payment_logs';
        $payment_logs_sql = "CREATE TABLE $payment_logs_table (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            order_id mediumint(9) NOT NULL,
            payment_method varchar(50) NOT NULL,
            transaction_id varchar(255),
            amount decimal(10, 2) NOT NULL,
            currency varchar(3) DEFAULT 'EUR',
            status varchar(20) NOT NULL,
            gateway_response text,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY order_id (order_id),
            KEY transaction_id (transaction_id),
            KEY status (status)
        ) $charset_collate;";
        
        // Settings table
        $settings_table = $wpdb->prefix . 'ticketpaygo_lite_settings';
        $settings_sql = "CREATE TABLE $settings_table (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            setting_key varchar(100) NOT NULL UNIQUE,
            setting_value longtext,
            autoload varchar(20) DEFAULT 'yes',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        
        $results = array();
        $results['events'] = dbDelta($events_sql);
        $results['orders'] = dbDelta($orders_sql);
        $results['tickets'] = dbDelta($tickets_sql);
        $results['analytics'] = dbDelta($analytics_sql);
        $results['payment_logs'] = dbDelta($payment_logs_sql);
        $results['settings'] = dbDelta($settings_sql);
        
        
        // Check and add missing columns for upgrades
        self::upgrade_database_schema();
        
        // Insert default settings
        self::insert_default_settings();
        
        return $results;
    }
    
    /**
     * Upgrade database schema - add missing columns to existing tables
     */
    public static function upgrade_database_schema() {
        global $wpdb;
        
        $events_table = $wpdb->prefix . 'ticketpaygo_lite_events';
        
        $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $events_table)); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        if ($table_exists != $events_table) {
            return;
        }
        
        // Get current database version
        $current_db_version = get_option('tpgl_db_version', '1.0.0');
        $updated = false;
        
        // Version 1.1.0: Add recurrence column
        if (version_compare($current_db_version, '1.1.0', '<')) {
            // Table name is safe: constructed from $wpdb->prefix + hardcoded 'ticketpaygo_lite_events'
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table, table name is safely constructed from prefix + hardcoded suffix
            $column_exists = $wpdb->get_results($wpdb->prepare(
                "SHOW COLUMNS FROM `{$events_table}` LIKE %s",
                'recurrence'
            ));
            
            if (empty($column_exists)) {
                // Table name is safe: constructed from $wpdb->prefix + hardcoded 'ticketpaygo_lite_events'
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.SchemaChange,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table schema change, table name is safely constructed from prefix + hardcoded suffix
                $wpdb->query("ALTER TABLE `{$wpdb->prefix}ticketpaygo_lite_events` ADD COLUMN `recurrence` text AFTER `ticket_types`");
                $updated = true;
            }
        }
        
        // Version 1.2.0: Add image_fit column (if needed in future)
        if (version_compare($current_db_version, '1.2.0', '<')) {
            // Table name is safe: constructed from $wpdb->prefix + hardcoded 'ticketpaygo_lite_events'
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table, table name is safely constructed from prefix + hardcoded suffix
            $column_exists = $wpdb->get_results($wpdb->prepare(
                "SHOW COLUMNS FROM `{$events_table}` LIKE %s",
                'image_fit'
            ));
            
            if (empty($column_exists)) {
                // Table name is safe: constructed from $wpdb->prefix + hardcoded 'ticketpaygo_lite_events'
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.SchemaChange,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table schema change, table name is safely constructed from prefix + hardcoded suffix
                $wpdb->query("ALTER TABLE `{$wpdb->prefix}ticketpaygo_lite_events` ADD COLUMN `image_fit` varchar(20) DEFAULT 'cover' AFTER `featured_image`");
                $updated = true;
            }
        }
        
        // Version 1.3.0: Add show_title_on_card column (if needed)
        if (version_compare($current_db_version, '1.3.0', '<')) {
            // Table name is safe: constructed from $wpdb->prefix + hardcoded 'ticketpaygo_lite_events'
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table, table name is safely constructed from prefix + hardcoded suffix
            $column_exists = $wpdb->get_results($wpdb->prepare(
                "SHOW COLUMNS FROM `{$events_table}` LIKE %s",
                'show_title_on_card'
            ));
            
            if (empty($column_exists)) {
                // Table name is safe: constructed from $wpdb->prefix + hardcoded 'ticketpaygo_lite_events'
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.SchemaChange,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table schema change, table name is safely constructed from prefix + hardcoded suffix
                $wpdb->query("ALTER TABLE `{$wpdb->prefix}ticketpaygo_lite_events` ADD COLUMN `show_title_on_card` tinyint(1) DEFAULT 1 AFTER `image_fit`");
                $updated = true;
            }
        }
        
        // Version 1.4.0: Add ticket_name column
        if (version_compare($current_db_version, '1.4.0', '<')) {
            // Table name is safe: constructed from $wpdb->prefix + hardcoded 'ticketpaygo_lite_events'
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table, table name is safely constructed from prefix + hardcoded suffix
            $column_exists = $wpdb->get_results($wpdb->prepare(
                "SHOW COLUMNS FROM `{$events_table}` LIKE %s",
                'ticket_name'
            ));
            
            if (empty($column_exists)) {
                // Table name is safe: constructed from $wpdb->prefix + hardcoded 'ticketpaygo_lite_events'
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.SchemaChange,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table schema change, table name is safely constructed from prefix + hardcoded suffix
                $wpdb->query("ALTER TABLE `{$wpdb->prefix}ticketpaygo_lite_events` ADD COLUMN `ticket_name` varchar(255) AFTER `show_title_on_card`");
                $updated = true;
            }
        }
        
        if ($updated) {
            update_option('tpgl_db_version', '1.4.0');
        }
    }
    
    /**
     * Insert default settings
     */
    private static function insert_default_settings() {
        global $wpdb;
        
        $settings_table = $wpdb->prefix . 'ticketpaygo_lite_settings';
        
        $default_settings = array(
            'currency' => 'EUR',
            'currency_symbol' => '€',
            'currency_position' => 'before',
            'date_format' => 'Y-m-d',
            'time_format' => 'H:i',
            'timezone' => 'Europe/Amsterdam',
            'email_from_name' => get_bloginfo('name'),
            'email_from_email' => get_option('admin_email'),
            'payment_methods' => json_encode(array('mollie', 'paypal', 'stripe')),
            'mollie_api_key' => '',
            'mollie_test_mode' => '1',
            'paypal_client_id' => '',
            'paypal_client_secret' => '',
            'paypal_sandbox' => '1',
            'stripe_publishable_key' => '',
            'stripe_secret_key' => '',
            'stripe_test_mode' => '1',
            'ticket_customization' => '1',
            'email_notifications' => '1',
            'analytics_enabled' => '1',
            'google_analytics_id' => '',
            'facebook_pixel_id' => '',
            'ticket_pdf_template' => 'default',
            'email_templates' => json_encode(array(
                'order_confirmation' => array(
                    'subject' => 'Order Confirmation - {event_title}',
                    'body' => 'Thank you for your order! Your tickets are attached.'
                ),
                'ticket_reminder' => array(
                    'subject' => 'Event Reminder - {event_title}',
                    'body' => 'Don\'t forget about your upcoming event!'
                )
            ))
        );
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table check
        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $settings_table)) == $settings_table) {
            foreach ($default_settings as $key => $value) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table insert/update
                $wpdb->replace(
                    $settings_table,
                    array(
                        'setting_key' => $key,
                        'setting_value' => $value
                    ),
                    array('%s', '%s')
                );
            }
        }
    }
    
    /**
     * Get setting value
     */
    public static function get_setting($key, $default = '') {
        global $wpdb;
        
        $settings_table = $wpdb->prefix . 'ticketpaygo_lite_settings';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table check
        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $settings_table)) != $settings_table) {
            return $default;
        }
        
        // Table name is safe: constructed from $wpdb->prefix + hardcoded 'ticketpaygo_lite_settings'
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table, table name is safely constructed from prefix + hardcoded suffix
        $value = $wpdb->get_var($wpdb->prepare(
            "SELECT setting_value FROM {$wpdb->prefix}ticketpaygo_lite_settings WHERE setting_key = %s",
            $key
        ));
        
        return $value !== null ? $value : $default;
    }
    
    /**
     * Update setting value
     */
    public static function update_setting($key, $value) {
        global $wpdb;
        
        $settings_table = $wpdb->prefix . 'ticketpaygo_lite_settings';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table insert/update
        return $wpdb->replace(
            $settings_table,
            array(
                'setting_key' => $key,
                'setting_value' => $value
            ),
            array('%s', '%s')
        );
    }
    
    /**
     * Delete setting
     */
    public static function delete_setting($key) {
        global $wpdb;
        
        $settings_table = $wpdb->prefix . 'ticketpaygo_lite_settings';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table delete
        return $wpdb->delete(
            $settings_table,
            array('setting_key' => $key),
            array('%s')
        );
    }
    
    /**
     * Drop all plugin tables (for uninstall)
     */
    public static function drop_tables() {
        global $wpdb;
        
        // Allowlist of plugin tables - only these tables can be dropped
        // All table names are constructed from $wpdb->prefix + hardcoded suffix
        $tables = array(
            $wpdb->prefix . 'ticketpaygo_lite_events',
            $wpdb->prefix . 'ticketpaygo_lite_orders',
            $wpdb->prefix . 'ticketpaygo_lite_tickets',
            $wpdb->prefix . 'ticketpaygo_lite_analytics',
            $wpdb->prefix . 'ticketpaygo_lite_payment_logs',
            $wpdb->prefix . 'ticketpaygo_lite_settings'
        );
        
        foreach ($tables as $table) {
            // Table name is validated via allowlist above - all names are $wpdb->prefix + hardcoded suffix
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.SchemaChange,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- DROP TABLE with allowlisted table names, all constructed from prefix + hardcoded suffix
            $wpdb->query("DROP TABLE IF EXISTS `{$table}`");
        }
    }
    
    /**
     * Get database version
     */
    public static function get_db_version() {
        return get_option('tpgl_db_version', '1.0.0');
    }
    
    /**
     * Update database version
     */
    public static function update_db_version($version) {
        update_option('tpgl_db_version', $version);
    }
}
