<?php
/**
 * Calendar Links Generator
 * Generates Add-to-Calendar links for Google, Apple, Outlook, and Yahoo
 */

if (!defined('ABSPATH')) {
    exit;
}

class TicketPayGo_Calendar_Links {
    
    /**
     * Generate all calendar links for an event
     */
    public static function generate($event) {
        if (is_numeric($event)) {
            $event = new TicketPayGo_Event($event);
        }
        
        if (!$event->exists()) {
            return array();
        }
        
        $event_data = $event->get_data();
        
        return array(
            'google' => self::google_link($event_data),
            'apple' => self::ics_link($event_data),
            'outlook' => self::outlook_link($event_data),
            'yahoo' => self::yahoo_link($event_data),
            'ics' => self::ics_link($event_data)
        );
    }
    
    /**
     * Generate Google Calendar link
     */
    public static function google_link($event_data) {
        $start = self::format_date_google($event_data['start_date']);
        $end = self::format_date_google($event_data['end_date']);
        
        $params = array(
            'action' => 'TEMPLATE',
            'text' => $event_data['title'],
            'dates' => $start . '/' . $end,
            'details' => wp_strip_all_tags($event_data['description']),
            'location' => $event_data['location'] . ', ' . $event_data['address'],
            'sf' => 'true',
            'output' => 'xml'
        );
        
        return 'https://calendar.google.com/calendar/render?' . http_build_query($params);
    }
    
    /**
     * Generate Outlook.com calendar link
     */
    public static function outlook_link($event_data) {
        $start = self::format_date_iso($event_data['start_date']);
        $end = self::format_date_iso($event_data['end_date']);
        
        $params = array(
            'path' => '/calendar/action/compose',
            'rru' => 'addevent',
            'subject' => $event_data['title'],
            'startdt' => $start,
            'enddt' => $end,
            'body' => wp_strip_all_tags($event_data['description']),
            'location' => $event_data['location'] . ', ' . $event_data['address']
        );
        
        return 'https://outlook.live.com/calendar/0/deeplink/compose?' . http_build_query($params);
    }
    
    /**
     * Generate Yahoo Calendar link
     */
    public static function yahoo_link($event_data) {
        $start = self::format_date_yahoo($event_data['start_date']);
        $end = self::format_date_yahoo($event_data['end_date']);
        
        // Calculate duration in hours and minutes
        $start_time = strtotime($event_data['start_date']);
        $end_time = strtotime($event_data['end_date']);
        $duration_seconds = $end_time - $start_time;
        $hours = floor($duration_seconds / 3600);
        $minutes = floor(($duration_seconds % 3600) / 60);
        $duration = sprintf('%02d%02d', $hours, $minutes);
        
        $params = array(
            'v' => '60',
            'title' => $event_data['title'],
            'st' => $start,
            'dur' => $duration,
            'desc' => wp_strip_all_tags($event_data['description']),
            'in_loc' => $event_data['location'] . ', ' . $event_data['address']
        );
        
        return 'https://calendar.yahoo.com/?' . http_build_query($params);
    }
    
    /**
     * Generate ICS file download link (for Apple Calendar and others)
     */
    public static function ics_link($event_data) {
        return add_query_arg(array(
            'ticketpaygo_action' => 'download_ics',
            'event_id' => $event_data['id']
        ), home_url('/'));
    }
    
    /**
     * Generate ICS file content
     */
    public static function generate_ics($event_data) {
        $start = self::format_date_ics($event_data['start_date']);
        $end = self::format_date_ics($event_data['end_date']);
        $now = self::format_date_ics(current_time('mysql'));
        $uid = 'ticketpaygo-' . $event_data['id'] . '@' . wp_parse_url(home_url(), PHP_URL_HOST);
        
        $description = wp_strip_all_tags($event_data['description']);
        $description = str_replace(array("\r\n", "\r", "\n"), "\\n", $description);
        
        $location = $event_data['location'];
        if (!empty($event_data['address'])) {
            $location .= ', ' . $event_data['address'];
        }
        
        $ics = "BEGIN:VCALENDAR\r\n";
        $ics .= "VERSION:2.0\r\n";
        $ics .= "PRODID:-//TicketPayGo//Event Calendar//EN\r\n";
        $ics .= "CALSCALE:GREGORIAN\r\n";
        $ics .= "METHOD:PUBLISH\r\n";
        $ics .= "BEGIN:VEVENT\r\n";
        $ics .= "UID:" . $uid . "\r\n";
        $ics .= "DTSTAMP:" . $now . "\r\n";
        $ics .= "DTSTART:" . $start . "\r\n";
        $ics .= "DTEND:" . $end . "\r\n";
        $ics .= "SUMMARY:" . self::escape_ics_text($event_data['title']) . "\r\n";
        $ics .= "DESCRIPTION:" . self::escape_ics_text($description) . "\r\n";
        $ics .= "LOCATION:" . self::escape_ics_text($location) . "\r\n";
        
        if (!empty($event_data['organizer_name']) && !empty($event_data['organizer_email'])) {
            $ics .= "ORGANIZER;CN=" . self::escape_ics_text($event_data['organizer_name']) . ":mailto:" . $event_data['organizer_email'] . "\r\n";
        }
        
        $ics .= "STATUS:CONFIRMED\r\n";
        $ics .= "SEQUENCE:0\r\n";
        $ics .= "END:VEVENT\r\n";
        $ics .= "END:VCALENDAR\r\n";
        
        return $ics;
    }
    
    /**
     * Handle ICS file download
     */
    public static function handle_ics_download() {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Public read-only endpoint for ICS file download, no state mutation
        if (!isset($_GET['ticketpaygo_action']) || sanitize_text_field(wp_unslash($_GET['ticketpaygo_action'])) !== 'download_ics') {
            return;
        }
        
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Public read-only endpoint for ICS file download, no state mutation
        $event_id = isset($_GET['event_id']) ? intval($_GET['event_id']) : 0;
        if (!$event_id) {
            return;
        }
        
        $event = new TicketPayGo_Event($event_id);
        if (!$event->exists()) {
            return;
        }
        
        $event_data = $event->get_data();
        $ics_content = self::generate_ics($event_data);
        $filename = sanitize_file_name($event_data['title']) . '.ics';
        
        header('Content-Type: text/calendar; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . strlen($ics_content));
        header('Cache-Control: no-cache, no-store, must-revalidate');
        
        // ICS content is output as a file download (text/calendar MIME type), not HTML.
        // All user-provided content is escaped using escape_ics_text() per RFC 5545 iCalendar spec.
        // HTML escaping functions like esc_html() would corrupt the ICS format.
        echo $ics_content; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- ICS file download with text/calendar MIME type. Content is escaped per RFC 5545 via escape_ics_text(), not HTML context.
        exit;
    }
    
    /**
     * Format date for Google Calendar (YYYYMMDDTHHmmssZ)
     */
    private static function format_date_google($date) {
        $timestamp = strtotime($date);
        return gmdate('Ymd\THis\Z', $timestamp);
    }
    
    /**
     * Format date for ISO 8601
     */
    private static function format_date_iso($date) {
        $timestamp = strtotime($date);
        return gmdate('Y-m-d\TH:i:s\Z', $timestamp);
    }
    
    /**
     * Format date for Yahoo Calendar (YYYYMMDDTHHmmss)
     */
    private static function format_date_yahoo($date) {
        $timestamp = strtotime($date);
        return gmdate('Ymd\THis', $timestamp);
    }
    
    /**
     * Format date for ICS file (YYYYMMDDTHHmmssZ)
     */
    private static function format_date_ics($date) {
        $timestamp = strtotime($date);
        return gmdate('Ymd\THis\Z', $timestamp);
    }
    
    /**
     * Escape text for ICS format
     */
    private static function escape_ics_text($text) {
        $text = str_replace('\\', '\\\\', $text);
        $text = str_replace(',', '\,', $text);
        $text = str_replace(';', '\;', $text);
        $text = str_replace("\n", '\n', $text);
        return $text;
    }
    
    /**
     * Get HTML for calendar buttons
     */
    public static function get_buttons_html($event, $style = 'icons') {
        $links = self::generate($event);
        
        if (empty($links)) {
            return '';
        }
        
        $html = '<div class="ticketpaygo-calendar-links">';
        
        if ($style === 'dropdown') {
            $html .= '<div class="calendar-dropdown">';
            $html .= '<button type="button" class="calendar-dropdown-toggle">';
            $html .= '<svg width="16" height="16" viewBox="0 0 24 24" fill="none"><path d="M19 3H18V1H16V3H8V1H6V3H5C3.89 3 3.01 3.9 3.01 5L3 19C3 20.1 3.89 21 5 21H19C20.1 21 21 20.1 21 19V5C21 3.9 20.1 3 19 3ZM19 19H5V8H19V19Z" fill="currentColor"/></svg>';
            $html .= ' ' . __('Add to Calendar', 'ticketpaygo-lite');
            $html .= '</button>';
            $html .= '<div class="calendar-dropdown-menu">';
        }
        
        $html .= '<a href="' . esc_url($links['google']) . '" target="_blank" rel="noopener" class="calendar-link calendar-google" title="' . __('Add to Google Calendar', 'ticketpaygo-lite') . '">';
        $html .= '<svg width="16" height="16" viewBox="0 0 24 24"><path d="M12 0C5.376 0 0 5.376 0 12s5.376 12 12 12 12-5.376 12-12S18.624 0 12 0zm5.568 8.16c-.18-.672-.72-1.152-1.392-1.152H7.824c-.672 0-1.212.48-1.392 1.152L6 9.6v6.624c0 .672.552 1.224 1.224 1.224h9.552c.672 0 1.224-.552 1.224-1.224V9.6l-.432-1.44zM12 14.4L6.624 10.8h10.752L12 14.4z" fill="currentColor"/></svg>';
        if ($style !== 'icons') {
            $html .= ' Google';
        }
        $html .= '</a>';
        
        $html .= '<a href="' . esc_url($links['apple']) . '" class="calendar-link calendar-apple" title="' . __('Add to Apple Calendar', 'ticketpaygo-lite') . '">';
        $html .= '<svg width="16" height="16" viewBox="0 0 24 24"><path d="M18.71 19.5c-.83 1.24-1.71 2.45-3.05 2.47-1.34.03-1.77-.79-3.29-.79-1.53 0-2 .77-3.27.82-1.31.05-2.3-1.32-3.14-2.53C4.25 17 2.94 12.45 4.7 9.39c.87-1.52 2.43-2.48 4.12-2.51 1.28-.02 2.5.87 3.29.87.78 0 2.26-1.07 3.81-.91.65.03 2.47.26 3.64 1.98-.09.06-2.17 1.28-2.15 3.81.03 3.02 2.65 4.03 2.68 4.04-.03.07-.42 1.44-1.38 2.83M13 3.5c.73-.83 1.94-1.46 2.94-1.5.13 1.17-.34 2.35-1.04 3.19-.69.85-1.83 1.51-2.95 1.42-.15-1.15.41-2.35 1.05-3.11z" fill="currentColor"/></svg>';
        if ($style !== 'icons') {
            $html .= ' Apple';
        }
        $html .= '</a>';
        
        $html .= '<a href="' . esc_url($links['outlook']) . '" target="_blank" rel="noopener" class="calendar-link calendar-outlook" title="' . __('Add to Outlook', 'ticketpaygo-lite') . '">';
        $html .= '<svg width="16" height="16" viewBox="0 0 24 24"><path d="M7.88 12.04q0 .45-.11.87-.1.41-.33.74-.22.33-.58.52-.37.2-.87.2t-.85-.2q-.35-.21-.57-.55-.22-.33-.33-.75-.1-.42-.1-.86t.1-.87q.1-.43.34-.76.22-.34.59-.54.36-.2.87-.2t.86.2q.35.21.57.55.22.34.31.77.1.43.1.88zM24 12v9.38q0 .46-.33.8-.33.32-.8.32H7.13q-.46 0-.8-.33-.32-.33-.32-.8V18H1q-.41 0-.7-.3-.3-.29-.3-.7V7q0-.41.3-.7Q.58 6 1 6h6.5V2.55q0-.44.3-.75.3-.3.75-.3h12.9q.44 0 .75.3.3.3.3.75V12zm-6-8.25v3h3v-3h-3zm0 4.5v3h3v-3h-3zm0 4.5v1.83l3.05-1.83H18zm-5.25-9v3h3.75v-3h-3.75zm0 4.5v3h3.75v-3h-3.75zm0 4.5v2.03l2.41 1.5 1.34-.8v-2.73h-3.75zM9 3.75V6h2l.13.01.12.04v-2.3H9zM3.38 11q0 .36.06.71.05.34.2.64.14.3.38.53.23.23.58.38.34.15.78.15.43 0 .78-.15.35-.15.58-.38.24-.23.38-.53.15-.3.2-.64.06-.35.06-.71 0-.36-.06-.72-.05-.35-.2-.65-.14-.3-.38-.54-.23-.24-.58-.39-.35-.14-.78-.14-.44 0-.78.14-.35.15-.58.39-.24.24-.38.54-.15.3-.2.65-.06.36-.06.72z" fill="currentColor"/></svg>';
        if ($style !== 'icons') {
            $html .= ' Outlook';
        }
        $html .= '</a>';
        
        $html .= '<a href="' . esc_url($links['yahoo']) . '" target="_blank" rel="noopener" class="calendar-link calendar-yahoo" title="' . __('Add to Yahoo Calendar', 'ticketpaygo-lite') . '">';
        $html .= '<svg width="16" height="16" viewBox="0 0 24 24"><path d="M12 0C5.4 0 0 5.4 0 12s5.4 12 12 12 12-5.4 12-12S18.6 0 12 0zm3.6 17.4h-2.4l-1.2-3-1.2 3H8.4l2.4-5.4-2.4-5.4h2.4l1.2 3 1.2-3h2.4l-2.4 5.4 2.4 5.4z" fill="currentColor"/></svg>';
        if ($style !== 'icons') {
            $html .= ' Yahoo';
        }
        $html .= '</a>';
        
        if ($style === 'dropdown') {
            $html .= '</div></div>';
        }
        
        $html .= '</div>';
        
        return $html;
    }
}

// Hook ICS download handler
add_action('template_redirect', array('TicketPayGo_Calendar_Links', 'handle_ics_download'));
