<?php
/**
 * REST API Class for TicketPayGo Lite
 * Provides read-only endpoints for calendar and event display
 * 
 * Note: Order creation uses AJAX handlers, not REST API (for security)
 */

if (!defined('ABSPATH')) {
    exit;
}

class TicketPayGo_Lite_API {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('rest_api_init', array($this, 'register_routes'));
    }
    
    /**
     * Register REST API routes
     * Only read-only endpoints for calendar display
     */
    public function register_routes() {
        $namespace = 'ticketpaygo-lite/v1';
        
        // Events endpoint - Read-only for calendar display
        register_rest_route($namespace, '/events', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_events'),
            'permission_callback' => '__return_true'
        ));
        
        register_rest_route($namespace, '/events/(?P<id>\d+)', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_event'),
            'permission_callback' => '__return_true',
            'args' => array(
                'id' => array(
                    'validate_callback' => function($param) {
                        return is_numeric($param);
                    }
                )
            )
        ));
    }
    
    /**
     * Get events - Read-only endpoint for calendar
     */
    public function get_events($request) {
        if (!class_exists('TicketPayGo_Event')) {
            return new WP_Error('not_available', __('Event system not available.', 'ticketpaygo-lite'), array('status' => 500));
        }
        
        $format = $request->get_param('format');
        $category = $request->get_param('category');
        
        $args = array('status' => 'active');
        if ($category) {
            $args['category'] = sanitize_text_field($category);
        }
        
        $events = TicketPayGo_Event::get_all_events($args);
        
        // Format for calendar view
        if ($format === 'calendar') {
            $calendar_events = array();
            foreach ($events as $event_data) {
                $event = new TicketPayGo_Event($event_data['id']);
                $calendar_events[] = array(
                    'id' => $event_data['id'],
                    'title' => $event_data['title'],
                    'start_date' => $event_data['start_date'],
                    'end_date' => $event_data['end_date'],
                    'location' => $event_data['location'],
                    'price' => $event_data['price'],
                    'currency' => $event_data['currency'],
                    'description' => wp_strip_all_tags($event_data['description']),
                    'featured_image' => $event_data['featured_image'],
                    'tickets_remaining' => $event->get_remaining_tickets(),
                    'url' => home_url('/event/' . $event_data['id'] . '/')
                );
            }
            return rest_ensure_response($calendar_events);
        }
        
        return rest_ensure_response($events);
    }
    
    /**
     * Get single event - Read-only endpoint
     */
    public function get_event($request) {
        if (!class_exists('TicketPayGo_Event')) {
            return new WP_Error('not_available', __('Event system not available.', 'ticketpaygo-lite'), array('status' => 500));
        }
        
        $event_id = intval($request['id']);
        $event = new TicketPayGo_Event($event_id);
        
        if (!$event->exists()) {
            return new WP_Error('event_not_found', __('Event not found.', 'ticketpaygo-lite'), array('status' => 404));
        }
        
        return rest_ensure_response($event->get_data());
    }
}
