<?php
/**
 * Admin interface class
 */

if (!defined('ABSPATH')) {
    exit;
}

class TicketPayGo_Admin {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_init', array($this, 'init'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_ticketpaygo_admin_action', array($this, 'handle_ajax_actions'));
        add_action('wp_ajax_ticketpaygo_create_tables_ajax', array($this, 'handle_create_tables_ajax'));
        add_action('wp_ajax_ticketpaygo_get_order_details', array($this, 'ajax_get_order_details'));
        add_action('wp_ajax_ticketpaygo_update_order', array($this, 'ajax_update_order'));
        add_action('wp_ajax_ticketpaygo_resend_tickets', array($this, 'ajax_resend_tickets'));
    }
    
    /**
     * Initialize admin
     */
    public function init() {
        // Register settings
        register_setting('ticketpaygo_lite_settings', 'ticketpaygo_lite_settings', array($this, 'sanitize_settings'));
        
        // Add meta boxes for events
        add_action('add_meta_boxes', array($this, 'add_meta_boxes'));
        add_action('save_post', array($this, 'save_event_meta'));
        
        // Handle database setup if needed
        $this->handle_database_setup();
        
        // Always check and upgrade database schema for missing columns
        require_once TICKETPAYGO_PLUGIN_PATH . 'includes/class-database.php';
        TicketPayGo_Database::upgrade_database_schema();
    }
    
    /**
     * Handle database setup
     */
    public function handle_database_setup() {
        require_once TICKETPAYGO_PLUGIN_PATH . 'includes/class-database.php';
        
        // Check if we need to create tables
        if (isset($_GET['ticketpaygo_create_tables']) && current_user_can('manage_options')) {
            if (isset($_GET['_wpnonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'ticketpaygo_create_tables')) {
                $results = TicketPayGo_Database::create_tables();
                
                // Force a fresh check after creation attempt
                global $wpdb;
                $wpdb->flush();
                
                // Debug: Log table existence check
                $tables_exist = TicketPayGo_Database::tables_exist();
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Logging table creation for debugging
                error_log('TicketPayGo: Tables exist after creation: ' . ($tables_exist ? 'YES' : 'NO'));
                
                if ($tables_exist) {
                    // Set a flag to show success message
                    set_transient('ticketpaygo_tables_created', true, 30);
                } else {
                    // Set a flag to show error message
                    set_transient('ticketpaygo_tables_error', true, 30);
                }
                
                // Redirect to remove the URL parameter
                wp_safe_redirect(admin_url('admin.php?page=ticketpaygo-events'));
                exit;
            }
        }
        
        // Show success/error messages from transients
        if (get_transient('ticketpaygo_tables_created')) {
            delete_transient('ticketpaygo_tables_created');
            add_action('admin_notices', function() {
                echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Database tables created successfully!', 'ticketpaygo-lite') . '</p></div>';
            });
        }
        
        if (get_transient('ticketpaygo_tables_error')) {
            delete_transient('ticketpaygo_tables_error');
            add_action('admin_notices', function() {
                echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Failed to create database tables. Please check error logs.', 'ticketpaygo-lite') . '</p></div>';
            });
        }
        
        // Show notice if tables don't exist (only on TicketPayGo admin pages)
        if ($this->is_ticketpaygo_admin_page() && !TicketPayGo_Database::tables_exist()) {
            add_action('admin_notices', array($this, 'show_database_notice'));
        }
    }
    
    /**
     * Check if we're on a TicketPayGo admin page
     */
    private function is_ticketpaygo_admin_page() {
        $screen = get_current_screen();
        return $screen && (strpos($screen->id, 'ticketpaygo-lite') !== false || 
                         // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Admin page check, no sensitive action
                         (isset($_GET['page']) && sanitize_text_field(wp_unslash($_GET['page'])) === 'ticketpaygo-lite-events'));
        // phpcs:enable WordPress.Security.NonceVerification.Recommended
    }
    
    /**
     * Show database setup notice
     */
    public function show_database_notice() {
        $create_url = wp_nonce_url(
            admin_url('admin.php?page=ticketpaygo-events&ticketpaygo_create_tables=1'),
            'ticketpaygo_create_tables'
        );
        
        // Debug: Show which tables are missing
        global $wpdb;
        $missing_tables = array();
        $tables = array(
            $wpdb->prefix . 'ticketpaygo_lite_events',
            $wpdb->prefix . 'ticketpaygo_lite_orders', 
            $wpdb->prefix . 'ticketpaygo_lite_tickets',
            $wpdb->prefix . 'ticketpaygo_analytics',
            $wpdb->prefix . 'ticketpaygo_payment_logs',
            $wpdb->prefix . 'ticketpaygo_lite_settings'
        );
        
        foreach ($tables as $table) {
            if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table)) != $table) { // phpcs:ignore WordPress.DB.DirectDatabaseQuery
                $missing_tables[] = str_replace($wpdb->prefix, '', $table);
            }
        }
        
        echo '<div class="notice notice-warning is-dismissible">';
        echo '<p><strong>' . esc_html__('TicketPayGo Database Setup Required', 'ticketpaygo-lite') . '</strong></p>';
        echo '<p>' . esc_html__('The plugin database tables are missing. This may happen after plugin updates or server migrations.', 'ticketpaygo-lite') . '</p>';
        if (!empty($missing_tables)) {
            /* translators: %s: comma-separated list of missing database table names */
            echo '<p><small>' . sprintf(esc_html__('Missing tables: %s', 'ticketpaygo-lite'), esc_html(implode(', ', $missing_tables))) . '</small></p>';
        }
        echo '<p>';
        echo '<a href="' . esc_url($create_url) . '" class="button button-primary">' . esc_html__('Create Database Tables Now', 'ticketpaygo-lite') . '</a> ';
        echo '<button type="button" class="button" id="ticketpaygo-create-tables-ajax">' . esc_html__('Create Tables (AJAX)', 'ticketpaygo-lite') . '</button>';
        echo '</p>';
        echo '</div>';
        
        // Add inline script for immediate table creation
        $inline_script = "
        jQuery(document).ready(function($) {
            $('#ticketpaygo-create-tables-ajax').on('click', function(e) {
                e.preventDefault();
                var button = $(this);
                button.prop('disabled', true);
                button.text('Creating...');
                
                $.post(ajaxurl, {
                    action: 'ticketpaygo_create_tables_ajax',
                    nonce: '" . esc_js(wp_create_nonce('ticketpaygo_create_tables_ajax')) . "'
                }, function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert('Error: ' + response.data);
                        button.prop('disabled', false);
                        button.text('Create Tables (AJAX)');
                    }
                });
            });
        });
        ";
        wp_add_inline_script('ticketpaygo-admin-js', $inline_script);
        ?><?php
    }
    
    /**
     * Handle AJAX table creation
     */
    public function handle_create_tables_ajax() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'ticketpaygo_create_tables_ajax')) {
            wp_die('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        require_once TICKETPAYGO_PLUGIN_PATH . 'includes/class-database.php';
        
        try {
            $results = TicketPayGo_Database::create_tables();
            
            // Force a fresh check
            global $wpdb;
            $wpdb->flush();
            
            if (TicketPayGo_Database::tables_exist()) {
                wp_send_json_success('Database tables created successfully!');
            } else {
                wp_send_json_error('Failed to create all database tables. Check error logs for details.');
            }
        } catch (Exception $e) {
            wp_send_json_error('Error creating tables: ' . $e->getMessage());
        }
    }
    
    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_scripts($hook) {
        // Enqueue color picker for settings page
        if ($hook === 'ticketpaygo-lite_page_ticketpaygo-lite-settings') {
            wp_enqueue_style('wp-color-picker');
            wp_enqueue_script('wp-color-picker');
            wp_enqueue_media();
            
            wp_enqueue_script(
                'ticketpaygo-admin-settings',
                TICKETPAYGO_PLUGIN_URL . 'assets/js/admin-settings.js',
                array('jquery', 'wp-color-picker'),
                TICKETPAYGO_VERSION,
                true
            );
            
            wp_localize_script('ticketpaygo-admin-settings', 'ticketpaygo_settings', array(
                'strings' => array(
                    'upload_logo' => __('Upload Logo', 'ticketpaygo-lite')
                )
            ));
        }
        // Only load on our admin pages
        if (strpos($hook, 'ticketpaygo-lite') === false) {
            return;
        }

        // Enqueue WordPress editor for wp_editor() to work properly
        wp_enqueue_editor();
        wp_enqueue_media();
        
        // Enqueue dashicons
        wp_enqueue_style('dashicons');
        
        // Enqueue Leaflet for maps (bundled locally for WordPress.org compliance)
        wp_enqueue_style('leaflet-css', TICKETPAYGO_PLUGIN_URL . 'assets/css/leaflet.css', array(), '1.9.4');
        wp_enqueue_script('leaflet-js', TICKETPAYGO_PLUGIN_URL . 'assets/js/leaflet.js', array(), '1.9.4', true);
        
        // Enqueue styles
        wp_enqueue_style(
            'ticketpaygo-admin-css',
            TICKETPAYGO_PLUGIN_URL . 'assets/css/admin.css',
            array('dashicons', 'leaflet-css'),
            TICKETPAYGO_VERSION
        );

        // Enqueue scripts
        wp_enqueue_script('chart-js', TICKETPAYGO_PLUGIN_URL . 'assets/js/chart.min.js', array(), '4.4.1', true);
        wp_enqueue_script(
            'ticketpaygo-admin-js',
            TICKETPAYGO_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery', 'chart-js', 'leaflet-js'),
            TICKETPAYGO_VERSION,
            true
        );
        
        // Enqueue events admin script on events page
        if (strpos($hook, 'ticketpaygo-lite-events') !== false) {
            wp_enqueue_script('tpgl-events-admin', TICKETPAYGO_PLUGIN_URL . 'assets/js/events-admin.js', array('jquery'), TICKETPAYGO_VERSION, true);
            wp_enqueue_style('tpgl-events-admin-css', TICKETPAYGO_PLUGIN_URL . 'assets/css/events-admin.css', array(), TICKETPAYGO_VERSION);
        }
        
        // Enqueue orders page scripts
        if (strpos($hook, 'ticketpaygo-lite-orders') !== false) {
            wp_enqueue_script(
                'ticketpaygo-admin-orders', 
                TICKETPAYGO_PLUGIN_URL . 'assets/js/admin-orders.js', 
                array('jquery'), 
                TICKETPAYGO_VERSION, 
                true
            );
            
            wp_localize_script('ticketpaygo-admin-orders', 'ticketpaygo_orders', array(
                'strings' => array(
                    'view_details' => __('Order details for Order ID: ', 'ticketpaygo-lite'),
                    'edit_feature' => __('Edit order functionality coming soon for Order ID: ', 'ticketpaygo-lite'),
                    'confirm_resend' => __('Are you sure you want to resend tickets for this order?', 'ticketpaygo-lite'),
                    'tickets_resent' => __('Tickets resent for Order ID: ', 'ticketpaygo-lite'),
                    'confirm_refund' => __('Are you sure you want to refund this order? This action cannot be undone.', 'ticketpaygo-lite'),
                    'refund_initiated' => __('Refund initiated for Order ID: ', 'ticketpaygo-lite')
                )
            ));
        }

        // Enqueue checkin script on checkin page
        if (strpos($hook, 'ticketpaygo-lite-checkin') !== false) {
            wp_enqueue_script('tpgl-checkin', TICKETPAYGO_PLUGIN_URL . 'assets/js/checkin.js', array('jquery'), TICKETPAYGO_VERSION, true);
        }
        
        // Enqueue upgrade page styles
        if (strpos($hook, 'ticketpaygo-lite-upgrade') !== false) {
            wp_enqueue_style('tpgl-upgrade-css', TICKETPAYGO_PLUGIN_URL . 'assets/css/upgrade.css', array(), TICKETPAYGO_VERSION);
        }

        // Enqueue dashboard styles
        if ($hook === 'toplevel_page_ticketpaygo-lite') {
            wp_enqueue_style(
                'ticketpaygo-admin-dashboard',
                TICKETPAYGO_PLUGIN_URL . 'assets/css/admin-dashboard.css',
                array('ticketpaygo-admin-css'),
                TICKETPAYGO_VERSION
            );
        }
        
        // Get current user info
        $current_user = wp_get_current_user();
        
        // Localize script for AJAX and translations
        wp_localize_script('ticketpaygo-admin-js', 'ticketpaygo_admin', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('ticketpaygo_admin_nonce'),
            'currency_symbol' => $this->get_setting('currency_symbol', '€'),
            'currency_position' => $this->get_setting('currency_position', 'before'),
            'current_user' => array(
                'display_name' => $current_user->display_name,
                'email' => $current_user->user_email
            ),
            'strings' => array(
                'revenue' => __('Revenue', 'ticketpaygo-lite'),
                'orders' => __('Orders', 'ticketpaygo-lite'),
                'testing' => __('Testing...', 'ticketpaygo-lite'),
                'connection_failed' => __('Connection test failed', 'ticketpaygo-lite'),
                'sending' => __('Sending...', 'ticketpaygo-lite'),
                'send_failed' => __('Failed to send reminders', 'ticketpaygo-lite'),
                'scan_ticket' => __('Scan Ticket', 'ticketpaygo-lite'),
                'ticket_number' => __('Ticket Number', 'ticketpaygo-lite'),
                'scan' => __('Scan', 'ticketpaygo-lite'),
                'scan_failed' => __('Scan failed', 'ticketpaygo-lite'),
                'form_validation_failed' => __('Please fill in all required fields correctly', 'ticketpaygo-lite'),
                'confirm_delete' => __('Are you sure you want to delete this item?', 'ticketpaygo-lite'),
                'invalid_dates' => __('End date must be after start date', 'ticketpaygo-lite'),
                'copied' => __('Copied!', 'ticketpaygo-lite'),
                'shortcode_copied' => __('Shortcode copied to clipboard!', 'ticketpaygo-lite'),
                'new_ticket_type' => __('New Ticket Type', 'ticketpaygo-lite'),
                'remove' => __('Remove', 'ticketpaygo-lite'),
                'name' => __('Name', 'ticketpaygo-lite'),
                'price' => __('Price', 'ticketpaygo-lite'),
                'quantity' => __('Quantity', 'ticketpaygo-lite'),
                'description' => __('Description', 'ticketpaygo-lite'),
                'name_placeholder' => __('e.g., General Admission', 'ticketpaygo-lite'),
                'description_placeholder' => __('Optional description', 'ticketpaygo-lite')
            )
        ));
    }
    
    /**
     * Get setting value
     */
    private function get_setting($key, $default = '') {
        global $wpdb;
        
        // Check if settings table exists
        $table_name = $wpdb->prefix . 'ticketpaygo_lite_settings';
        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) != $table_name) { // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            return $default;
        }
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
        $value = $wpdb->get_var($wpdb->prepare(
            "SELECT setting_value FROM `{$wpdb->prefix}ticketpaygo_lite_settings` WHERE setting_key = %s",
            $key
        ));
        
        return $value ? $value : $default;
    }
    
    /**
     * Add meta boxes
     */
    public function add_meta_boxes() {
        // Check permissions
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Only add if we have the required files
        if (file_exists(TICKETPAYGO_PLUGIN_PATH . 'admin/meta-boxes/event-details.php')) {
            include TICKETPAYGO_PLUGIN_PATH . 'admin/meta-boxes/event-details.php';
        }
    }
    
    /**
     * Save event meta
     */
    public function save_event_meta($post_id) {
        if (isset($_POST['ticketpaygo_event_nonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['ticketpaygo_event_nonce'])), 'ticketpaygo_save_event')) {
            return;
        }
        
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        // Save event data
        $event_data = array(
            'start_date' => isset($_POST['start_date']) ? sanitize_text_field(wp_unslash($_POST['start_date'])) : '',
            'end_date' => isset($_POST['end_date']) ? sanitize_text_field(wp_unslash($_POST['end_date'])) : '',
            'location' => isset($_POST['location']) ? sanitize_text_field(wp_unslash($_POST['location'])) : '',
            'address' => isset($_POST['address']) ? sanitize_textarea_field(wp_unslash($_POST['address'])) : '',
            'price' => isset($_POST['price']) ? floatval(wp_unslash($_POST['price'])) : 0,
            'max_tickets' => isset($_POST['max_tickets']) ? absint(wp_unslash($_POST['max_tickets'])) : 0,
            'organizer_name' => isset($_POST['organizer_name']) ? sanitize_text_field(wp_unslash($_POST['organizer_name'])) : '',
            'organizer_email' => isset($_POST['organizer_email']) ? sanitize_email(wp_unslash($_POST['organizer_email'])) : ''
        );
        
        update_post_meta($post_id, '_ticketpaygo_event_data', $event_data);
    }
    
    /**
     * Handle AJAX actions
     */
    public function handle_ajax_actions() {
        check_ajax_referer('ticketpaygo_admin_nonce', 'nonce');
        
        $admin_action = isset($_POST['admin_action']) ? sanitize_text_field(wp_unslash($_POST['admin_action'])) : '';
        
        switch ($admin_action) {
            case 'get_event_stats':
                $this->ajax_get_event_stats();
                break;
            case 'export_data':
                $this->ajax_export_data();
                break;
            case 'send_reminder':
                $this->ajax_send_reminder();
                break;
            case 'scan_ticket':
                $this->ajax_scan_ticket();
                break;
            default:
                wp_send_json_error('Invalid action');
        }
    }
    
    /**
     * AJAX: Get event statistics (Lite version - basic stats only)
     */
    private function ajax_get_event_stats() {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified by check_ajax_referer in handle_ajax_actions
        if (empty($_POST['event_id'])) {
            wp_send_json_error('Event ID required');
            return;
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified by check_ajax_referer in handle_ajax_actions
        $event_id = absint(wp_unslash($_POST['event_id']));
        
        global $wpdb;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
        $tickets_sold = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM `{$wpdb->prefix}ticketpaygo_lite_tickets` WHERE event_id = %d",
            $event_id
        ));
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
        $total_revenue = $wpdb->get_var($wpdb->prepare(
            "SELECT SUM(total_amount) FROM `{$wpdb->prefix}ticketpaygo_lite_orders` WHERE event_id = %d AND status = 'completed'",
            $event_id
        ));
        
        $stats = array(
            'tickets_sold' => $tickets_sold,
            'total_revenue' => $total_revenue
        );
        
        wp_send_json_success($stats);
    }
    
    /**
     * AJAX: Export data (Lite version - basic export)
     */
    private function ajax_export_data() {
        // Advanced export available in Full version
        wp_send_json_error(array('message' => __('Advanced export available in TicketPayGo Full', 'ticketpaygo-lite')));
    }
    
    /**
     * AJAX: Send reminder email
     */
    private function ajax_send_reminder() {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified by check_ajax_referer in handle_ajax_actions
        if (empty($_POST['event_id'])) {
            wp_send_json_error('Event ID required');
            return;
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified by check_ajax_referer in handle_ajax_actions
        $event_id = absint(wp_unslash($_POST['event_id']));
        
        $email = new TicketPayGo_Email();
        $sent_count = $email->send_event_reminder($event_id);
        
        /* translators: %d: number of customers who received the reminder */
        wp_send_json_success(array(
            /* translators: %d: number of customers who received the reminder */
            'message' => sprintf(esc_html__('Reminder sent to %d customers', 'ticketpaygo-lite'), intval($sent_count))
        ));
    }
    
    /**
     * AJAX: Scan ticket
     */
    private function ajax_scan_ticket() {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified by check_ajax_referer in handle_ajax_actions
        $ticket_number = isset($_POST['ticket_number']) ? sanitize_text_field(wp_unslash($_POST['ticket_number'])) : '';
        
        // Find ticket by number
        global $wpdb;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
        $ticket_id = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM `{$wpdb->prefix}ticketpaygo_lite_tickets` WHERE ticket_number = %s",
            $ticket_number
        ));
        
        if (!$ticket_id) {
            wp_send_json_error('Ticket not found');
        }
        
        $ticket = new TicketPayGo_Ticket($ticket_id);
        $result = $ticket->scan(get_current_user_id());
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result['message']);
        }
    }
    
    /**
     * AJAX: Get order details
     */
    public function ajax_get_order_details() {
        check_ajax_referer('ticketpaygo_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        $order_id = isset($_POST['order_id']) ? absint($_POST['order_id']) : 0;
        if (!$order_id) {
            wp_send_json_error('Invalid order ID');
        }
        $order = new TicketPayGo_Order($order_id);
        if (!$order->exists()) {
            wp_send_json_error('Order not found');
        }
        $tickets = $order->get_tickets();
        wp_send_json_success(array(
            'order'   => $order->get_data(),
            'tickets' => $tickets,
        ));
    }

    /**
     * AJAX: Update order status and notes
     */
    public function ajax_update_order() {
        check_ajax_referer('ticketpaygo_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        $order_id = isset($_POST['order_id']) ? absint($_POST['order_id']) : 0;
        $status   = isset($_POST['status']) ? sanitize_text_field(wp_unslash($_POST['status'])) : '';
        $notes    = isset($_POST['notes'])  ? sanitize_textarea_field(wp_unslash($_POST['notes'])) : '';
        if (!$order_id || !in_array($status, array('pending','completed','cancelled','refunded'), true)) {
            wp_send_json_error('Invalid data');
        }
        global $wpdb;
        $orders_table = $wpdb->prefix . 'ticketpaygo_lite_orders';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        $result = $wpdb->update(
            $orders_table,
            array('status' => $status, 'notes' => $notes),
            array('id' => $order_id),
            array('%s', '%s'),
            array('%d')
        );
        wp_cache_delete('tpg_order_' . $order_id, 'ticketpaygo');
        if ($result !== false) {
            wp_send_json_success('Order updated');
        } else {
            wp_send_json_error('DB error: ' . $wpdb->last_error);
        }
    }

    /**
     * AJAX: Resend confirmation email with tickets
     */
    public function ajax_resend_tickets() {
        check_ajax_referer('ticketpaygo_resend_tickets', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        $order_id = isset($_POST['order_id']) ? absint($_POST['order_id']) : 0;
        if (!$order_id) {
            wp_send_json_error('Invalid order ID');
        }
        $order = new TicketPayGo_Order($order_id);
        if (!$order->exists()) {
            wp_send_json_error('Order not found');
        }
        $email = new TicketPayGo_Email();
        $sent  = $email->send_order_confirmation($order);
        if ($sent) {
            wp_send_json_success('Tickets resent successfully');
        } else {
            wp_send_json_error('Failed to send email');
        }
    }

    /**
     * Get admin menu capability
     */
    public function get_menu_capability() {
        return apply_filters('ticketpaygo_admin_capability', 'manage_options');
    }
    
    /**
     * Render admin notices
     */
    public function render_admin_notices() {
        // Check for missing payment configuration
        $mollie_key = TicketPayGo_Database::get_setting('mollie_api_key');
        $paypal_id = TicketPayGo_Database::get_setting('paypal_client_id');
        $stripe_key = TicketPayGo_Database::get_setting('stripe_secret_key');
        
        if (empty($mollie_key) && empty($paypal_id) && empty($stripe_key)) {
            echo '<div class="notice notice-warning is-dismissible">';
            echo '<p>' . esc_html__('TicketPayGo: Please configure at least one payment method in the settings.', 'ticketpaygo-lite') . '</p>';
            echo '</div>';
        }
        
        // Check for events without tickets sold
        $events_without_sales = $this->get_events_without_sales();
        if (!empty($events_without_sales)) {
            echo '<div class="notice notice-info is-dismissible">';
            /* translators: %d: number of events without ticket sales */
            echo '<p>' . sprintf(esc_html__('You have %d events with no ticket sales yet. Consider promoting them!', 'ticketpaygo-lite'), intval(count($events_without_sales))) . '</p>';
            echo '</div>';
        }
    }
    
    /**
     * Get events without sales
     */
    private function get_events_without_sales() {
        global $wpdb;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
        return $wpdb->get_results(
            "SELECT 
                e.id,
                e.title,
                e.start_date,
                COUNT(DISTINCT o.id) as total_orders,
                SUM(o.quantity) as tickets_sold,
                SUM(o.total_amount) as revenue
            FROM `{$wpdb->prefix}ticketpaygo_lite_events` e
            LEFT JOIN `{$wpdb->prefix}ticketpaygo_lite_orders` o ON e.id = o.event_id AND o.status = 'completed'
             WHERE e.status = 'active' 
             AND o.id IS NULL
             AND e.start_date > NOW()",
            ARRAY_A
        );
        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
    }
    
    /**
     * Add admin bar menu (Lite version - simplified)
     */
    public function add_admin_bar_menu($wp_admin_bar) {
        if (!current_user_can($this->get_menu_capability())) {
            return;
        }
        
        // Get basic stats for Lite version
        global $wpdb;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
        $today_orders = $wpdb->get_var("SELECT COUNT(*) FROM `{$wpdb->prefix}ticketpaygo_lite_orders` WHERE status = 'completed' AND DATE(created_at) = CURDATE()");
        
        $wp_admin_bar->add_menu(array(
            'id' => 'ticketpaygo-lite',
            'title' => 'TicketPayGo Lite (' . intval($today_orders) . ')',
            'href' => admin_url('admin.php?page=ticketpaygo-lite')
        ));
        
        $wp_admin_bar->add_menu(array(
            'parent' => 'ticketpaygo-lite',
            'id' => 'ticketpaygo-lite-events',
            'title' => __('Events', 'ticketpaygo-lite'),
            'href' => admin_url('admin.php?page=ticketpaygo-lite-events')
        ));
        
        $wp_admin_bar->add_menu(array(
            'parent' => 'ticketpaygo-lite',
            'id' => 'ticketpaygo-lite-orders',
            'title' => __('Orders', 'ticketpaygo-lite'),
            'href' => admin_url('admin.php?page=ticketpaygo-lite-orders')
        ));
    }
    
    /**
     * Register custom post types
     */
    public function register_post_types() {
        // Register event post type for WordPress integration
        register_post_type('ticketpaygo_event', array(
            'labels' => array(
                'name' => __('Events', 'ticketpaygo-lite'),
                'singular_name' => __('Event', 'ticketpaygo-lite'),
                'add_new' => __('Add New Event', 'ticketpaygo-lite'),
                'add_new_item' => __('Add New Event', 'ticketpaygo-lite'),
                'edit_item' => __('Edit Event', 'ticketpaygo-lite'),
                'new_item' => __('New Event', 'ticketpaygo-lite'),
                'view_item' => __('View Event', 'ticketpaygo-lite'),
                'search_items' => __('Search Events', 'ticketpaygo-lite'),
                'not_found' => __('No events found', 'ticketpaygo-lite'),
                'not_found_in_trash' => __('No events found in trash', 'ticketpaygo-lite')
            ),
            'public' => true,
            'has_archive' => true,
            'supports' => array('title', 'editor', 'thumbnail', 'excerpt'),
            'menu_icon' => 'dashicons-tickets-alt',
            'rewrite' => array('slug' => 'events')
        ));
    }
    
    /**
     * Add custom columns to events list
     */
    public function add_event_columns($columns) {
        $new_columns = array();
        
        foreach ($columns as $key => $value) {
            $new_columns[$key] = $value;
            
            if ($key === 'title') {
                $new_columns['event_date'] = __('Event Date', 'ticketpaygo-lite');
                $new_columns['tickets_sold'] = __('Tickets Sold', 'ticketpaygo-lite');
                $new_columns['revenue'] = __('Revenue', 'ticketpaygo-lite');
            }
        }
        
        return $new_columns;
    }
    
    /**
     * Populate custom columns
     */
    public function populate_event_columns($column, $post_id) {
        $event_data = get_post_meta($post_id, '_ticketpaygo_event_data', true);
        
        switch ($column) {
            case 'event_date':
                if (isset($event_data['start_date'])) {
                    echo esc_html(gmdate('M j, Y g:i A', strtotime($event_data['start_date'])));
                }
                break;
            case 'tickets_sold':
                // Get tickets sold from database
                global $wpdb;
                
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                $tickets_sold = $wpdb->get_var($wpdb->prepare(
                "SELECT SUM(quantity) FROM `{$wpdb->prefix}ticketpaygo_lite_orders` 
                 WHERE event_id = %d AND status = 'completed'",
                    $post_id
                ));
                
                echo intval($tickets_sold);
                break;
            case 'revenue':
                // Get revenue from database
                global $wpdb;
                
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                $revenue = $wpdb->get_var($wpdb->prepare(
                "SELECT SUM(total_amount) FROM `{$wpdb->prefix}ticketpaygo_lite_orders` 
                 WHERE event_id = %d AND status = 'completed'",
                    $post_id
                ));
                
                $symbol = TicketPayGo_Database::get_setting('currency_symbol', '€');
                echo esc_html($symbol . number_format(floatval($revenue), 2));
                break;
        }
    }
    
    public function events_page() {
        include TICKETPAYGO_PLUGIN_PATH . 'admin/events.php';
    }
    
    public function orders_page() {
        include TICKETPAYGO_PLUGIN_PATH . 'admin/orders.php';
    }
    
    public function analytics_page() {
        include TICKETPAYGO_PLUGIN_PATH . 'admin/analytics.php';
    }
    
    public function customization_page() {
        include TICKETPAYGO_PLUGIN_PATH . 'admin/customization.php';
    }
}
