/**
 * TicketPayGo Frontend JavaScript
 */

// Global function to open event modal
window.openEventModal = function(eventId) {
    var modal = document.getElementById('modal-' + eventId);
    if (modal) {
        modal.style.display = 'flex';
        document.body.style.overflow = 'hidden'; // Prevent background scroll
    }
};

// Global function to close event modal
window.closeEventModal = function(eventId) {
    var modal = document.getElementById('modal-' + eventId);
    if (modal) {
        modal.style.display = 'none';
        document.body.style.overflow = ''; // Restore scroll
    }
};

// Close modal on Escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        var modals = document.querySelectorAll('.ticketpaygo-modal');
        modals.forEach(function(modal) {
            if (modal.style.display === 'flex') {
                modal.style.display = 'none';
                document.body.style.overflow = '';
            }
        });
    }
});

// Ticket quantity management - SINGLE handler for all qty buttons
document.addEventListener('click', function(e) {
    // Handle plus button
    if (e.target.classList.contains('qty-plus')) {
        e.preventDefault();
        e.stopImmediatePropagation();
        
        var input = e.target.parentElement.querySelector('.qty-input');
        if (!input) return;
        
        var max = parseInt(input.getAttribute('max')) || 999;
        var current = parseInt(input.value) || 0;
        console.log('QTY+ clicked: current=' + current + ', max=' + max);
        if (current < max) {
            input.value = current + 1;
            updateTicketTotal(input);
        }
        return false;
    }
    
    // Handle minus button
    if (e.target.classList.contains('qty-minus')) {
        e.preventDefault();
        e.stopImmediatePropagation();
        
        var input = e.target.parentElement.querySelector('.qty-input');
        if (!input) return;
        
        var min = parseInt(input.getAttribute('min')) || 0;
        var current = parseInt(input.value) || 0;
        console.log('QTY- clicked: current=' + current + ', min=' + min);
        if (current > min) {
            input.value = current - 1;
            updateTicketTotal(input);
        }
        return false;
    }
}, true); // Use capture phase to run first

// Handle manual input change
document.addEventListener('input', function(e) {
    if (e.target.classList.contains('qty-input')) {
        var max = parseInt(e.target.getAttribute('max')) || 999;
        var value = parseInt(e.target.value) || 0;
        if (value < 0) e.target.value = 0;
        if (value > max) e.target.value = max;
        updateTicketTotal(e.target);
    }
});

// Calculate and update total
function updateTicketTotal(input) {
    var modal = input.closest('.ticketpaygo-modal');
    if (!modal) return;
    
    var totalAmount = 0;
    var totalQuantity = 0;
    var inputs = modal.querySelectorAll('.qty-input');
    
    inputs.forEach(function(inp) {
        var qty = parseInt(inp.value) || 0;
        var price = parseFloat(inp.getAttribute('data-price')) || 0;
        totalAmount += qty * price;
        totalQuantity += qty;
    });
    
    // Update total display
    var totalElement = modal.querySelector('.total-amount');
    if (totalElement) {
        var currencySymbol = totalElement.getAttribute('data-currency-symbol') || '€';
        var currencyPosition = totalElement.getAttribute('data-currency-position') || 'before';
        var formattedTotal = totalAmount.toFixed(2);
        
        if (currencyPosition === 'before') {
            totalElement.textContent = currencySymbol + ' ' + formattedTotal;
        } else {
            totalElement.textContent = formattedTotal + ' ' + currencySymbol;
        }
    }
    
    // Update hidden quantity field for direct purchase
    var quantityField = modal.querySelector('.purchase-quantity');
    if (quantityField) {
        quantityField.value = totalQuantity;
    }
}

// Legacy function for backward compatibility
window.toggleEventDetails = function(eventId) {
    console.log('Toggle event details for ID:', eventId);
    
    // Find the card container
    var card = document.getElementById('ticketpaygo-event-' + eventId) || document.getElementById('event-list-' + eventId);
    
    // Find the details section
    var details = document.getElementById('details-' + eventId) || document.getElementById('details-list-' + eventId);
    
    if (!card || !details) {
        console.error('Card or details not found for ID:', eventId);
        return;
    }
    
    // Check if currently expanded
    var isExpanded = details.style.display !== 'none' && details.style.display !== '';
    
    if (isExpanded) {
        // Collapse
        details.style.display = 'none';
        card.classList.remove('event-card-expanded');
        
        // Update expand indicator rotation
        var indicator = card.querySelector('.expand-indicator svg');
        if (indicator) {
            indicator.style.transform = 'rotate(0deg)';
        }
    } else {
        // Expand
        details.style.display = 'block';
        card.classList.add('event-card-expanded');
        
        // Update expand indicator rotation
        var indicator = card.querySelector('.expand-indicator svg');
        if (indicator) {
            indicator.style.transform = 'rotate(180deg)';
        }
        
        // Smooth scroll to show expanded content
        setTimeout(function() {
            details.scrollIntoView({ 
                behavior: 'smooth', 
                block: 'nearest' 
            });
        }, 100);
    }
};

(function($) {
    'use strict';

    // Frontend object
    const TicketPayGoFrontend = {
        
        init: function() {
            this.initBookingForm();
            this.initTicketVerification();
            this.initEventList();
            this.bindEvents();
        },

        initBookingForm: function() {
            const bookingForm = $('.ticketpaygo-booking-form');
            if (!bookingForm.length) return;

            // Initialize quantity selectors
            this.initQuantitySelectors();
            
            // Initialize price calculation
            this.updatePriceSummary();
            
            // Initialize payment methods
            this.initPaymentMethods();
            
            // Form validation
            this.initFormValidation();
        },

        initQuantitySelectors: function() {
            // Standard quantity buttons
            $('.quantity-btn').on('click', function(e) {
                e.preventDefault();
                const isIncrease = $(this).hasClass('increase');
                const input = $(this).siblings('.quantity-input');
                const currentValue = parseInt(input.val()) || 0;
                const maxTickets = parseInt(input.attr('max')) || 999;
                const minTickets = parseInt(input.attr('min')) || 1;

                let newValue = currentValue;
                if (isIncrease && currentValue < maxTickets) {
                    newValue = currentValue + 1;
                } else if (!isIncrease && currentValue > minTickets) {
                    newValue = currentValue - 1;
                }

                input.val(newValue);
                TicketPayGoFrontend.updatePriceSummary();
            });

            $('.quantity-input').on('input', function() {
                const value = parseInt($(this).val()) || 0;
                const max = parseInt($(this).attr('max')) || 999;
                const min = parseInt($(this).attr('min')) || 1;

                if (value > max) $(this).val(max);
                if (value < min) $(this).val(min);

                TicketPayGoFrontend.updatePriceSummary();
            });
            
            // Multiple ticket types quantity buttons (modal) - REMOVED: handled by vanilla JS at top of file
            
            // Multiple ticket types input change
            $(document).on('input change', '.ticket-type-qty-input, .qty-input', function() {
                const value = parseInt($(this).val()) || 0;
                const max = parseInt($(this).attr('max')) || 10;
                const min = parseInt($(this).attr('min')) || 0;

                if (value > max) $(this).val(max);
                if (value < min) $(this).val(min);

                TicketPayGoFrontend.updateModalPriceSummary($(this).closest('.modal-form'));
            });
        },
        
        updateModalPriceSummary: function($form) {
            if (!$form || !$form.length) return;
            
            let subtotal = 0;
            let totalQuantity = 0;
            const currencySymbol = $form.find('.subtotal').text().replace(/[\d.,]/g, '').trim() || '€';
            
            // Check for multiple ticket types
            const ticketTypeInputs = $form.find('.ticket-type-qty-input');
            if (ticketTypeInputs.length > 0) {
                // Multiple ticket types
                ticketTypeInputs.each(function() {
                    const qty = parseInt($(this).val()) || 0;
                    const price = parseFloat($(this).data('price')) || 0;
                    subtotal += qty * price;
                    totalQuantity += qty;
                });
                
                // Update hidden total quantity field
                $form.find('input[name="quantity"]').val(totalQuantity);
            } else {
                // Single ticket type
                const qtyInput = $form.find('.qty-input');
                const qty = parseInt(qtyInput.val()) || 0;
                const price = parseFloat(qtyInput.data('price')) || 0;
                subtotal = qty * price;
                totalQuantity = qty;
            }
            
            const processingFee = subtotal * 0.029; // 2.9% processing fee
            const total = subtotal + processingFee;
            
            // Update display
            $form.find('.subtotal').text(currencySymbol + subtotal.toFixed(2));
            $form.find('.processing-fee').text(currencySymbol + processingFee.toFixed(2));
            $form.find('.total-amount').text(currencySymbol + total.toFixed(2));
            
            // Enable/disable continue button based on quantity
            const continueBtn = $form.find('.btn-next-step').first();
            if (totalQuantity > 0) {
                continueBtn.prop('disabled', false).removeClass('disabled');
            } else {
                continueBtn.prop('disabled', true).addClass('disabled');
            }
        },

        updatePriceSummary: function() {
            const quantity = parseInt($('.quantity-input').val()) || 0;
            const price = parseFloat($('.quantity-input').data('price')) || 0;
            const subtotal = quantity * price;
            const fees = subtotal * 0.029; // 2.9% processing fee
            const total = subtotal + fees;

            $('.price-subtotal .amount').text(this.formatCurrency(subtotal));
            $('.price-fees .amount').text(this.formatCurrency(fees));
            $('.price-total .amount').text(this.formatCurrency(total));

            // Update submit button
            const submitBtn = $('.submit-booking');
            if (quantity > 0) {
                submitBtn.prop('disabled', false);
                submitBtn.text(ticketpaygo_frontend.strings.book_tickets.replace('%d', quantity));
            } else {
                submitBtn.prop('disabled', true);
                submitBtn.text(ticketpaygo_frontend.strings.select_tickets);
            }
        },

        initPaymentMethods: function() {
            $('.payment-method input[type="radio"]').on('change', function() {
                $('.payment-method').removeClass('selected');
                $(this).closest('.payment-method').addClass('selected');
            });

            // Select first payment method by default
            $('.payment-method input[type="radio"]:first').prop('checked', true).trigger('change');
        },

        initFormValidation: function() {
            $('.ticketpaygo-booking-form').on('submit', function(e) {
                e.preventDefault();
                return TicketPayGoFrontend.submitBooking($(this));
            });

            // Real-time validation
            $('input[required], select[required]').on('blur', function() {
                TicketPayGoFrontend.validateField($(this));
            });

            $('input[type="email"]').on('blur', function() {
                if ($(this).val() && !TicketPayGoFrontend.isValidEmail($(this).val())) {
                    TicketPayGoFrontend.showFieldError($(this), ticketpaygo_frontend.strings.invalid_email);
                } else {
                    TicketPayGoFrontend.clearFieldError($(this));
                }
            });
        },

        submitBooking: function($form) {
            // Validate form
            if (!this.validateBookingForm($form)) {
                return false;
            }

            // Show loading state
            $form.addClass('booking-form-loading');
            $('.submit-booking').prop('disabled', true);

            // Prepare form data
            const formData = new FormData($form[0]);
            formData.append('action', 'ticketpaygo_purchase_ticket');
            formData.append('nonce', ticketpaygo_frontend.nonce);

            // Submit via AJAX
            $.ajax({
                url: ticketpaygo_frontend.ajax_url,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    if (response.success) {
                        if (response.data.redirect_url) {
                            // Redirect to payment gateway
                            window.location.href = response.data.redirect_url;
                        } else {
                            // Show success message
                            TicketPayGoFrontend.showMessage('success', response.data.message);
                        }
                    } else {
                        TicketPayGoFrontend.showMessage('error', response.data);
                    }
                },
                error: function() {
                    TicketPayGoFrontend.showMessage('error', ticketpaygo_frontend.strings.booking_failed);
                },
                complete: function() {
                    $form.removeClass('booking-form-loading');
                    $('.submit-booking').prop('disabled', false);
                }
            });

            return false;
        },

        validateBookingForm: function($form) {
            let isValid = true;

            // Check required fields
            $form.find('[required]').each(function() {
                if (!TicketPayGoFrontend.validateField($(this))) {
                    isValid = false;
                }
            });

            // Check email fields
            $form.find('input[type="email"]').each(function() {
                if ($(this).val() && !TicketPayGoFrontend.isValidEmail($(this).val())) {
                    TicketPayGoFrontend.showFieldError($(this), ticketpaygo_frontend.strings.invalid_email);
                    isValid = false;
                }
            });

            // Check quantity
            const quantity = parseInt($('.quantity-input').val()) || 0;
            if (quantity < 1) {
                TicketPayGoFrontend.showMessage('error', ticketpaygo_frontend.strings.select_quantity);
                isValid = false;
            }

            // Check payment method
            if (!$('.payment-method input[type="radio"]:checked').length) {
                TicketPayGoFrontend.showMessage('error', ticketpaygo_frontend.strings.select_payment);
                isValid = false;
            }

            return isValid;
        },

        validateField: function($field) {
            const value = $field.val().trim();
            const isRequired = $field.prop('required');

            if (isRequired && !value) {
                this.showFieldError($field, ticketpaygo_frontend.strings.field_required);
                return false;
            } else {
                this.clearFieldError($field);
                return true;
            }
        },

        showFieldError: function($field, message) {
            $field.addClass('error');
            
            let errorDiv = $field.siblings('.error-message');
            if (!errorDiv.length) {
                errorDiv = $('<div class="error-message"></div>');
                $field.after(errorDiv);
            }
            errorDiv.text(message);
        },

        clearFieldError: function($field) {
            $field.removeClass('error');
            $field.siblings('.error-message').remove();
        },

        initTicketVerification: function() {
            const verificationForm = $('.ticketpaygo-verification');
            if (!verificationForm.length) return;

            $('.verify-btn').on('click', function(e) {
                e.preventDefault();
                const ticketNumber = $('.verification-input input').val().trim();
                if (ticketNumber) {
                    TicketPayGoFrontend.verifyTicket(ticketNumber);
                }
            });

            $('.verification-input input').on('keypress', function(e) {
                if (e.which === 13) { // Enter key
                    e.preventDefault();
                    $('.verify-btn').click();
                }
            });

            // QR Code scanner (if available)
            if (typeof Html5Qrcode !== 'undefined') {
                this.initQRScanner();
            }
        },

        verifyTicket: function(ticketNumber) {
            const resultDiv = $('.verification-result');
            const verifyBtn = $('.verify-btn');
            
            // Show loading state
            verifyBtn.prop('disabled', true).text(ticketpaygo_frontend.strings.verifying);
            resultDiv.removeClass('success error warning').hide();

            $.ajax({
                url: ticketpaygo_frontend.ajax_url,
                type: 'POST',
                data: {
                    action: 'ticketpaygo_verify_ticket',
                    ticket_number: ticketNumber,
                    nonce: ticketpaygo_frontend.nonce
                },
                success: function(response) {
                    if (response.success) {
                        const ticket = response.data;
                        let resultClass = 'success';
                        let message = ticketpaygo_frontend.strings.ticket_valid;

                        if (ticket.status === 'used') {
                            resultClass = 'warning';
                            message = ticketpaygo_frontend.strings.ticket_already_used;
                        } else if (ticket.status === 'expired') {
                            resultClass = 'error';
                            message = ticketpaygo_frontend.strings.ticket_expired;
                        }

                        resultDiv.removeClass('success error warning')
                                 .addClass(resultClass)
                                 .html(TicketPayGoFrontend.formatTicketInfo(ticket, message))
                                 .show();
                    } else {
                        resultDiv.removeClass('success warning')
                                 .addClass('error')
                                 .html('<strong>' + ticketpaygo_frontend.strings.ticket_invalid + '</strong><br>' + response.data)
                                 .show();
                    }
                },
                error: function() {
                    resultDiv.removeClass('success warning')
                             .addClass('error')
                             .html('<strong>' + ticketpaygo_frontend.strings.verification_failed + '</strong>')
                             .show();
                },
                complete: function() {
                    verifyBtn.prop('disabled', false).text(ticketpaygo_frontend.strings.verify);
                }
            });
        },

        formatTicketInfo: function(ticket, message) {
            return `
                <strong>${message}</strong><br>
                <strong>${ticketpaygo_frontend.strings.event}:</strong> ${ticket.event_title}<br>
                <strong>${ticketpaygo_frontend.strings.date}:</strong> ${ticket.event_date}<br>
                <strong>${ticketpaygo_frontend.strings.holder}:</strong> ${ticket.holder_name}<br>
                <strong>${ticketpaygo_frontend.strings.ticket_number}:</strong> ${ticket.ticket_number}
            `;
        },

        initQRScanner: function() {
            const scannerDiv = $('<div id="qr-scanner" style="margin: 20px 0;"></div>');
            const scannerBtn = $('<button type="button" class="verify-btn" id="start-scanner">' + ticketpaygo_frontend.strings.scan_qr + '</button>');
            
            $('.verification-form').append(scannerBtn).append(scannerDiv);

            let html5QrCode = null;

            scannerBtn.on('click', function() {
                if (html5QrCode && html5QrCode.getState() === Html5QrcodeScannerState.SCANNING) {
                    // Stop scanning
                    html5QrCode.stop().then(() => {
                        scannerBtn.text(ticketpaygo_frontend.strings.scan_qr);
                        scannerDiv.hide();
                    });
                } else {
                    // Start scanning
                    html5QrCode = new Html5Qrcode("qr-scanner");
                    scannerDiv.show();
                    scannerBtn.text(ticketpaygo_frontend.strings.stop_scan);

                    html5QrCode.start(
                        { facingMode: "environment" },
                        {
                            fps: 10,
                            qrbox: { width: 250, height: 250 }
                        },
                        (decodedText) => {
                            // Success callback
                            $('.verification-input input').val(decodedText);
                            TicketPayGoFrontend.verifyTicket(decodedText);
                            
                            // Stop scanning
                            html5QrCode.stop();
                            scannerBtn.text(ticketpaygo_frontend.strings.scan_qr);
                            scannerDiv.hide();
                        },
                        (errorMessage) => {
                            // Error callback (can be ignored for continuous scanning)
                        }
                    ).catch(err => {
                        console.error('QR Scanner failed to start:', err);
                        TicketPayGoFrontend.showMessage('error', ticketpaygo_frontend.strings.scanner_failed);
                    });
                }
            });
        },

        initEventList: function() {
            // Filter events
            $('.event-filter').on('change', function() {
                TicketPayGoFrontend.filterEvents();
            });

            // Search events
            $('.event-search').on('input', function() {
                clearTimeout(TicketPayGoFrontend.searchTimeout);
                TicketPayGoFrontend.searchTimeout = setTimeout(() => {
                    TicketPayGoFrontend.filterEvents();
                }, 300);
            });

            // Load more events (pagination)
            $('.load-more-events').on('click', function(e) {
                e.preventDefault();
                TicketPayGoFrontend.loadMoreEvents();
            });
        },

        filterEvents: function() {
            const category = $('.event-category-filter').val();
            const date = $('.event-date-filter').val();
            const search = $('.event-search').val();
            const location = $('.event-location-filter').val();

            $.ajax({
                url: ticketpaygo_frontend.ajax_url,
                type: 'POST',
                data: {
                    action: 'ticketpaygo_filter_events',
                    category: category,
                    date: date,
                    search: search,
                    location: location,
                    nonce: ticketpaygo_frontend.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $('.ticketpaygo-events-list').html(response.data.html);
                    }
                },
                error: function() {
                    TicketPayGoFrontend.showMessage('error', ticketpaygo_frontend.strings.filter_failed);
                }
            });
        },

        loadMoreEvents: function() {
            const button = $('.load-more-events');
            const page = parseInt(button.data('page')) || 1;
            const nextPage = page + 1;

            button.prop('disabled', true).text(ticketpaygo_frontend.strings.loading);

            $.ajax({
                url: ticketpaygo_frontend.ajax_url,
                type: 'POST',
                data: {
                    action: 'ticketpaygo_load_events',
                    page: nextPage,
                    nonce: ticketpaygo_frontend.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $('.ticketpaygo-events-list').append(response.data.html);
                        
                        if (response.data.has_more) {
                            button.data('page', nextPage);
                        } else {
                            button.hide();
                        }
                    }
                },
                error: function() {
                    TicketPayGoFrontend.showMessage('error', ticketpaygo_frontend.strings.load_failed);
                },
                complete: function() {
                    button.prop('disabled', false).text(ticketpaygo_frontend.strings.load_more);
                }
            });
        },

        bindEvents: function() {
            // Handle payment return
            const urlParams = new URLSearchParams(window.location.search);
            if (urlParams.get('payment_status')) {
                this.handlePaymentReturn(urlParams);
            }

            // Modal functionality
            this.initModalEvents();
            
            // Handle direct purchase form submission
            $(document).on('submit', '.direct-purchase-form', function(e) {
                e.preventDefault();
                console.log('TicketPayGo: Direct purchase form submitted');
                
                const form = $(this);
                const submitBtn = form.find('button[type="submit"]');
                const quantity = parseInt(form.find('.purchase-quantity').val()) || 0;
                
                console.log('TicketPayGo: Quantity:', quantity);
                console.log('TicketPayGo: Form data:', form.serialize());
                
                // Validate quantity
                if (quantity < 1) {
                    alert(ticketpaygo_frontend.strings.select_quantity || 'Please select at least 1 ticket');
                    return false;
                }
                
                // Show loading state
                submitBtn.prop('disabled', true).text(ticketpaygo_frontend.strings.loading || 'Processing...');
                console.log('TicketPayGo: Submitting AJAX request...');
                
                // Submit via AJAX
                $.ajax({
                    url: ticketpaygo_frontend.ajax_url,
                    type: 'POST',
                    data: form.serialize(),
                    success: function(response) {
                        console.log('TicketPayGo: AJAX response:', response);
                        if (response.success && response.data.redirect_url) {
                            console.log('TicketPayGo: Redirecting to:', response.data.redirect_url);
                            // Redirect to payment gateway
                            window.location.href = response.data.redirect_url;
                        } else {
                            console.error('TicketPayGo: Purchase failed:', response);
                            alert(response.data.message || response.data || 'Purchase failed');
                            submitBtn.prop('disabled', false).text(ticketpaygo_frontend.strings.buy_tickets || 'Buy Tickets');
                        }
                    },
                    error: function(xhr, status, error) {
                        console.error('TicketPayGo: AJAX error:', status, error);
                        console.error('TicketPayGo: Response:', xhr.responseText);
                        alert(ticketpaygo_frontend.strings.booking_failed || 'Booking failed. Please try again.');
                        submitBtn.prop('disabled', false).text(ticketpaygo_frontend.strings.buy_tickets || 'Buy Tickets');
                    }
                });
                
                return false;
            });

            // Print ticket
            $('.print-ticket').on('click', function(e) {
                e.preventDefault();
                window.print();
            });

            // Download ticket
            $('.download-ticket').on('click', function(e) {
                e.preventDefault();
                const ticketId = $(this).data('ticket-id');
                TicketPayGoFrontend.downloadTicket(ticketId);
            });

            // Share event
            $('.share-event').on('click', function(e) {
                e.preventDefault();
                const eventUrl = $(this).data('event-url');
                const eventTitle = $(this).data('event-title');
                TicketPayGoFrontend.shareEvent(eventUrl, eventTitle);
            });
        },

        initModalEvents: function() {
            // Buy button clicks (both old and new button classes)
            $(document).on('click', '.ticketpaygo-buy-btn, .ticketpaygo-book-now', function(e) {
                e.preventDefault();
                const eventId = $(this).data('event-id');
                TicketPayGoFrontend.openBookingModal(eventId);
            });

            // Modal close events
            $(document).on('click', '.ticketpaygo-modal-close, .ticketpaygo-modal-overlay', function(e) {
                e.preventDefault();
                TicketPayGoFrontend.closeBookingModal();
            });

            // Prevent modal close when clicking inside modal content
            $(document).on('click', '.ticketpaygo-modal-container', function(e) {
                e.stopPropagation();
            });

            // Quantity controls - REMOVED: handled by vanilla JS at top of file
            // Quantity input changes - REMOVED: handled by vanilla JS at top of file

            // Step navigation
            $(document).on('click', '.btn-next-step', function(e) {
                e.preventDefault();
                const modal = $(this).closest('.ticketpaygo-modal');
                const currentStep = modal.find('.form-step.active').data('step');
                
                if (TicketPayGoFrontend.validateStep(modal, currentStep)) {
                    TicketPayGoFrontend.goToStep(modal, currentStep + 1);
                }
            });

            $(document).on('click', '.btn-prev-step', function(e) {
                e.preventDefault();
                const modal = $(this).closest('.ticketpaygo-modal');
                const currentStep = modal.find('.form-step.active').data('step');
                TicketPayGoFrontend.goToStep(modal, currentStep - 1);
            });

            // Form submission
            $(document).on('submit', '.modal-form', function(e) {
                e.preventDefault();
                const modal = $(this).closest('.ticketpaygo-modal');
                TicketPayGoFrontend.submitModalBooking(modal, $(this));
            });

            // ESC key to close modal
            $(document).on('keydown', function(e) {
                if (e.keyCode === 27) { // ESC key
                    TicketPayGoFrontend.closeBookingModal();
                }
            });
        },

        openBookingModal: function(eventId) {
            // Try both modal ID formats
            let modal = $('#ticketpaygo-modal-' + eventId);
            if (!modal.length) {
                modal = $('#modal-' + eventId);
            }
            
            if (modal.length) {
                // Check if it's the new booking modal format
                if (modal.find('.form-step').length) {
                    modal.show();
                    $('body').addClass('modal-open');
                    // Reset to first step
                    this.goToStep(modal, 1);
                    // Update pricing
                    this.updateModalPricing(modal);
                    
                    // Initialize map if it exists and Leaflet is loaded
                    setTimeout(function() {
                        if (typeof L !== 'undefined') {
                            const mapContainer = modal.find('.event-location-map');
                            if (mapContainer.length && !mapContainer.data('map-initialized')) {
                                const mapId = mapContainer.attr('id');
                                const lat = parseFloat(mapContainer.data('lat') || mapContainer.closest('.modal-event-meta').find('[data-lat]').data('lat'));
                                const lng = parseFloat(mapContainer.data('lng') || mapContainer.closest('.modal-event-meta').find('[data-lng]').data('lng'));
                                
                                if (lat && lng && mapId) {
                                    try {
                                        const map = L.map(mapId, {
                                            scrollWheelZoom: false,
                                            dragging: true,
                                            zoomControl: true
                                        }).setView([lat, lng], 15);
                                        
                                        L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                                            attribution: '&copy; OpenStreetMap',
                                            maxZoom: 19
                                        }).addTo(map);
                                        
                                        L.marker([lat, lng]).addTo(map);
                                        
                                        mapContainer.data('map-initialized', true);
                                        setTimeout(function() { map.invalidateSize(); }, 100);
                                    } catch(e) {
                                        console.log('Map initialization error:', e);
                                    }
                                }
                            }
                        }
                    }, 300);
                } else {
                    // Use the simple modal open function for event-default.php modals
                    if (typeof openEventModal === 'function') {
                        openEventModal(eventId);
                    } else {
                        modal.css('display', 'flex');
                        $('body').css('overflow', 'hidden');
                    }
                }
            }
        },

        closeBookingModal: function() {
            $('.ticketpaygo-modal').fadeOut(300);
            $('body').removeClass('modal-open').css('overflow', '');
        },

        goToStep: function(modal, stepNumber) {
            // Update form steps
            modal.find('.form-step').removeClass('active');
            modal.find('.form-step[data-step="' + stepNumber + '"]').addClass('active');

            // Update progress indicator
            modal.find('.progress-step').removeClass('active');
            modal.find('.progress-step[data-step="' + stepNumber + '"]').addClass('active');

            // Update progress bar
            const progressPercent = (stepNumber / 3) * 100;
            modal.find('.progress-fill').css('width', progressPercent + '%');
        },

        validateStep: function(modal, stepNumber) {
            let isValid = true;

            if (stepNumber === 1) {
                // Validate quantity
                const quantity = parseInt(modal.find('.qty-input').val()) || 0;
                if (quantity < 1) {
                    this.showMessage('error', ticketpaygo_frontend.strings.select_quantity);
                    isValid = false;
                }
            } else if (stepNumber === 2) {
                // Validate customer information
                const requiredFields = modal.find('.form-step[data-step="2"] [required]');
                requiredFields.each(function() {
                    if (!$(this).val().trim()) {
                        $(this).addClass('error');
                        isValid = false;
                    } else {
                        $(this).removeClass('error');
                    }
                });

                // Validate email
                const email = modal.find('input[type="email"]').val();
                if (email && !this.isValidEmail(email)) {
                    modal.find('input[type="email"]').addClass('error');
                    isValid = false;
                }

                if (!isValid) {
                    this.showMessage('error', ticketpaygo_frontend.strings.fill_required_fields);
                }
            }

            return isValid;
        },

        updateModalPricing: function(modal) {
            const quantity = parseInt(modal.find('.qty-input').val()) || 0;
            const pricePerTicket = parseFloat(modal.find('.qty-input').data('price')) || 0;
            const subtotal = quantity * pricePerTicket;
            const processingFee = subtotal * 0.029; // 2.9% processing fee
            const total = subtotal + processingFee;

            modal.find('.subtotal').text(this.formatCurrency(subtotal));
            modal.find('.processing-fee').text(this.formatCurrency(processingFee));
            modal.find('.total-amount').text(this.formatCurrency(total));
        },

        submitModalBooking: function(modal, form) {
            // Validate final step
            if (!this.validateStep(modal, 3)) {
                return;
            }

            // Check terms agreement
            if (!modal.find('input[name="agree_terms"]').is(':checked')) {
                this.showMessage('error', ticketpaygo_frontend.strings.agree_terms || 'Please agree to the terms and conditions');
                return;
            }

            // Show loading state
            const submitBtn = modal.find('.btn-complete-booking');
            submitBtn.find('.btn-text').hide();
            submitBtn.find('.btn-loading').show();
            submitBtn.prop('disabled', true);

            // Prepare form data
            const formData = new FormData(form[0]);

            // Submit via AJAX
            $.ajax({
                url: ticketpaygo_frontend.ajax_url,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    if (response.success) {
                        if (response.data.redirect_url) {
                            // Redirect to payment gateway
                            window.location.href = response.data.redirect_url;
                        } else {
                            // Show success message and close modal
                            TicketPayGoFrontend.showMessage('success', response.data.message);
                            TicketPayGoFrontend.closeBookingModal();
                        }
                    } else {
                        TicketPayGoFrontend.showMessage('error', response.data);
                    }
                },
                error: function() {
                    TicketPayGoFrontend.showMessage('error', ticketpaygo_frontend.strings.booking_failed);
                },
                complete: function() {
                    // Reset button state
                    submitBtn.find('.btn-text').show();
                    submitBtn.find('.btn-loading').hide();
                    submitBtn.prop('disabled', false);
                }
            });
        },

        handlePaymentReturn: function(urlParams) {
            const status = urlParams.get('payment_status');
            const orderId = urlParams.get('order_id');

            if (status === 'success') {
                this.showMessage('success', ticketpaygo_frontend.strings.payment_success);
                // Redirect to ticket page after 3 seconds
                setTimeout(() => {
                    window.location.href = ticketpaygo_frontend.ticket_url + '?order_id=' + orderId;
                }, 3000);
            } else if (status === 'cancelled') {
                this.showMessage('warning', ticketpaygo_frontend.strings.payment_cancelled);
            } else if (status === 'failed') {
                this.showMessage('error', ticketpaygo_frontend.strings.payment_failed);
            }
        },

        downloadTicket: function(ticketId) {
            const link = document.createElement('a');
            link.href = ticketpaygo_frontend.ajax_url + '?action=ticketpaygo_download_ticket&ticket_id=' + ticketId + '&nonce=' + ticketpaygo_frontend.nonce;
            link.download = 'ticket-' + ticketId + '.pdf';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        },

        shareEvent: function(eventUrl, eventTitle) {
            if (navigator.share) {
                // Use native sharing if available
                navigator.share({
                    title: eventTitle,
                    url: eventUrl
                }).catch(err => console.log('Error sharing:', err));
            } else {
                // Fallback to copying URL
                navigator.clipboard.writeText(eventUrl).then(() => {
                    this.showMessage('success', ticketpaygo_frontend.strings.url_copied);
                }).catch(() => {
                    // Fallback for older browsers
                    const textArea = document.createElement('textarea');
                    textArea.value = eventUrl;
                    document.body.appendChild(textArea);
                    textArea.select();
                    document.execCommand('copy');
                    document.body.removeChild(textArea);
                    this.showMessage('success', ticketpaygo_frontend.strings.url_copied);
                });
            }
        },

        openCheckoutModal: function(eventId, eventTitle, quantity, totalAmount) {
            console.log('Opening checkout modal:', {eventId, eventTitle, quantity, totalAmount});
            
            const modal = $('#ticketpaygo-checkout-modal');
            
            // Populate order summary
            $('#checkout-event-title').text(eventTitle);
            $('#checkout-quantity').text(quantity + ' x ticket(s)');
            $('#checkout-total').text('€ ' + totalAmount.toFixed(2));
            
            // Set hidden form fields
            $('#checkout-event-id').val(eventId);
            $('#checkout-quantity-input').val(quantity);
            $('#checkout-total-input').val(totalAmount.toFixed(2));
            
            // Show modal
            modal.fadeIn(300);
            $('body').css('overflow', 'hidden');
        },

        closeCheckoutModal: function() {
            $('#ticketpaygo-checkout-modal').fadeOut(300);
            $('body').css('overflow', '');
        },

        showMessage: function(type, message) {
            const messageDiv = $(`<div class="ticketpaygo-message ${type}">${message}</div>`);
            
            // Remove existing messages
            $('.ticketpaygo-message').remove();
            
            // Add new message
            $('body').prepend(messageDiv);
            
            // Auto-hide after 5 seconds
            setTimeout(() => {
                messageDiv.fadeOut(() => messageDiv.remove());
            }, 5000);
        },

        // Utility functions
        formatCurrency: function(amount) {
            const symbol = ticketpaygo_frontend.currency_symbol || '€';
            const position = ticketpaygo_frontend.currency_position || 'before';
            const formatted = parseFloat(amount).toFixed(2);
            
            return position === 'before' ? symbol + formatted : formatted + symbol;
        },

        isValidEmail: function(email) {
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return emailRegex.test(email);
        },

        formatDate: function(dateString) {
            const date = new Date(dateString);
            return date.toLocaleDateString() + ' ' + date.toLocaleTimeString();
        }
    };

    // Initialize all components when DOM is ready
    $(document).ready(function() {
        initializeEventCards();
        TicketPayGoFrontend.init();
        initializeCheckoutModal();
    });

    /**
     * Initialize Checkout Modal
     */
    function initializeCheckoutModal() {
        // Open checkout modal button
        $(document).on('click', '.open-checkout-modal', function(e) {
            e.preventDefault();
            console.log('TicketPayGo: Opening checkout modal');
            
            const btn = $(this);
            const eventId = btn.data('event-id');
            const eventTitle = btn.data('event-title');
            
            // Get quantity and total from the modal
            const modal = btn.closest('.ticketpaygo-modal');
            let quantity = 0;
            let totalAmount = 0;
            
            // Sum up all ticket quantities and calculate total from individual prices
            modal.find('.qty-input').each(function() {
                const qty = parseInt($(this).val()) || 0;
                const price = parseFloat($(this).data('price')) || 0;
                quantity += qty;
                totalAmount += qty * price;
            });
            
            console.log('TicketPayGo: Quantity:', quantity, 'Total:', totalAmount);
            
            if (quantity < 1) {
                alert('Please select at least 1 ticket');
                return;
            }
            
            // Close event modal
            if (typeof closeEventModal === 'function') {
                closeEventModal(eventId);
            }
            
            // Open checkout modal
            TicketPayGoFrontend.openCheckoutModal(eventId, eventTitle, quantity, totalAmount);
        });
        
        // Close checkout modal
        $(document).on('click', '.checkout-modal-close, .checkout-modal-overlay', function(e) {
            e.preventDefault();
            TicketPayGoFrontend.closeCheckoutModal();
        });
        
        // Auth toggle (guest/login)
        $(document).on('click', '.auth-toggle-btn', function(e) {
            e.preventDefault();
            const mode = $(this).data('mode');
            
            $('.auth-toggle-btn').removeClass('active');
            $(this).addClass('active');
            
            if (mode === 'guest') {
                $('#guest-form').show();
                $('#login-form').hide();
            } else {
                $('#guest-form').hide();
                $('#login-form').show();
            }
        });
        
        // Checkout form submission - handle both form submit and button click
        $(document).on('click', '.btn-checkout-submit', function(e) {
            e.preventDefault();
            e.stopPropagation();
            console.log('TicketPayGo: Checkout button clicked');
            
            const form = $('#checkout-form');
            const submitBtn = $(this);
            
            // Prevent double submission
            if (submitBtn.prop('disabled')) {
                console.log('TicketPayGo: Button already disabled, ignoring');
                return;
            }
            
            processCheckout(form, submitBtn);
        });
        
        $(document).on('submit', '#checkout-form', function(e) {
            e.preventDefault();
            console.log('TicketPayGo: Checkout form submitted');
            
            const form = $(this);
            const submitBtn = form.find('.btn-checkout-submit');
            
            // Prevent double submission
            if (submitBtn.prop('disabled')) {
                console.log('TicketPayGo: Button already disabled, ignoring');
                return;
            }
            
            processCheckout(form, submitBtn);
        });
        
        function processCheckout(form, submitBtn) {
            
            // Debug: Log all form fields
            const formData = form.serializeArray();
            console.log('TicketPayGo: Form fields:', formData);
            formData.forEach(field => {
                console.log(`  ${field.name}: "${field.value}"`);
            });
            
            // Show loading state
            submitBtn.prop('disabled', true);
            submitBtn.find('.btn-text').hide();
            submitBtn.find('.btn-loading').show();
            
            console.log('TicketPayGo: Form data serialized:', form.serialize());
            
            // Submit via AJAX
            $.ajax({
                url: ticketpaygo_frontend.ajax_url,
                type: 'POST',
                data: form.serialize(),
                success: function(response) {
                    console.log('TicketPayGo: Checkout response:', response);
                    
                    // Handle both wp_send_json (direct) and wp_send_json_success (wrapped in data)
                    var redirectUrl = response.redirect_url || (response.data && response.data.redirect_url);
                    var isSuccess = response.success;
                    
                    if (isSuccess && redirectUrl) {
                        console.log('TicketPayGo: Redirecting to:', redirectUrl);
                        window.location.href = redirectUrl;
                    } else {
                        console.error('TicketPayGo: Checkout failed:', response);
                        var errorMsg = response.message || (response.data && response.data.message) || 'Purchase failed. Please try again.';
                        alert(errorMsg);
                        
                        // Reset button
                        submitBtn.prop('disabled', false);
                        submitBtn.find('.btn-text').show();
                        submitBtn.find('.btn-loading').hide();
                    }
                },
                error: function(xhr, status, error) {
                    console.error('TicketPayGo: Checkout error:', status, error);
                    console.error('TicketPayGo: Response:', xhr.responseText);
                    alert('Checkout failed. Please try again.');
                    
                    // Reset button
                    submitBtn.prop('disabled', false);
                    submitBtn.find('.btn-text').show();
                    submitBtn.find('.btn-loading').hide();
                }
            });
        }
    }

    /**
     * Initialize Event Card functionality
     */
    function initializeEventCards() {
        console.log('TicketPayGo: Initializing event cards');
        
        // Simple, reliable click handler
        $(document).on('click', '.event-card-compact', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            console.log('TicketPayGo: Card clicked');
            
            // Find the parent container
            var $container = $(this).closest('.ticketpaygo-event-card, .ticketpaygo-event-card-list');
            var eventId;
            
            if ($container.attr('id')) {
                if ($container.attr('id').indexOf('ticketpaygo-event-') !== -1) {
                    eventId = $container.attr('id').replace('ticketpaygo-event-', '');
                } else if ($container.attr('id').indexOf('event-list-') !== -1) {
                    eventId = $container.attr('id').replace('event-list-', '');
                }
            }
            
            if (eventId) {
                console.log('TicketPayGo: Event ID:', eventId);
                
                // Find the details section
                var $details = $('#details-' + eventId + ', #details-list-' + eventId);
                
                if ($details.length) {
                    console.log('TicketPayGo: Found details section');
                    
                    // Toggle visibility
                    if ($details.is(':visible')) {
                        $details.slideUp(300);
                        $container.removeClass('event-card-expanded');
                        console.log('TicketPayGo: Collapsed');
                    } else {
                        $details.slideDown(300);
                        $container.addClass('event-card-expanded');
                        console.log('TicketPayGo: Expanded');
                        
                        // Scroll to show expanded content
                        setTimeout(function() {
                            $('html, body').animate({
                                scrollTop: $details.offset().top - 100
                            }, 500);
                        }, 350);
                    }
                } else {
                    console.log('TicketPayGo: Details section not found for ID:', eventId);
                }
            } else {
                console.log('TicketPayGo: Could not determine event ID');
            }
            
            return false;
        });
        
        // Add direct click handlers to all event cards (toggleEventDetails is defined globally at top of file)
        document.addEventListener('DOMContentLoaded', function() {
            console.log('Adding click handlers to event cards');
            
            // For horizontal cards
            const horizontalCards = document.querySelectorAll('.ticketpaygo-event-card .event-card-compact');
            horizontalCards.forEach(function(card) {
                const eventId = card.closest('.ticketpaygo-event-card').id.replace('ticketpaygo-event-', '');
                card.addEventListener('click', function(e) {
                    e.preventDefault();
                    console.log('Horizontal card clicked:', eventId);
                    toggleEventDetails(eventId);
                });
            });
            
            // For vertical cards
            const verticalCards = document.querySelectorAll('.ticketpaygo-event-card-list .event-card-compact');
            verticalCards.forEach(function(card) {
                const eventId = card.closest('.ticketpaygo-event-card-list').id.replace('event-list-', '');
                card.addEventListener('click', function(e) {
                    e.preventDefault();
                    console.log('Vertical card clicked:', eventId);
                    toggleEventDetails(eventId);
                });
            });
        });

        // Close expanded details when clicking outside
        document.addEventListener('click', function(e) {
            if (!e.target.closest('.ticketpaygo-event-card') && !e.target.closest('.ticketpaygo-event-card-list')) {
                const expandedCards = document.querySelectorAll('.event-card-expanded');
                expandedCards.forEach(card => {
                    let eventId, details;
                    
                    if (card.id.includes('ticketpaygo-event-')) {
                        eventId = card.id.replace('ticketpaygo-event-', '');
                        details = document.getElementById('details-' + eventId);
                    } else if (card.id.includes('event-list-')) {
                        eventId = card.id.replace('event-list-', '');
                        details = document.getElementById('details-list-' + eventId);
                    }
                    
                    if (details) {
                        details.style.display = 'none';
                        card.classList.remove('event-card-expanded');
                    }
                });
            }
        });
    }

    // Make TicketPayGoFrontend globally available
    window.TicketPayGoFrontend = TicketPayGoFrontend;

})(jQuery);
